/** 
* @file ColorTableReferenceColor.cpp
* Implementation of class ColorTableReferenceColor.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#include "ColorTableReferenceColor.h"
#include "Platform/SystemCall.h"
#include "Tools/Math/Common.h"


ColorTableReferenceColor::ColorTableReferenceColor()
{
  clear();
}

ColorTableReferenceColor::~ColorTableReferenceColor()
{
}

void ColorTableReferenceColor::clear()
{
  greenMin[0] = 255; greenMin[1] = 255; greenMin[2] = 255;
  greenMax[0] = 0; greenMax[1] = 0; greenMax[2] = 0;
  setGreen();
  greenIsValid = false;
  setGreenCounter = 0;
}


int ColorTableReferenceColor::getGreenMin(int channel) const
{
  return greenMin[channel];
}

int ColorTableReferenceColor::getGreenMax(int channel) const
{
  return greenMax[channel];
}

void ColorTableReferenceColor::setGreenMin(int channel, int value)
{
  greenMin[channel] = value;
  setGreen();
}

void ColorTableReferenceColor::setGreenMax(int channel, int value)
{
  greenMax[channel] = value;
  setGreen();
}

void ColorTableReferenceColor::expandGreen(int greenMin[3], int greenMax[3])
{
  if(greenMin[0] < this->greenMin[0]) this->greenMin[0] = greenMin[0];
  if(greenMin[1] < this->greenMin[1]) this->greenMin[1] = greenMin[1];
  if(greenMin[2] < this->greenMin[2]) this->greenMin[2] = greenMin[2];

  if(greenMax[0] > this->greenMax[0]) this->greenMax[0] = greenMax[0];
  if(greenMax[1] > this->greenMax[1]) this->greenMax[1] = greenMax[1];
  if(greenMax[2] > this->greenMax[2]) this->greenMax[2] = greenMax[2];
  setGreen();
}

int ColorTableReferenceColor::getAverageGreen(int channel) const
{
  return averageGreen[channel];
}

void ColorTableReferenceColor::setGreen()
{
  setGreenCounter++;
  if(setGreenCounter > 5) greenIsValid = true;
  int greenRange[3]; 
  greenRange[0] = greenMax[0] - greenMin[0];
  greenRange[1] = greenMax[1] - greenMin[1];
  greenRange[2] = greenMax[2] - greenMin[2];

  averageGreen[0] = (greenMax[0] + greenMin[0]) / 2;
  averageGreen[1] = (greenMax[1] + greenMin[1]) / 2;
  averageGreen[2] = (greenMax[2] + greenMin[2]) / 2;

//  skyblueAutoMin[0] = (int)(greenMin[0] * 0.8);
//  skyblueAutoMax[0] = min(255, greenMax[0] * 2);
  skyblueAutoMin[0] = 0;
  skyblueAutoMax[0] = 255;
  skyblueAutoMax[1] = averageGreen[1];
  skyblueAutoMin[2] = greenMax[2];

//  yellowAutoMin[0] = greenMin[0] / 3;
  yellowAutoMax[0] = min(255, greenMax[0] * 2);
  yellowAutoMin[0] = 0;
//  yellowAutoMax[0] = 255;
  yellowAutoMin[1] = averageGreen[1];
  yellowAutoMax[2] = greenMin[2] - 10;
//  yellowAutoMin[1] = greenMax[1];
//  yellowAutoMax[2] = greenMin[2];

//  orangeAutoMin[0] = averageGreen[0] - 90;
//  orangeAutoMax[0] = averageGreen[0] + 120;
  orangeAutoMin[0] = 0;
  orangeAutoMax[0] = 255;
  orangeAutoMin[1] = averageGreen[1] + 40;
  orangeAutoMax[2] = averageGreen[2];
/*
newSegment.averageIntensity[0] > colorTableReferenceColor.getAverageGreen(0) - 90 &&
newSegment.averageIntensity[0] < colorTableReferenceColor.getAverageGreen(0) + 120 &&
newSegment.averageIntensity[1] > colorTableReferenceColor.getAverageGreen(1) + 40 &&
newSegment.averageIntensity[2] < colorTableReferenceColor.getAverageGreen(2) - 0 &&
*/

  pinkAutoMin[0] = averageGreen[0];
  pinkAutoMin[1] = averageGreen[1] + 20;
  pinkAutoMin[2] = averageGreen[2];

  whiteAutoMin[0] = averageGreen[0];
  whiteAutoMin[1] = averageGreen[1] - 2 * greenRange[1];
  whiteAutoMax[1] = averageGreen[1] + 2 * greenRange[1];
  whiteAutoMin[2] = averageGreen[2] - 2 * greenRange[2];
  whiteAutoMax[2] = averageGreen[2] + 2 * greenRange[2];

  whiteFixedMin[0] = 180;
  whiteFixedMin[1] = 127 - 20;
  whiteFixedMax[1] = 127 + 50;
  whiteFixedMin[2] = 127 - 70;
  whiteFixedMax[2] = 127 + 30;
}

colorClass ColorTableReferenceColor::getColorClass
(
 const unsigned char y, 
 const unsigned char u, 
 const unsigned char v
 ) const
{
  colorClass toReturn = noColor;
  if(
    // green of the field
    greenIsValid &&
    y >= greenMin[0] && y <= greenMax[0] && 
    u >= greenMin[1] && u <= greenMax[1] &&
    v >= greenMin[2] && v <= greenMax[2]
    )
  {
    return green;
  }
  if(
    greenIsValid &&
//    y > skyblueAutoMin[0] && 
//    y < skyblueAutoMax[0] &&   
    u < skyblueAutoMax[1] && 
    v > skyblueAutoMin[2])
  {
    return skyblue;
  }
  if(
    greenIsValid &&
    y > yellowAutoMin[0] && 
    y < yellowAutoMax[0] && 
    u > yellowAutoMin[1] && 
    v < yellowAutoMax[2]
    )
  {
//    return yellow;
    toReturn = yellow;
  }
  if(
    greenIsValid &&
//    y > orangeAutoMin[0] && 
//    y < orangeAutoMax[0] && 
    u > orangeAutoMin[1] && 
    v < orangeAutoMax[2]
    )
  {
    if (toReturn == yellow) return yellowOrange;
    return orange;
  }
  if(toReturn == yellow)
  {
    return toReturn;
  }
  if(
//    y > whiteMin[0] &&
//    u > whiteMin[1] && u < whiteMax[1] &&
//    v > whiteMin[2] && v < whiteMax[2] 
//    ||
    y > whiteFixedMin[0] &&
    u > whiteFixedMin[1] && u < whiteFixedMax[1] &&
    v > whiteFixedMin[2] && v < whiteFixedMax[2] 
    )
  {
    return white;
  }
  if(
    greenIsValid &&
    y > pinkAutoMin[0] &&
    u > pinkAutoMin[1] &&
    v > pinkAutoMin[2]
    )
  {
    return pink;
  }
  return noColor;
}

void ColorTableReferenceColor::generateColorClassImage
(
 const Image& image, 
 ColorClassImage& colorClassImage) const
{
  colorClass color;
  colorClassImage.width = image.cameraInfo.resolutionWidth;
  colorClassImage.height = image.cameraInfo.resolutionHeight;

  for (register int y = 0; y < image.cameraInfo.resolutionHeight; y++) 
  {
    for (register int x = 0; x < image.cameraInfo.resolutionWidth; x++) 
    {
      color = getColorClass(image.image[y][0][x],image.image[y][1][x],image.image[y][2][x]);
      colorClassImage.image[y][x] = color;
    }
  }
}

In& operator>>(In& stream, ColorTableReferenceColor& colorTableReferenceColor)
{
  int greenMin;
  int greenMax;
  for(int channel = 0; channel < 3; channel++)
  {
    stream >> greenMin;
    stream >> greenMax;
    colorTableReferenceColor.setGreenMin(channel, greenMin);
    colorTableReferenceColor.setGreenMax(channel, greenMax);
  }
  return stream;
}

Out& operator<<(Out& stream, const ColorTableReferenceColor& colorTableReferenceColor)
{
  for(int channel = 0; channel < 3; channel++)
  {
    stream << colorTableReferenceColor.getGreenMin(channel);
    stream << colorTableReferenceColor.getGreenMax(channel);
  }
  return stream;
}

/*
* Change log :
* 
* $Log: ColorTableReferenceColor.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:19:48  cvsadm
* created new repository GT2004_WM
*
* Revision 1.10  2003/12/30 23:55:01  roefer
* Image size is now 208 x 160. Smaller images are placed in the upper left corner
*
* Revision 1.9  2003/12/15 11:46:14  juengel
* Introduced CameraInfo
*
* Revision 1.8  2003/11/13 09:42:08  juengel
* no message
*
* Revision 1.7  2003/11/11 17:01:32  juengel
* improved pink
*
* Revision 1.6  2003/11/10 11:35:59  juengel
* Added pink.
*
* Revision 1.5  2003/11/05 16:46:06  juengel
* changed skyblue
*
* Revision 1.4  2003/11/03 20:19:22  juengel
* Added color class yellowOrange
*
* Revision 1.3  2003/10/31 11:48:26  juengel
* Replaced fixed image height and width.
*
* Revision 1.2  2003/10/29 13:13:14  juengel
* added method "expandGreen"
*
* Revision 1.1  2003/10/23 07:15:17  juengel
* Renamed ColorTable   to ColorTableReferenceColor,
* added ColorTableCuboids.
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/08/29 13:12:50  juengel
* changed time after which orange is forgotten
*
* Revision 1.3  2003/08/18 12:06:01  juengel
* Added streaming operators.
*
* Revision 1.2  2003/07/29 12:44:26  juengel
* Optimization
*
* Revision 1.1  2003/07/21 13:40:24  juengel
* Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.2  2003/07/19 19:31:04  juengel
* Improved AutoCalibration
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/05 12:27:27  juengel
* Added method setGreen.
*
* Revision 1.5  2003/04/23 16:17:55  juengel
* ColorTableReferenceColor::getColorClassForFlagsAndGoals returns a color class now.
*
* Revision 1.4  2003/04/03 18:50:08  juengel
* Improved AutoCalibration.
*
* Revision 1.3  2003/04/02 13:21:11  juengel
* Improved flag colors.
*
* Revision 1.2  2003/04/01 19:28:45  juengel
* Improved getColorClassFlagsAndGoals.
*
* Revision 1.1  2003/04/01 13:59:34  juengel
* Added ColorTableReferenceColor
*
*
*/
