/** 
* @file ColorSpaceUsageCounter.h
* Declaration of class ColorSpaceUsageCounter.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#ifndef _ColorSpaceUsageCounter_h_
#define _ColorSpaceUsageCounter_h_

#include "Representations/Perception/Image.h"

/**
* @class ColorSpaceUsageCounter
*
* Contains a ColorSpaceUsageCounter which can count the frequency for
* every 16 * 16 * 16 cube in the 255 * 255 * 255 YUV color space.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

class ColorSpaceUsageCounter
{
public:
  /** Constructor */
  ColorSpaceUsageCounter();

  /** Destructor */
  ~ColorSpaceUsageCounter();

  /** 
  * Returns the count for the cube associated to the y,u,v values.
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  * @return the color class
  */
  int getCount (const unsigned char y, 
    const unsigned char u, 
    const unsigned char v) const
  {
    return count[y/16][u/16][v/16];
  }

  /**
  * Generates an image that visualizes the current ColorSpaceUsageCounter
  *
  * @param image A reference to the original image.
  * @param resultImage A reference to the resulting image.
  */
  virtual void generateImage(const Image& image, Image& resultImage) const;

  /** 
  * Each element in the array contains the count for a 16x16x16 cube in the color space.
  */
  unsigned short int count[16][16][16];

  /** Sets the count of every 16x16x16 cube to 0 */
  void reset();
  
  /** 
  */
  void addColor(unsigned char y, unsigned char u, unsigned char v);

  void addColor(unsigned char y, unsigned char u, unsigned char v, unsigned char range);

  bool isColorFrequent(unsigned char y, unsigned char u, unsigned char v, int threshold);

  int maxCount;
};

#endif   //  _ColorSpaceUsageCounter_h_

/*
 * Change log :
 * 
 * $Log: ColorSpaceUsageCounter.h,v $
 * Revision 1.1.1.1  2004/05/22 17:19:47  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2003/10/30 18:30:19  juengel
 * Reduced number of cubes.
 *
 * Revision 1.2  2003/10/29 13:11:13  juengel
 * added methods "isColorFrequent" and "addColor(...,range)
 *
 * Revision 1.1  2003/10/23 07:15:17  juengel
 * Renamed ColorTableAuto to ColorTableReferenceColor,
 * added ColorTableCuboids.
 *
 *
 */
