/**
* @file Modules/ImageProcessor/ImageProcessorTools/ColorCorrector.cpp
* 
* This file contains a class that represents a table used for color correction.
* It is a mixture among the BB2004, DDD2004 and MSH2004 correction.
* Everything is static so it is loaded only once in the simulator.
*
* @author <A href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</A>
* @author Max Risler
* @author <A href="mailto:roefer@tzi.de">Thomas Rfer</A>
*/

#include "ColorCorrector.h"
#include "Platform/GTAssert.h"
#include "Tools/Location.h"
#include "Tools/Streams/InStreams.h"

bool ColorCorrector::loaded = false;
unsigned char ColorCorrector::radiusTable[cameraResolutionHeight_ERS7][cameraResolutionWidth_ERS7];
unsigned char ColorCorrector::correctionTable[ColorCorrector::maxRadius][256][3];

#ifdef MSH
int ColorCorrector::radialOrder = 0;
int ColorCorrector::colorOrder = 0;
double ColorCorrector::radialP[3*maxRadialOrder];
double ColorCorrector::colorP[3*maxColorOrder];
const double centralPointX = 102.0;
const double centralPointY = 75.0;

#endif

#ifdef _WIN32  
unsigned char ColorCorrector::distortionTable[ColorCorrector::maxRadius][256][3];
#endif

void ColorCorrector::load()
{
  if(loaded) 
    return;
  loaded = true;

#ifndef MSH
  //setup radius table
  for (int x = 0; x < cameraResolutionWidth_ERS7; x++)
    for (int y = 0; y < cameraResolutionHeight_ERS7; y++)
      radiusTable[y][x] = calcRadius(x,y);

  int maxRadius = radiusTable[0][0];
  InBinaryFile file(getLocation().getModelFilename("white.img"));
  if(file.exists())
  {
    int i, c, j, x, y;
    long pixelSum[cameraResolutionWidth_ERS7][3];
    int pixelNum[cameraResolutionWidth_ERS7];

    for(i = 0; i <= maxRadius; i++)
    {
      for(j = 0; j < 3; j++)
        pixelSum[i][j] = 0;
      pixelNum[i] = 0;
    }

    do 
    {
      Image image;
      file >> image.cameraInfo.resolutionWidth 
           >> image.cameraInfo.resolutionHeight 
           >> image.frameNumber;
 	    for(y = 0; y < image.cameraInfo.resolutionHeight; ++y)
        for(c = 0; c < 3; ++c)
          file.read(&image.image[y][c][0], image.cameraInfo.resolutionWidth);

      for(x = 0; x < image.cameraInfo.resolutionWidth; ++x)
        for(y = 0; y < image.cameraInfo.resolutionHeight; ++y)
        {
          int r = radiusTable[y][x];
          pixelSum[r][0] += image.image[y][0][x];
          pixelSum[r][1] += image.image[y][1][x];
          pixelSum[r][2] += image.image[y][2][x];
          ++pixelNum[r];
        }
    }
    while(!file.eof());

    if(pixelNum[0] == 0)
      disable();
    else
      for(i = 0; i <= maxRadius; i++)
        for(j = 0; j < 3; j++)
        {
          pixelSum[i][j] /= pixelNum[i];
          for (c = 0; c < 256; c++)
          {
            long x = c * pixelSum[0][j] / pixelSum[i][j];
            if (x > 255) x = 255;
            correctionTable[i][c][j] = (unsigned char) x;
#ifdef _WIN32  
            x = c * pixelSum[i][j] / pixelSum[0][j];
            if (x > 255) x = 255;
            distortionTable[i][c][j] = (unsigned char) x;
#endif
          }
        }
  }
  else
    disable();
}
#else
  bool nothingLoaded = true;
  const char* channels[3] = {"coeff.cy", "coeff.cu", "coeff.cv"};
  double dummy;
  bool channelLoaded[3] = {false, false, false};
  for(int channel = 0; channel < 3; ++channel)
	{
    
    InBinaryFile stream(getLocation().getModelFilename(channels[channel]));
    if(stream.exists())
    {
      channelLoaded[channel] = true;
      stream >> radialOrder;
      stream >> colorOrder;
      int i;
      if(radialOrder < maxRadialOrder)
      { //source array bigger than file
        for(i = 0; i < radialOrder; ++i)
          stream >> radialP[i + channel * maxRadialOrder];
        for(; i < maxRadialOrder; i++)
          radialP[i] = 0.0;
      }
      else
      {
        for(i = 0; i < maxRadialOrder; ++i)
          stream >> radialP[i + channel * maxRadialOrder];
        for(; i < radialOrder; i++) //discard higher order coeff that doesnt fit in the array size
          stream >> dummy;
      }
      if (colorOrder < maxColorOrder)
      { //source array bigger than file
        for (i = 0; i < colorOrder; ++i)
          stream >> colorP[i + channel * maxColorOrder];
        for (; i < maxColorOrder; ++i)
          colorP[i] = 0.0;
      }
      else
      {
        for (i = 0; i < maxColorOrder; ++i)
          stream >> colorP[i + channel * maxColorOrder];
        for (; i < colorOrder; ++i) //discard higher order coeff that doesnt fit in the array size
          stream >> dummy;
      }
      nothingLoaded = false;
    }
    else
    {
      int i;
      for(i = 0; i < radialOrder; ++i)
        radialP[i + channel * maxRadialOrder] = 0.0;
      for(i = 0; i < colorOrder; ++i)
        colorP[i + channel * maxColorOrder] = 0.0;
    }
  }
  double dx, dy;
  for (int y=0; y<cameraResolutionHeight_ERS7; y++)
    for (int x=0; x<cameraResolutionWidth_ERS7; x++)
    {
      dx = x - centralPointX;
      dy = y - centralPointY;
      double radius = sqrt(dx*dx + dy*dy);
      radiusTable[y][x] = (unsigned char)radius;
    }
  for (int radius=0; radius < maxRadius; ++radius)
  {
    int color;
    for(color = 0; color < 256; ++color)
      for(int channel = 0; channel < 3; ++channel)
        correctionTable[radius][color][channel] = colorDistortionCorrection(radius, color, channel);
#ifdef _WIN32
    for(color = 0; color < 256; ++color)
      for(int channel = 0; channel < 3; ++channel)
      {
        int best = 0;
        for(int i = 1; i < 256; ++i)
          if((radius == 0 || abs(i - distortionTable[radius - 1][color][channel]) < 3) &&
             abs(correctionTable[radius][i][channel] - color) < abs(correctionTable[radius][best][channel] - color))
            best = i;
        distortionTable[radius][color][channel] = (unsigned char) best;
      }
#endif
  }
}

unsigned char ColorCorrector::colorDistortionCorrection(const unsigned char radius_i, 
                                                        const unsigned char color, const unsigned char channel)
{
  int radOffsetBase = channel * maxRadialOrder;
  int colOffsetBase = channel * maxColorOrder;
  double radius = radius_i;
  double radial_base;
  double color_base;
  double rad_correction = 0;
  double col_correction = 0;
  int i;
  for(i = 0; i < radialOrder; ++i)
  {
    radial_base = 1;
    for (int k = 0; k < i; ++k)
      radial_base *= radius;
    rad_correction += radial_base*radialP[radOffsetBase + i];
  }
  for(i = 0; i < colorOrder; ++i)
  {
    color_base = 1;
    for(int k = 0; k < i; ++k)
      color_base *= color;
    col_correction += color_base * colorP[colOffsetBase + i];
  }
  double result = color + rad_correction * col_correction;
  if(result < 0)
    result = 0;
  else if(result > 255)
    result = 255;
  return (unsigned char) result;
}
#endif

void ColorCorrector::disable()
{
  for(int x = 0; x < cameraResolutionWidth_ERS7; ++x)
    for(int y = 0; y < cameraResolutionHeight_ERS7; ++y)
      radiusTable[y][x] = 0;

  for(int c = 0; c < 256; ++c)
    for(int j = 0; j < 3; ++j)
    {
      correctionTable[0][c][j] = c;
#ifdef _WIN32
      distortionTable[0][c][j] = c;
#endif
    }
}

void ColorCorrector::correct(Image& image)
{
  for(int x = 0; x < image.cameraInfo.resolutionWidth; ++x)
    for(int y = 0; y < image.cameraInfo.resolutionHeight; ++y)
      correct(x, y, image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);
}

#ifdef _WIN32
void ColorCorrector::distort(Image& image)
{
  for(int x = 0; x < image.cameraInfo.resolutionWidth; ++x)
    for(int y = 0; y < image.cameraInfo.resolutionHeight; ++y)
      distort(x, y, image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);
}
#endif

/*
 * Change log :
 * 
 * $Log: ColorCorrector.cpp,v $
 * Revision 1.6  2004/06/13 11:30:13  nistico
 * *** empty log message ***
 *
 * Revision 1.5  2004/06/07 16:08:18  nistico
 * Changed central point of color correction (experimental!)
 * see if it works better/worse for your robots
 *
 * Revision 1.4  2004/05/30 17:25:25  roefer
 * Color correction parameters only in ERS-7 directory so it is not used on an ERS-210
 *
 * Revision 1.3  2004/05/30 14:35:06  roefer
 * Adaptation to new location handling
 *
 * Revision 1.2  2004/05/27 10:08:08  thomas
 * added model-specific locations
 *
 * Revision 1.1.1.1  2004/05/22 17:19:47  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.8  2004/05/22 09:07:12  roefer
 * Appearance of disturbed images improved
 *
 * Revision 1.7  2004/05/15 14:31:21  nistico
 * ColorCorrector now by default loads coefficients from Config/ folder.
 * If other coefficients are present in the folder pointed by location.cfg,
 * these override the default ones.
 *
 * Revision 1.6  2004/05/04 13:33:47  tim
 * corrected error in load()
 *
 * Revision 1.5  2004/04/27 19:33:26  roefer
 * Calculation of distortion is slower but produces no artefacts anymore
 *
 * Revision 1.4  2004/04/24 08:14:03  roefer
 * Author name was missing
 *
 * Revision 1.3  2004/04/23 14:13:26  dueffert
 * VC6 compilation bug fixed
 *
 * Revision 1.2  2004/04/22 14:28:46  roefer
 * ColorCorrector now supports MSH and DDD color correction
 *
 * Revision 1.1  2004/04/17 21:50:51  roefer
 * New color corrector (BB/DDD mix) used in BB2004ImageProcessor and Simulator
 * Displaying corrected images as debug image
 *
 */
