/**
* @file Modules/ImageProcessor/ImageProcessorTools/BresenhamLineScan.h
* 
* Utility class which performs the Bresenham algorithm 
* for line scanning
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#ifndef __BresenhamLineScan_h_
#define __BresenhamLineScan_h_

#include "Tools/Math/Geometry.h"
#include "Representations/Perception/CameraInfo.h"

#define DEG2RAD(x) x/180.0*3.1415926535897

class BresenhamLineScan
{
public:
  /** Constructor: Computes parameters for a line
  * @param start The start point of the line
  * @param end The end point of the line
  */
  BresenhamLineScan(const Vector2<int>& start, const Vector2<int>& end);

  /** Constructor: Computes parameters for a line
  * @param direction The direction vector of the line
  * @param cameraInfo Contains image related parameters
  */
  BresenhamLineScan(const Vector2<double>& direction, const CameraInfo& cameraInfo);

  /** Constructor: Computes parameters for a line
  * @param direction The direction (angle) of the line, expressed in radians
  * @param cameraInfo Contains image related parameters
  */
  BresenhamLineScan(const double& direction, const CameraInfo& cameraInfo);

  /** initializes the error counter */
  inline void init()
  {
    error = baseError;
  }

  /** 
  * Increments the coordinates to the next point on the line.
  * @param pos The position of the current pixel on the line, which is incremented by the Bresenham algorithm
  */
  inline void getNext(Vector2<int>& pos)
  {
    pos += standardOffset;
    error += delta;
    if(error > 0)
    {
      pos += correctionOffset;
      error += resetError;
    }
  }

  /** The numberOfPixels, can be used as a termination condition for the scan, 
  * but only if the first constructor has been used (the other 2 constructors
  * are in fact meant for infinite/unbounded scans, so it doesn't make sense)
  */
  int numberOfPixels;

private:
  
  /** Increase x-values, if true*/
  bool alongX;
  /** The error per step*/
  int delta;
  /** The initial error value*/
  int baseError;
  /** Resets the error to a value less than zero*/
  int resetError;
  /** The standard offset per step*/
  Vector2<int> standardOffset;
  /** The additional offset, if the error is above zero*/
  Vector2<int> correctionOffset;
  /** The current error counter*/
  int error;
  /** Computes the Bresenham parameters*/
  void setup(const Vector2<int>& start, const Vector2<int>& end);
};


#endif //__BresenhamLineScan_h_

/*
* $Log: BresenhamLineScan.h,v $
* Revision 1.2  2004/06/12 18:32:49  nistico
* Mean and nasty warning removed :-)
*
* Revision 1.1  2004/06/12 17:37:00  nistico
* Eventually, would be nice to have only one Bresenham on the whole
* GT2004ImageProcessor
*
*
*/
