/**
* @file Modules/ImageProcessor/ImageProcessorTools/BresenhamLineScan.h
* 
* Utility class which performs the Bresenham algorithm 
* for line scanning
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#include "BresenhamLineScan.h"

void BresenhamLineScan::setup(const Vector2<int>& start, const Vector2<int>& end)
{
  int dx(end.x - start.x);
  int dy(end.y - start.y);
  int incX = ((dx>0) ? 1:-1);
  int incY = ((dy>0) ? 1:-1);
  int absDx(abs(dx));
  int absDy(abs(dy));
  alongX = (absDy < absDx);
  if(alongX)
  {
    baseError = -absDx;
    delta = 2*absDy;
    standardOffset.x = incX;
    standardOffset.y = 0;
    correctionOffset.x = 0;
    correctionOffset.y = incY;
    numberOfPixels = absDx;
  }
  else
  {
    baseError = -absDy;
    delta = 2*absDx;
    standardOffset.x = 0;
    standardOffset.y = incY;
    correctionOffset.x = incX;
    correctionOffset.y = 0;
    numberOfPixels = absDy;
  }
  resetError = 2*baseError;
}

BresenhamLineScan::BresenhamLineScan(const Vector2<int>& start, const Vector2<int>& end)
{
  setup(start, end);
}

BresenhamLineScan::BresenhamLineScan(const double& direction, const CameraInfo& cameraInfo)
{
  const Vector2<int> frameUpperLeft(0,0);
  const Vector2<int> frameLowerRight(cameraInfo.resolutionWidth-1, cameraInfo.resolutionHeight-1);
  Geometry::Line scanLine(frameLowerRight/2, Vector2<double>(cos(direction), sin(direction)));
  Vector2<int> lineStart, lineEnd;
  Geometry::getIntersectionPointsOfLineAndRectangle(
                                      frameUpperLeft,
                                      frameLowerRight,
                                      scanLine, lineStart, lineEnd);
  setup(lineStart, lineEnd);
}


BresenhamLineScan::BresenhamLineScan(const Vector2<double>& direction, const CameraInfo& cameraInfo)
{
  const Vector2<int> frameUpperLeft(0,0);
  const Vector2<int> frameLowerRight(cameraInfo.resolutionWidth-1, cameraInfo.resolutionHeight-1);
  Geometry::Line scanLine(frameLowerRight/2, direction);
  Vector2<int> lineStart, lineEnd;
  Geometry::getIntersectionPointsOfLineAndRectangle(
                                      frameUpperLeft,
                                      frameLowerRight,
                                      scanLine, lineStart, lineEnd);
  setup(lineStart, lineEnd);
}

/*
* $Log: BresenhamLineScan.cpp,v $
* Revision 1.2  2004/06/12 18:32:49  nistico
* Mean and nasty warning removed :-)
*
* Revision 1.1  2004/06/12 17:37:00  nistico
* Eventually, would be nice to have only one Bresenham on the whole
* GT2004ImageProcessor
*
*
*/
