/**
* @file ImageProcessorSelector.h
*
* Definition of class ImageProcessorSelector
*
* @author Max Risler
* @author Martin Ltzsch
* @author Matthias Jngel
*/

#ifndef __ImageProcessorSelector_h_
#define __ImageProcessorSelector_h_

#include "Tools/Module/ModuleSelector.h"

#include "GT2004ImageProcessor/GT2004ImageProcessor.h"
#include "RegionGrowingImageProcessor/RegionGrowingImageProcessor.h"
#include "SLAMImageProcessor/SLAMImageProcessor.h"
#include "CheckerboardDetector.h"

/**
* ImageProcessor selection module.
*
* A selector for image processor modules.
*/
class ImageProcessorSelector : public ModuleSelector, public ImageProcessorInterfaces
{
public:
/**
* Constructor.
* @param handler The module handler of the process.
* @param interfaces The parameters of the ImageProcessor module.
  */
  ImageProcessorSelector(
    ModuleHandler &handler,
    const ImageProcessorInterfaces& interfaces)
    : ModuleSelector(SolutionRequest::imageProcessor),
    ImageProcessorInterfaces(interfaces)
  {
    handler.setModuleSelector(SolutionRequest::imageProcessor, this);
  }

  /** 
  * Is called on start and when the selected solution changes
  * to create a specific solution.
  * @param id The id of the solution to create
  * @return The created solution or 0
  */
  virtual Module* createSolution(SolutionRequest::ModuleSolutionID id)
  {
    switch(id)
    {
    case SolutionRequest::gt2004ImageProcessor:
      return new GT2004ImageProcessor(*this);
    
    case SolutionRequest::slamImageProcessor:
      return new SLAMImageProcessor(*this);

    case SolutionRequest::regionGrowingImageProcessor:
      return new RegionGrowingImageProcessor(*this);

    case SolutionRequest::checkerboardDetector:
      return new CheckerboardDetector(*this);
    
    default:
      return 0;
    }
  }
};

#endif // ImageProcessorSelector_h_

/*
* Change log :
* 
* $Log: ImageProcessorSelector.h,v $
* Revision 1.5  2004/07/02 10:11:47  nistico
* Cloned main image processor and created
* SpecialLandmarks specialist for SLAM challenge
*
* Revision 1.4  2004/06/17 11:24:04  roefer
* Added RGIP and GT2004SL
*
* Revision 1.3  2004/05/29 18:21:27  dueffert
* walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
*
* Revision 1.2  2004/05/22 18:47:42  juengel
* Removed some solutions.
*
* Revision 1.1.1.1  2004/05/22 17:19:27  cvsadm
* created new repository GT2004_WM
*
* Revision 1.16  2004/05/04 13:56:23  tim
* added GT2004ImageProcessor
*
* Revision 1.15  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.14  2004/04/08 15:33:05  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.13  2004/03/28 18:35:18  nistico
* Temporarily created MSH2004ImageProcessor2, it has some important advancements compared to
* the MSH2004ImageProcessor1 but it is untested on real robots yet, hence the creation of the
* new module.
* It will be removed soon, don't use it!
*
* Revision 1.12  2004/03/20 14:43:09  pg_thki
* - removed unused image processors.
*
* Revision 1.11  2004/03/19 11:31:55  nistico
* ColorTable32KImageProcessor removed
*
* Revision 1.10  2004/03/16 14:16:06  risler
* cloned GT2003ImageProcessor to DDD2004ImageProcessor
*
* Revision 1.9  2004/03/08 01:38:57  roefer
* Interfaces should be const
*
* Revision 1.8  2004/02/16 18:52:57  tim
* Added BB2004ImageProcessor
*
* Revision 1.7  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.6  2004/01/20 12:40:08  nistico
* - Added support for ColorTable32K (65K elements in packed format)
* - RobotControl can now convert GT *.log files into AIBOVision (external ColorTable32K calibration tool) *.amv file format
*
* Revision 1.5  2004/01/15 15:43:00  kindler
* Added ComboImageProcessor (only temporarily, for debugging only..)
*
* Revision 1.4  2003/12/08 15:01:42  schmidtb
* reactivated RIP
*
* Revision 1.3  2003/12/06 06:31:19  loetzsch
* no message
*
* Revision 1.2  2003/11/20 10:28:59  schmidtb
* Ball Detection added
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/07/30 14:50:02  dueffert
* CheckerboardDetector added
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.14  2003/05/20 15:52:42  deutsch
* Added GridImageProcessorTSL.
*
* Revision 1.13  2003/05/01 13:24:24  roefer
* Copied LinesImageProcessor2 to GT2003ImageProcessor
*
* Revision 1.12  2003/04/15 17:19:43  risler
* moved DDDGO2003 ImageProcessor to own module
* removed ContinuousBasicBehaviorTester
*
* Revision 1.11  2003/04/04 17:18:03  juengel
* AddedHUGO2003ImageProcessor
*
* Revision 1.10  2003/03/12 22:26:22  roefer
* LinesImageProcessor2 added
*
* Revision 1.9  2003/03/05 14:07:09  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.8  2003/03/04 15:02:39  osterhues
* Changed GridImageProcessorDo to GridImageProcessor3
*
* Revision 1.7  2003/01/22 15:00:03  dueffert
* checkerboard stuff added
*
* Revision 1.6  2003/01/09 14:10:39  jhoffman
* no message
*
* Revision 1.5  2003/01/03 17:22:07  deutsch
* added class GridmageProcesserDo
*
* Revision 1.4  2002/11/26 14:19:41  juengel
* GridImageProcessor2 added.
*
* Revision 1.3  2002/11/25 14:50:12  jhoffman
* added motion detector
*
* Revision 1.2  2002/09/12 09:45:58  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:14  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.4  2002/09/07 13:36:55  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.3  2002/08/22 14:41:04  risler
* added some doxygen comments
*
* Revision 1.2  2002/06/02 23:21:09  roefer
* Single color table and progress in LinesSelfLocator
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.6  2002/04/24 10:35:55  jhoffman
* made the sub pixel thing a imagetoperceptcollection rather than an image processor
*
* Revision 1.5  2002/04/23 14:16:47  jhoffman
* added gradient image processor
*
* Revision 1.4  2002/04/02 13:10:19  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.3  2002/02/05 20:02:16  risler
* handleDebugMessage now returns bool, added debug message handling to ImageProcessor
*
* Revision 1.2  2002/02/05 04:00:08  loetzsch
* added a few new module selectors
*
* Revision 1.1  2002/01/31 19:41:24  risler
* ImageProcessorSelector added
*
*
*/
