/** 
* @file ImageProcessor.h
*
* This file contains a generic class for ImageProcessor.
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#ifndef __ImageProcessor_h_
#define __ImageProcessor_h_

#ifdef _WIN32
#pragma warning(disable:4786) 
// the constructor with all it's long parameter type names results in a too
// long debug identifier
#endif

#include "Tools/Module/Module.h"

#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/EdgesPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/SpecialPercept.h"
#include "Representations/Perception/CalibrationRequest.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/RobotState.h"

/**
* @class ImageProcessorInterfaces
* 
* The interfaces of the ImageProcessor module.
*/
class ImageProcessorInterfaces
{
public:
  /** Constructor. */
  ImageProcessorInterfaces(
    const Image& image,
    const CameraMatrix& cameraMatrix,
    ColorTable& colorTable,
    const RobotPose& robotPose,
    const BallModel& ballModel,
    const PlayerPoseCollection& playerPoseCollection,
    const RobotState& robotState,
    const CalibrationRequest& calibrationRequest,
    LandmarksPercept& landmarksPercept,
    BallPercept& ballPercept,
    LinesPercept& linesPercept,
    EdgesPercept& edgesPercept,
    PlayersPercept& playersPercept,
    ObstaclesPercept& obstaclesPercept,
    SpecialPercept& specialPercept)
    : image(image),
    cameraMatrix(cameraMatrix),
    colorTable(colorTable),
    robotPose(robotPose),
    ballModel(ballModel),
    playerPoseCollection(playerPoseCollection),
    robotState(robotState),
    calibrationRequest(calibrationRequest),
    landmarksPercept(landmarksPercept),
    ballPercept(ballPercept),
    linesPercept(linesPercept),
    edgesPercept(edgesPercept),
    playersPercept(playersPercept),
    obstaclesPercept(obstaclesPercept),
    specialPercept(specialPercept)
  {}

  /** The image to be processed */
  const Image& image;

  /** The offset and the rotation of the camera */
  const CameraMatrix& cameraMatrix;

  /** The color table */ 
  ColorTable& colorTable;

  /** The last calculated ball robot pose */
  const RobotPose& robotPose;

  /** The last calculated ball position */
  const BallModel& ballModel;

  /** The last calculated player pose collection */
  const PlayerPoseCollection& playerPoseCollection;

  /** The last calculated robot state */
  const RobotState& robotState;

  /** The calibration request */
  const CalibrationRequest& calibrationRequest;

  /** The landmarks percept to be generated */
  LandmarksPercept& landmarksPercept;

  /** The ball percept to be generated */
  BallPercept& ballPercept;

  /** The lines percept to be generated */
  LinesPercept& linesPercept;

  /** The edges percept to be generated */
  EdgesPercept& edgesPercept;

  /** The players percept to be generated */
  PlayersPercept& playersPercept;

  /** The obstacles percept to be generated */
  ObstaclesPercept& obstaclesPercept;

  /** */
  SpecialPercept& specialPercept; 
};


/**
* @class ImageProcessor 
*
* The base class for image processors.
* ImageProcessors calculate percepts.
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class ImageProcessor : public Module, public ImageProcessorInterfaces
{
public:
/*
* Constructor.
* @param interfaces The parameters of the ImageProcessor module.
  */
  ImageProcessor(const ImageProcessorInterfaces& interfaces)
    : ImageProcessorInterfaces(interfaces)
  {}
  
  /** Destructor */
  virtual ~ImageProcessor() {}
};

#endif // __ImageProcessor_h_

/*
 * Change log :
 * 
 * $Log: ImageProcessor.h,v $
 * Revision 1.3  2004/06/15 10:58:26  thomas
 * added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
 *
 * Revision 1.2  2004/05/22 22:52:03  juengel
 * Renamed ballP_osition to ballModel.
 *
 * Revision 1.1.1.1  2004/05/22 17:19:26  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.6  2004/03/08 01:38:57  roefer
 * Interfaces should be const
 *
 * Revision 1.5  2004/02/16 18:52:57  tim
 * Added BB2004ImageProcessor
 *
 * Revision 1.4  2004/02/03 13:19:49  spranger
 * renamed all references to  class BallP_osition to BallModel
 *
 * Revision 1.3  2003/10/12 20:18:07  juengel
 * ColorTable is not const anymore.
 *
 * Revision 1.2  2003/10/12 11:46:30  juengel
 * Added CalibrationRequest.
 *
 * Revision 1.1  2003/10/06 14:10:13  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:27:49  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.9  2003/03/11 18:50:47  loetzsch
 * disabled VC warning 4786
 *
 * Revision 1.8  2003/03/10 13:54:54  juengel
 * Added ObstaclesPercept to the GridImageProcessorInterfaces.
 *
 * Revision 1.7  2003/02/19 14:59:54  roefer
 * pColorTable -> colorTable
 *
 * Revision 1.6  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.5  2003/01/30 22:31:48  juengel
 * Added LinesPercept to ImageProcessorInterfaces.
 *
 * Revision 1.4  2002/11/28 14:47:49  jhoffman
 * added special percetp for motion detector
 *
 * Revision 1.3  2002/09/17 23:55:21  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.2  2002/09/12 09:45:58  juengel
 * continued change of module/solution mechanisms
 *
 * Revision 1.1  2002/09/10 15:36:14  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.5  2002/09/07 13:36:55  loetzsch
 * unified the vision modules into one module "ImageProcessor"
 * - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
 *   and PlayersPerceptor were are combined to the new solution
 *   "BlobImageProcessor"
 * - The GridImageProcessor and the SubPixelGradientCalculator became
 *   a solution of "ImageProcessor"
 *
 * Revision 1.4  2002/08/30 13:36:19  dueffert
 * removed unused includes
 *
 * Revision 1.3  2002/07/23 13:33:43  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.2  2002/06/02 23:21:09  roefer
 * Single color table and progress in LinesSelfLocator
 *
 * Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.6  2002/04/02 13:10:19  dueffert
 * big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
 *
 * Revision 1.5  2002/02/05 20:02:16  risler
 * handleDebugMessage now returns bool, added debug message handling to ImageProcessor
 *
 * Revision 1.4  2001/12/21 14:09:39  roefer
 * Added several destructors
 *
 * Revision 1.3  2001/12/10 17:47:06  risler
 * change log added
 *
 */
