/**
* @file GT2004GoalRecognizer.h
*
* Definition of class GT2004GoalRecognizer
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef GT2004GoalRecognizer_h_
#define GT2004GoalRecognizer_h_

/**
* @class GoalRecognizer
*
* The goal recognizer recognizes the goals in the image.
*
* The recognizer scans horizontal lines in the image left to right.
* Indications for a goal are grouped and checked vertical.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/ 
class GT2004GoalRecognizer
{
public:
  GT2004GoalRecognizer(
    const Image& image, 
    const CameraMatrix& cameraMatrix, 
    const CameraMatrix& prevCameraMatrix, 
    const ColorTable& colorTable,
    const ColorCorrector& colorCorrector,
    ObstaclesPercept& obstaclesPercept,
    LandmarksPercept& landmarksPercept
    );
  
  ~GT2004GoalRecognizer();

  void execute();

  struct ColoredPartsCheck
  {
    Vector2<int> firstPoint;
    Vector2<int> lastPoint;
    int rangeOfColor;
    int numberOfColoredPixels;
    
    enum{maxNumberOfParts = 20};
    
    int numberOfLargeParts;
    int sizeOfLargePart[maxNumberOfParts];
    Vector2<int> largePartBegin[maxNumberOfParts];
    Vector2<int> largePartEnd[maxNumberOfParts];
    Vector2<double> largePartBeginAngles[maxNumberOfParts];
    Vector2<double> largePartEndAngles[maxNumberOfParts];
    bool largePartBeginIsOnBorder[maxNumberOfParts];
    bool largePartEndIsOnBorder[maxNumberOfParts];
    
    ColoredPartsCheck() 
    {
      numberOfLargeParts = 0;
    }
    
    bool determineLargePart(int size, bool beginIsOnBorder, bool endIsOnBorder, CameraMatrix cameraMatrix, CameraInfo cameraInfo)
    {
      bool foundLargePart = false;
//      if(numberOfColoredPixels > 3)
      {
        if(rangeOfColor > size)
        {
          sizeOfLargePart[numberOfLargeParts] = rangeOfColor;
          largePartBegin[numberOfLargeParts].x = firstPoint.x;
          largePartBegin[numberOfLargeParts].y = firstPoint.y;
          largePartEnd[numberOfLargeParts].x = lastPoint.x;
          largePartEnd[numberOfLargeParts].y = lastPoint.y;
          largePartBeginIsOnBorder[numberOfLargeParts] = beginIsOnBorder;
          largePartEndIsOnBorder[numberOfLargeParts] = endIsOnBorder;
          
          Vector2<double> minAngles, maxAngles;
          Geometry::calculateAnglesForPoint(largePartBegin[numberOfLargeParts], cameraMatrix, cameraInfo, largePartBeginAngles[numberOfLargeParts]);
          Geometry::calculateAnglesForPoint(largePartEnd[numberOfLargeParts], cameraMatrix, cameraInfo, largePartEndAngles[numberOfLargeParts]);
          
          numberOfLargeParts++;
          foundLargePart = true;
          LINE(imageProcessor_flagsAndGoals, firstPoint.x, firstPoint.y, lastPoint.x, lastPoint.y,
            2, Drawings::ps_solid, Drawings::pink);
        }
        
        if(numberOfLargeParts >= maxNumberOfParts)
        {
          numberOfLargeParts = maxNumberOfParts-1;
        }
      }
      return foundLargePart;
    }
  };

private:
  
  /** Calculates the scan lines near the horizon for the goals */
  void calculateScanLinesParallelToHorizon();
  
  /** Calculates the scan lines near the horizon for the goals */
  void calculateScanLinesParallelToHorizon(
    int distanceAboveHorizon,
    int distanceBelowHorizon,
    int numberOfScanLines
    );
  
  /** Scans horizontal for goals */
  void scanHorizontalForGoals();

  /** Calculates the vertical scan lines for the goals */
  void calculateVerticalGoalScanLines();

  /** Scans vertical for goals */
  void scanLinesForGoals();

  DECLARE_DEBUG_IMAGE(imageProcessorGoals);

  /** A reference to the image that is scanned for a goal */
  const Image& image;

  /** A reference to the camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& cameraMatrix;

  /** A reference to the previous camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& prevCameraMatrix;

  /** A reference to the color table */
  const ColorTable& colorTable;

  int goalIndicationAboveHorizon;
  int goalIndicationBelowHorizon;
  bool useFixedScanLines;

  /** A reference to the obstacles percept */
  ObstaclesPercept& obstaclesPercept;

  /** A reference to the obstacles percept */
  LandmarksPercept& landmarksPercept;

  /** A reference to the color corrector */
  const ColorCorrector& colorCorrector;

  /** The color class of the opponent goal */
  colorClass colorOfOpponentGoal;

  /** The color class of the own goal */
  colorClass colorOfOwnGoal;

  /** A representation of the horizon */
  Geometry::Line horizonLine, verticalLine;

  /** The number of horizontal scan lines */
  int numberOfHorizontalScanLines;

  enum{maxNumberOfHorizontalScanLines = 32};
  enum{maxNumberOfGoalScanLines = 255};

  /** representation of the left points of the horizontal scan lines */
  Vector2<int> leftPoint[maxNumberOfHorizontalScanLines];
  
  /** representation of the right points of the horizontal scan lines */
  Vector2<int> rightPoint[maxNumberOfHorizontalScanLines];

  /** the number of indications for goals */
  int numberOfGoalIndications;

  /** left point of an indications for a goal */
  Vector2<int> goalIndicationLeft[maxNumberOfGoalScanLines];
  
  /** left point of an indications for a goal */
  Vector2<int> goalIndicationCenter[maxNumberOfGoalScanLines];
  
  /** right point of an indication for a goal */
  Vector2<int> goalIndicationRight[maxNumberOfGoalScanLines];

  /** true if the left end of the goal indication is on the border of the image */
  bool leftOfGoalIndicationIsOnBorder[maxNumberOfGoalScanLines];
  
  /** true if the right end of the goal indication is on the border of the image */
  bool rightOfGoalIndicationIsOnBorder[maxNumberOfGoalScanLines];

  /** the color class of the goal indication */
  colorClass colorOfGoalIndication[maxNumberOfGoalScanLines];

  /** The number of vertical scan lines where a goal is searched*/
  int numberOfGoalScanLines;

  /** representation of the top points of the goal scan lines */
  Vector2<int> topGoalPoint[maxNumberOfGoalScanLines];
  
  /** representation of the bottom points of the goal scan lines */
  Vector2<int> bottomGoalPoint[maxNumberOfGoalScanLines];
  
  bool scanLineToCheckBestAngle[maxNumberOfGoalScanLines];
  
  /** */
  int indexOfGoalIndication[maxNumberOfGoalScanLines];
  
  /** */
  colorClass colorOfGoalScanLine[maxNumberOfGoalScanLines];

};

#endif // GT2004GoalRecognizer

/*
* $Log: GT2004GoalRecognizer.h,v $
* Revision 1.2  2004/06/05 07:58:21  roefer
* Compensation for motion distortions of images
*
* Revision 1.1.1.1  2004/05/22 17:19:44  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
