/**
* @file GT2004ColorCorrector.h
*
* Definition of class GT2004ColorCorrector
*
* @author Max Risler
*/

#ifndef __GT2004ColorCorrector_h_
#define __GT2004ColorCorrector_h_

#include "Representations/Perception/Image.h"

/**
* @class GT2004ColorCorrector
* This class provides a table to do a radial color correction to compensate 
* for the ERS7 camera image error.
*/
class GT2004ColorCorrector
{
public:

  /**
  * correct an Y pixel value
  * @param x x-coordinate
  * @param y y-coordinate
  * @param c old y value
  * @return corrected y value
  */
  unsigned char correctY(int x, int y, int c) const
  {
    return correctionTable[radiusTable[x][y]][c][0];
  }

  /**
  * correct an U pixel value
  * @param x x-coordinate
  * @param y y-coordinate
  * @param c old u value
  * @return corrected u value
  */
  unsigned char correctU(int x, int y, int c) const
  {
    return correctionTable[radiusTable[x][y]][c][1];
  }

  /**
  * correct an V pixel value
  * @param x x-coordinate
  * @param y y-coordinate
  * @param c old v value
  * @return corrected v value
  */
  unsigned char correctV(int x, int y, int c) const
  {
    return correctionTable[radiusTable[x][y]][c][2];
  }

  /**
  * Constructor
  */
  GT2004ColorCorrector();

  /**
  * Initializes the correction table.
  * Table is created from images in whiteImage.log.
  */
  void init();

  /**
  * Disable the color correction.
  * Table is cleared.
  */
  void disable();
   
  /**
  * Creates a color corrected image.
  *
  * @param image A reference to the image to be corrected
  * @param correctedImage A reference to the image to be created
  */
  void generateCorrectedImage(
    const Image& image, 
    Image& correctedImage) const
  {
    correctedImage.cameraInfo = image.cameraInfo;
    correctedImage.colorTable = image.colorTable;
    for (int x=0; x<image.cameraInfo.resolutionWidth; x++)
      for (int y=0; y<image.cameraInfo.resolutionHeight; y++)
      {
        correctedImage.setHighResY (x, y,
          correctY(x, y, image.getHighResY(x * 2, y * 2)),
          correctY(x, y, image.getHighResY(x * 2, y * 2 + 1)),
          correctY(x, y, image.getHighResY(x * 2 + 1, y * 2)),
          correctY(x, y, image.getHighResY(x * 2 + 1, y * 2 + 1)));

        correctedImage.image[y][1][x] = correctU(x, y, image.image[y][1][x]);
        correctedImage.image[y][2][x] = correctV(x, y, image.image[y][2][x]);
      }
  }
private:

  /**
  * Calculates the corresponding radius to an image coordinate
  */
  int getRadius(int x, int y) const
  {
    int r = (int)sqrt( (double)
        (x - cameraResolutionWidth_ERS7 / 2) * (x - cameraResolutionWidth_ERS7 / 2) +
        (y - cameraResolutionHeight_ERS7 / 2) * (y - cameraResolutionHeight_ERS7 / 2)) - 10;
    if (r < 0) r = 0;
    return r;
  }

  /** the redius table */
  int radiusTable[cameraResolutionWidth_ERS7][cameraResolutionHeight_ERS7];
  /** the correction table */
  unsigned char correctionTable[cameraResolutionWidth_ERS7][256][3];
};

#endif // __GT2004ColorCorrector_h_

/*
* $Log: GT2004ColorCorrector.h,v $
* Revision 1.1.1.1  2004/05/22 17:19:42  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
