/**
* @file GT2004BeaconDetector.h
* 
* Declaration of class GT2004BeaconDetector.
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/ 

#ifndef GT2004BeaconDetector_h_
#define GT2004BeaconDetector_h_

#include "Tools/Math/Vector2.h"
#include "Tools/Math/Matrix2x2.h"
#include "GT2004FlagSpecialist.h"
#include "Modules/ImageProcessor/ImageProcessorTools/ImageProcessorUtilityClasses.h"
#include "Modules/ImageProcessor/ImageProcessorTools/SUSANEdgeDetectionLite.h"
#include "Modules/ImageProcessor/ImageProcessorTools/BresenhamLineScan.h"

#define MAX_NUMBER_OF_PINK_RUNS 20

/**
* @class GT2004BeaconDetector
*
* A class for detecting beacons in images
*/
class GT2004BeaconDetector
{
public:
  
  enum {edgeThresholdU = 13, edgeThresholdV = 15};

  /** Constructor */ 
  GT2004BeaconDetector(const Image& image, const CameraMatrix& cameraMatrix,
                       const CameraMatrix& prevCameraMatrix,
                       const ImageInfo& imageInf,
                       const ColorTable& colorTable, 
                       const ColorCorrector& colorCorrector,
                       LandmarksPercept& landmarksPercept);

  /** Executes the beacon detection*/
  void execute();

  /** Gains information about pink from the current color table*/
  void analyzeColorTable();

private:
  
  /** Edge detection operators (U and V components) used to find the boundaries of flags*/
  const SUSANEdgeDetectionLite edgeDetectionU;
  const SUSANEdgeDetectionLite edgeDetectionV;

  /** The image*/
  const Image& image;
  /** The camera matrix*/
  const CameraMatrix& cameraMatrix;
  /** The previous camera matrix*/
  const CameraMatrix& prevCameraMatrix;
  /** Additional Image information*/
  const ImageInfo& imageInf;
  /** The color table*/
  const ColorTable& colorTable;
  /** The landmarks percept*/
  LandmarksPercept& landmarksPercept;
  /** The number of possible beacons*/
  int numOfBeaconCandidates;
  /** The list of candidates*/
  Run beaconCandidates[MAX_NUMBER_OF_PINK_RUNS];
  /** A buffer for transformed candidates*/
  TransformedRun transformedCandidates[MAX_NUMBER_OF_PINK_RUNS];
  /** The base offset for horizontal scan lines*/
  const double horizontalBaseOffset;
  /** The number of horizontal scan lines above the horizon*/
  const int numOfHorizontalScanLineAbove;
  /** The number of horizontal scan lines below the horizon*/
  const int numOfHorizontalScanLineBelow;
  /** The growth factor of the distance between two scan lines*/
  const double horizontalOffsetModifier;
  /** The maximum distance between the merged runs*/
  const int clusteringDistanceTolerance;
  /** The minimum length of a pink run*/
  const int minPinkRunLength;
  /** The maximum vertical distance of scanlines to be clustered, over the 
  * estimated merged horizontal distance (half beacon aspect ratio should be 1:1) */
  const double clusteringAspectRatio;
  /** The minimum value for the ratio of the 2 dimensions of the pink part of
  * the beacon, to be used for projection on the unknown colored part*/
  const double projectionAspectRatio;
  /** The maximum number of pixels which are examined after an edge response has been found
  * for determining the color beyond the edge*/
  const int edgeScanDepth;
  /** The minimum U channel value of a pink pixel*/
  unsigned char minPinkUValue;
  /** A confidence threshold used to determine if a target beacon, whose type is
  not completely clear, can be accepted or not */
  const double minFlagConfidence;
  /** A flag specialist*/
  GT2004FlagSpecialist flagSpecialist;
  /** The color corrector*/
  const ColorCorrector& colorCorrector;

  /** Adds a pink run to the list of candidates
  * @param pinkRun A pink run
  * @return true, if the run has been appended
  */
  bool addCandidate(const Run& pinkRun);

  /** Scans along a line*/ 
  void scanForPink(const Vector2<int>& start, const Vector2<int>& end);

  /** Scans for one other beacon part*/
  bool scanForBeaconPart(const Vector2<int>& start, const Vector2<int>& end,
                         Vector2<int>& position, Vector2<int>& edge, colorClass& color); 

  /** Clusters the pink elements to find the pink beacon parts*/
  void clusterPinkBeaconParts();

  /** Tries to detect a beacon near a pink part*/
  void analyzeBeacon(const Vector2<double>& left, const double pinkRunWidth);
  
  /** Looks for the vertical edges and the color type of a beacon */
  int scanForBeaconEdges(const Vector2<int>& position, const double pinkRunWidth,  
        Flag::FlagType& flagType, Vector2<int>& topEdge, Vector2<int>& bottomEdge);

  enum {lowReliability = 1, mediumReliability = 4, highReliability = 6};
};

#endif //GT2004BeaconDetector_h_

/*
* $Log: GT2004BeaconDetector.h,v $
* Revision 1.12  2004/07/02 10:11:47  nistico
* Cloned main image processor and created
* SpecialLandmarks specialist for SLAM challenge
*
* Revision 1.11  2004/06/12 17:37:00  nistico
* Eventually, would be nice to have only one Bresenham on the whole
* GT2004ImageProcessor
*
* Revision 1.10  2004/06/09 15:04:14  nistico
* Changed distribution of scanlines
* Some cleanup
* Test result positive
*
* Revision 1.9  2004/06/08 16:00:33  nistico
* Final(?) improvement to the beaconSpecialist: 3 or 4 columns (depending on size)
* are always scanned, and the results are merged based on validity, which is
* calculated from the number of edges found and consistency checks
*
* Revision 1.8  2004/06/05 19:48:45  nistico
* Added one more special situation to BeaconDetector
* imageProcessorGradients now visualized edges used by
* BeaconDetector, for debugging
*
* Revision 1.7  2004/06/05 07:58:21  roefer
* Compensation for motion distortions of images
*
* Revision 1.6  2004/06/04 16:07:33  nistico
* AnalyzeBeacon: more special cases added,
* should be (almost) final
*
* Revision 1.5  2004/06/03 15:13:16  nistico
* AnalyzeBeacon further improved, finds more edges and more redundancy checks
* Scanline spacing reviewed
* Non-rotated coordinates bug fixed
*
* Revision 1.3  2004/06/01 13:07:10  nistico
* All Bresenham functionality encapsulated into the proper class
*
* Revision 1.1.1.1  2004/05/22 17:19:42  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/05/18 18:39:25  nistico
* BeaconDetector improved, the landmarks are recognized much more often,
* and the width is estimated correctly most of the time.
* Some things are still less then ideal, though (i'll post something on the forum),
*
* Revision 1.2  2004/05/14 12:19:24  tim
* fixed bug
*
* Revision 1.1  2004/05/04 13:40:19  tim
* added GT2004ImageProcessor
*
*/
