/**
* @file CheckerboardDetector.h
* 
* Definition of class Checkerboard
*
* @author Uwe Dffert
*/


#ifndef __CheckerboardDetector_h_
#define __CheckerboardDetector_h_

#include "ImageProcessor.h"
#include "Tools/Debugging/DebugImages.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Vector2.h"

/**
* @class CheckerboardDetector
*
* A solution of the ImageProcessor module that looks for vertical checkerboards
* (a row of black and white ~10x10cm tiles with a white boarder and a middle mark
* in eye height that allows precise location of the robot relative to the
* checkerboard from a single image
*
* @author <A href=mailto:dueffert@informatik.hu-berlin.de>Uwe Dffert</A>
*/
class CheckerboardDetector : public ImageProcessor
{
public:
/** 
* Constructor.
* @param interfaces The paramters of the ImageProcessor module.
  */
  CheckerboardDetector(const ImageProcessorInterfaces& interfaces);
  
  /** Executes the module */
  virtual void execute();
  
private:
  DECLARE_DEBUG_IMAGE(imageProcessorGeneral);

  int minY,maxY,maxDelta;
  typedef Vector2<double> v2dArray[100];
  typedef bool bArray[100];

  /** calculates the exact middle of an black white transition in a PixeledLine.
  * @param p1 one point in camera coordinates
  * @param p2 another point in camera coordinates
  * @return the resulting arc difference between these 2 points
  */
  double getAngleBetweenScreenPoints(const Vector2<double>& p1, const Vector2<double>& p2);

  /** calculates the exact middle of an black white transition in a PixeledLine.
  * @param lin the PixeledLine to investigate
  * @param start index of black white transition in lin to investigate
  * @param amount the total change of brightness in this black white transition
  * @return the resulting position of the middle of the black white transition
  */
  Vector2<double> getExactTransitionMiddle(const Geometry::PixeledLine lin, const int start, const int amount);

  /** finds all black white transitions in a PixeledLine
  * @param lin the PixeledLine to investigate
  * @param transPos returned array of found transition middle positions
  * @param transWhiteBlack returned array of booleans whether transition is from white to black or vice versa
  * @param numOfTrans returned number of found transitions
  */
  void getTransitionsOnLine(const Geometry::PixeledLine lin, v2dArray* transPos, bArray* transWhiteBlack, int& numOfTrans);

  /** finds the first transition to white starting from within a black block
  * @param lin the PixeledLine to investigate
  * @return the position of the found transition to white, (-1,-1) otherwise
  */
  Vector2<double> getTransitionToWhite(const Geometry::PixeledLine lin);

  /** calculate a middle perpendicular to t1-t2, find transitions to white on that perpendicular and return the middle between the transitions to white. This gives a better and more secure black block middle than the the middle between t1 and t2 itself.
  * @param t1 beginning of black block
  * @param t2 end of black block
  * @param len returned length of the found perpendicular in black, 0 otherwise
  * @return the position of of found middle of the perpendicular (if it exists), (-1,-1) otherwise
  */
  Vector2<double> getMiddleAndLengthOfPerpendicular(const Vector2<double> t1, const Vector2<double> t2, double& len);

  /** approximates a line f(x)=m*x+n through a number of points
  * @param points an array of points that should be approximated by a line
  * @param numOfPoints number of points in the array
  * @param m the returned Anstieg of the line (f(x)=mx+n)
  * @param n the returned offset of the line (f(x)=mx+n)
  * @return true if a line (candidate) was found
  */
  bool getLineThroughPixelsCandidate(const v2dArray* points, const int numOfPoints, double& m, double& n);

  /** approximates a PixeledLine through a number of points without found elopers
  * @param points an array of points that should be approximated by a line
  * @param numOfPoints number of points in the array
  * @param lin the returned PixeledLine
  * @return true if a line was found
  */
  bool getLineThroughPixels(const v2dArray* points, const int numOfPoints, Geometry::PixeledLine& lin);

  /** calculate a position relative to the checkerboard
  * @param alpha2 absolute angle between the left most and the middle seen black white transition
  * @param a2 absolute distance on the checkboard between the left most and the middle seen black white transition
  * @param alpha1 absolute angle between the middle and the right most seen black white transition
  * @param a1 absolute distance on the checkboard between the middle and the right most seen black white transition
  * @return returned position relative to seen middle transition in checkerboard
  */
  Vector2<double> getPositionFromAngles(const double alpha2,const double a2,const double alpha1,const double a1);

  /** calculate the real y position of a black white transition on the checkerboard
  * @param index index of the black white transition relative to the left side (0) of the large black middle block, left is negative, right positive
  * @return real y position
  */
  double yPosFromTransitionIndex(int index);
};


#endif// __CheckerboardDetector_h_

/*
* $Log: CheckerboardDetector.h,v $
* Revision 1.1.1.1  2004/05/22 17:19:26  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/03/08 01:38:56  roefer
* Interfaces should be const
*
* Revision 1.3  2004/02/29 13:46:48  dueffert
* localization in critical cases improved
*
* Revision 1.2  2003/12/15 10:57:24  dueffert
* calculation bug fixed
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/01 10:20:11  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/08/08 11:24:51  dueffert
* doxygen docu corrected
*
* Revision 1.1  2003/07/30 14:50:02  dueffert
* CheckerboardDetector added
*
* Revision 1.5  2003/02/21 14:16:12  dueffert
* recognition of our checkerboard continued, see logs/special_localisator2.log
*
* Revision 1.4  2003/02/20 15:43:27  dueffert
* bug fixed
*
* Revision 1.3  2003/02/20 15:32:17  dueffert
* work continued
*
* Revision 1.2  2003/02/19 15:57:34  dueffert
* some checkerboard detection work
*
* Revision 1.1  2003/01/22 15:00:03  dueffert
* checkerboard stuff added
*
*
*/
