/**
* @file HeadControl.h
* 
* This file contains a generic class for HeadControl.
*/

#ifndef __HeadControl_h_
#define __HeadControl_h_

#ifdef _WIN32
#pragma warning(disable:4786) 
// the constructor with all it's long parameter type names results in a too
// long debug identifier
#endif


#include "Tools/Module/Module.h"

#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Perception/BodyPosture.h"
#include "Representations/Perception/CameraMatrix.h"

#include "Representations/Cognition/LandmarksState.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/RobotState.h"
#include "Representations/Cognition/BallModel.h"

#include "Representations/Motion/OdometryData.h"
#include "Representations/Motion/HeadControlMode.h"
#include "Representations/Motion/HeadMotionRequest.h"
#include "Representations/Motion/PIDData.h"
#include "Representations/Motion/MotionInfo.h"

#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/RingBuffer.h"

#ifdef _WIN32
#pragma warning(disable:4786) 
// the constructor with all it's long parameter type names results in a too
// long debug identifier
#endif

/**
* @class HeadControlInterfaces
* 
* The interfaces of the HeadControl module.
*/
class HeadControlInterfaces
{
public:
  /** Constructor. */
  HeadControlInterfaces(
    const unsigned long& frameNumber,
    const SensorDataBuffer& sensorDataBuffer, 
    const CameraMatrix& cameraMatrix,
    const OdometryData& currentOdometryData,
    const BodyPosture& bodyPosture,
    const BallModel& ballModel,
    const RobotPose& robotPose,
    const LandmarksState& landmarksState,
    const RobotState& robotState,
    const MotionInfo& motionInfo,
    const HeadControlMode& headControlMode,
    const bool headIsBlockedBySpecialActionOrWalk,
    HeadMotionRequest& headMotionRequest,
    PIDData& pidData)
    :
    frameNumber(frameNumber),
    headControlMode(headControlMode),
    robotPose(robotPose),
    cameraMatrix(cameraMatrix),
    currentOdometryData(currentOdometryData),
    ballModel(ballModel),
    robotState(robotState),
    sensorDataBuffer(sensorDataBuffer),
    bodyPosture(bodyPosture),
    motionRequest(motionRequest),
    motionInfo(motionInfo),
    landmarksState(landmarksState),
    headIsBlockedBySpecialActionOrWalk(headIsBlockedBySpecialActionOrWalk),
    headMotionRequest(headMotionRequest),
    pidData(pidData)
  {}

protected:
  /** A reference to the frame number */
  const unsigned long& frameNumber;

  /** A modus from the behavior how to move the head */
  const HeadControlMode& headControlMode;

  /** The robots current position and orientation */
  const RobotPose& robotPose;
  
  /** The position and rotation of the camera relative to the robot */
  const CameraMatrix& cameraMatrix;

  /** The odometry of the last Motion frame */ 
  const OdometryData& currentOdometryData;

  /** The position and the speed of the ball */
  const BallModel& ballModel;
  
  /** The current body sensor data */ 
  const SensorDataBuffer& sensorDataBuffer; 
  
  /** The neck height and body tilt */
  const BodyPosture& bodyPosture;

  /** motionRequest that is currently executed */
  const MotionRequest& motionRequest;

  /** information about the executed motions */
  const MotionInfo& motionInfo;

  /** the currently seen landmarks */
  const LandmarksState& landmarksState;

  /** The current state of the robot*/
  const RobotState& robotState;

  /** Specifies if the head is blocked by a special action or walk.*/
  const bool headIsBlockedBySpecialActionOrWalk;

  /** Head joint angles that have to be set. */
  HeadMotionRequest& headMotionRequest;

  /** PID servo gains */
  PIDData& pidData;
};

/**
* A generic class for HeadControl modules.
*
* The modul calculates a new head motion which has to be set by the motion module 
* based on the desired head mode and the current
* collection of percepts stored in the world state as well as the current
* sensor informations .
*/
class HeadControl : public Module, public HeadControlInterfaces, public RobotDimensions
{
public:
/*
* Constructor.
* @param interfaces The paramters of the HeadControl module.
  */
  HeadControl(const HeadControlInterfaces& interfaces)
    : HeadControlInterfaces(interfaces),
      RobotDimensions(getRobotConfiguration().getRobotDimensions())
  {}
  
  /** Destructor */
  virtual ~HeadControl() {}
};

#endif //__HeadControl_h_

/*
* Change log :
* 
* $Log: HeadControl.h,v $
* Revision 1.8  2004/05/29 18:18:56  dueffert
* walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
*
* Revision 1.7  2004/05/27 18:42:24  loetzsch
* removed warnings
*
* Revision 1.6  2004/05/26 18:36:03  loetzsch
* cleanup in the head control interfaces
*
* Revision 1.5  2004/05/26 17:21:47  dueffert
* better data types used
*
* Revision 1.4  2004/05/25 12:49:07  tim
* added RobotState to HeadControl
*
* Revision 1.3  2004/05/23 18:56:33  spranger
* added framenumber to HeadControl interface
*
* Revision 1.2  2004/05/22 22:52:03  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:19:21  cvsadm
* created new repository GT2004_WM
*
* Revision 1.13  2004/05/18 11:38:16  loetzsch
* the Motion process now also executes a SensorDataProcessor for the calculation of the cameraMatrix
*
* Revision 1.12  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.11  2004/05/10 10:30:15  juengel
* Added executedMotionRequest to HeadControlInterfaces.
*
* Revision 1.10  2004/04/07 14:42:56  risler
* moved LandsmarksState to Cognition directory, generated by SelfLocator
*
* Revision 1.9  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.2  2004/04/01 19:52:45  Charlie
* added LandmarkState
*
* Revision 1.1.1.1  2004/03/29 08:28:48  Administrator
* initial transfer from tamara
*
* Revision 1.8  2004/03/20 09:55:24  roefer
* Preparation for improved odometry
*
* Revision 1.7  2004/03/08 01:38:54  roefer
* Interfaces should be const
*
* Revision 1.6  2004/03/04 16:00:19  fritsche
* added pidData to HeadControlInterfaces
*
* Revision 1.5  2004/03/04 10:05:20  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.4  2004/02/03 13:19:49  spranger
* renamed all references to  class BallP_osition to BallModel
*
* Revision 1.3  2004/01/01 10:58:50  roefer
* RobotDimensions are in a class now
*
* Revision 1.2  2003/11/24 16:11:10  dueffert
* SpecialPercept removed from PackageCognition
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.27  2003/06/16 20:01:43  loetzsch
* bug fix
*
* Revision 1.26  2003/06/15 22:42:22  loetzsch
* no message
*
* Revision 1.25  2003/06/15 16:44:24  jhoffman
* no message
*
* Revision 1.24  2003/05/28 15:34:27  dueffert
* new GT2003HeadControl
*
* Revision 1.23  2003/05/27 16:16:30  juengel
* Added headIsBlockedBySpecialActionOrWalk.
*
* Revision 1.22  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.21  2003/04/16 07:00:15  roefer
* Bremen GO checkin
*
* Revision 1.21  2003/04/08 16:26:15  timrie
* added current MotionRequest
*
* Revision 1.20  2003/04/07 16:50:37  jhoffman
* parameter tuning and cleaning up
*
* Revision 1.19  2003/04/06 21:02:19  roefer
* Mouth added to setJoints
*
* Revision 1.18  2003/04/02 15:35:13  jhoffman
* another round of bug fixes
*
* Revision 1.17  2003/04/02 12:28:00  roefer
* HeadControl additions from yesterday restored
*
* Revision 1.16  2003/04/02 09:25:58  jhoffman
* everything back to how it was before; but there's the additional
* gt2003 with all the old functions dublicated.
*
* Revision 1.9  2003/03/06 11:58:29  dueffert
* re-order warning removed
*
* Revision 1.8  2002/11/28 14:46:30  jhoffman
* moved "setjoints" in in the class HeadControl
*
* Revision 1.7  2002/11/05 19:22:28  loetzsch
* made the reference to the HeadState const and changed the comment
*
* Revision 1.6  2002/09/17 23:55:21  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.5  2002/09/12 13:17:27  loetzsch
* made member variables of module interfaces protected
*
* Revision 1.4  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.3  2002/09/11 17:26:31  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.2  2002/09/10 21:07:30  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:14  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.7  2002/04/24 18:16:05  risler
* HeadState not const in HeadControl
*
* Revision 1.6  2002/04/23 10:38:28  risler
* renamed headOdometry to headState
*
* Revision 1.5  2002/04/02 13:10:19  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.4  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.3  2001/12/11 13:39:11  kosen
* parameter SensorData added
*
* Revision 1.2  2001/12/10 17:47:06  risler
* change log added
*
*/
