/** 
* @file GT2004HeadControlBasicBehaviors.h
*
* Declaration of basic behaviors defined in "basic-behaviors.xml".
*
* @author Martin Ltzsch
*/

#ifndef __GT2004HeadControlBasicBehaviors_h_
#define __GT2004HeadControlBasicBehaviors_h_

#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"


/** 
* @class GT2004HeadControlBasicBehavior
*
* Base class for all basic behaviors in the GT2004HeadControl
*
* @author Martin Ltzsch
*/
class GT2004HeadControlBasicBehavior : public Xabsl2BasicBehavior, public HeadControlInterfaces
{
public:
/*
* Constructor.
* @param name The name of the basic behavior
* @param errorHandler Is invoked when errors occur
* @param interfaces The interfaces of the HeadControl module
* @param headControl A reference to the headControl module
* @param headPathPlanner A reference to the head path planner 
* @param lastScanWasLeft A reference to the variable "lastScanWasLeft" in the HeadControl
* @param cameraInfor A reference to the camera info in the head control
  */
  GT2004HeadControlBasicBehavior(const char* name, 
    Xabsl2ErrorHandler& errorHandler,
    HeadControlInterfaces interfaces,
    GT2004HeadControl& headControl,
    GT2004HeadPathPlanner& headPathPlanner,
    bool& lastScanWasLeft,
    CameraInfo& cameraInfo)
    : Xabsl2BasicBehavior(name, errorHandler),
    HeadControlInterfaces(interfaces),
    headControl(headControl),
    headPathPlanner(headPathPlanner),
    lastScanWasLeft(lastScanWasLeft),
    cameraInfo(cameraInfo)
  {}
  
  /**  Executes the basic behavior. */
  virtual void execute() = 0;
  
protected:
  /** A reference to the GT2004HeadControl */
  GT2004HeadControl& headControl;

  /** A reference to the head path planner */
  GT2004HeadPathPlanner& headPathPlanner;

  /** A reference to the variable "lastScanWasLeft" in the head control */
  bool& lastScanWasLeft;

  /** A reference to a camara info in the head control */
  CameraInfo& cameraInfo;
  
  /** current odometry data of last call to head control */
  OdometryData lastOdometryData;
  /** time of last good self localization */
  unsigned long lastTimeOfGoodSL;

  /** Number of executed scans */
  int panCount;
};

/** 
* Declares a class for a basic behavior in the GT2004HeadControl 
* @param className The name of the basic behavior class
* @param xmlName The string that is used to register the basic behavior at the engine.
*/
#define DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(className,xmlName) \
  class className : public GT2004HeadControlBasicBehavior \
{ \
public:\
  className(Xabsl2ErrorHandler& errorHandler, HeadControlInterfaces& interfaces, GT2004HeadControl& headControl,GT2004HeadPathPlanner& headPathPlanner,bool& lastScanWasLeft,CameraInfo& cameraInfo) \
  : GT2004HeadControlBasicBehavior(xmlName, errorHandler, interfaces, headControl,headPathPlanner,lastScanWasLeft,cameraInfo) {} \
  virtual void execute();\
};  

/** classes for the basic behaviors which have neither parameters nor additional members */
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorOtherHeadMovements,"other-head-movements");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorLookAtBall,"look-at-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorLookAtBallAndClosestLandmark, "look-at-ball-and-closest-landmark");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorBeginBallSearchAtBallPositionSeen, "begin-ball-search-at-seen");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorBeginBallSearchAtBallPositionCommunicated, "begin-ball-search-at-communicated");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorBeginBallSearchAtBallPositionPropagated, "begin-ball-search-at-propagated");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorLookAroundAtSeenBall, "look-around-at-seen-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorFindBall,"find-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorReturnToBall,"return-to-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorScanBackToBall,"scan-back-to-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorScanAwayFromBall,"scan-away-from-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorGrabBall,"grab-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorReleaseBall,"release-ball");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorWaitForGrab,"wait-for-grab");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorLookAtCloseLandmark,"look-at-close-landmark");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorSearchForBallLeft,"search-for-ball-left");
DECLARE_GT2004_HEADCONTROL_BASIC_BEHAVIOR(GT2004BasicBehaviorSearchForBallRight,"search-for-ball-right");


/** The basic behavior "look-at-close-landmarks" */
class GT2004BasicBehaviorDirectedScanForLandmarks : public GT2004HeadControlBasicBehavior \
{ 
public:
  /** Constructor */
  GT2004BasicBehaviorDirectedScanForLandmarks(Xabsl2ErrorHandler& errorHandler, HeadControlInterfaces& interfaces, GT2004HeadControl& headControl,GT2004HeadPathPlanner& headPathPlanner,bool& lastScanWasLeft,CameraInfo& cameraInfo) 
  : GT2004HeadControlBasicBehavior("directed-scan-for-landmarks", errorHandler, interfaces, headControl,headPathPlanner,lastScanWasLeft,cameraInfo), leftOrRight(1) {} 

  /** Executes the basic behavior */
  virtual void execute();

  bool nextLandmarkIsWithinReach;

private:
  double leftOrRight;
  int currentLandmark;
  int nextLandmark;
  int waitSomeBeforeLookingAtNextLandmark;
};


/** Initializes a GT2004HeadControl basic behavior with the right parameter list */
#define INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehavior) \
basicBehavior(errorHandler, interfaces, headControl, headPathPlanner, lastScanWasLeft, cameraInfo)


/**
* @class GT2004HeadControlBasicBehaviors
*
* Creates and registers the basic behaviors
*/
class GT2004HeadControlBasicBehaviors
{
public:
  /** Constructor */
  GT2004HeadControlBasicBehaviors(Xabsl2ErrorHandler& errorHandler, HeadControlInterfaces& interfaces, GT2004HeadControl& headControl,GT2004HeadPathPlanner& headPathPlanner,bool& lastScanWasLeft,CameraInfo& cameraInfo)
    : errorHandler(errorHandler), headControl(headControl),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorLookAtBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorLookAtCloseLandmark),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorLookAtBallAndClosestLandmark),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorBeginBallSearchAtBallPositionSeen),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorBeginBallSearchAtBallPositionPropagated),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorBeginBallSearchAtBallPositionCommunicated),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorFindBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorReturnToBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorScanAwayFromBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorScanBackToBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorGrabBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorReleaseBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorWaitForGrab),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorOtherHeadMovements),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorDirectedScanForLandmarks),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorLookAroundAtSeenBall),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorSearchForBallLeft),
    INIT_GT2004_HEADCONTROL_BASIC_BEHAVIOR(basicBehaviorSearchForBallRight)
  {}
  
  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);  
  
  //!@name Basic Behaviors
  //!@{
  GT2004BasicBehaviorLookAtBall basicBehaviorLookAtBall;
  GT2004BasicBehaviorFindBall basicBehaviorFindBall;
  GT2004BasicBehaviorReturnToBall basicBehaviorReturnToBall;
  GT2004BasicBehaviorScanAwayFromBall basicBehaviorScanAwayFromBall;
  GT2004BasicBehaviorDirectedScanForLandmarks basicBehaviorDirectedScanForLandmarks;
  GT2004BasicBehaviorScanBackToBall basicBehaviorScanBackToBall;
  GT2004BasicBehaviorGrabBall basicBehaviorGrabBall;
  GT2004BasicBehaviorReleaseBall basicBehaviorReleaseBall;
  GT2004BasicBehaviorWaitForGrab basicBehaviorWaitForGrab;
  GT2004BasicBehaviorBeginBallSearchAtBallPositionSeen basicBehaviorBeginBallSearchAtBallPositionSeen;
  GT2004BasicBehaviorBeginBallSearchAtBallPositionCommunicated basicBehaviorBeginBallSearchAtBallPositionCommunicated;
  GT2004BasicBehaviorBeginBallSearchAtBallPositionPropagated basicBehaviorBeginBallSearchAtBallPositionPropagated;
  GT2004BasicBehaviorOtherHeadMovements basicBehaviorOtherHeadMovements;
  GT2004BasicBehaviorLookAtCloseLandmark basicBehaviorLookAtCloseLandmark;
  GT2004BasicBehaviorLookAtBallAndClosestLandmark basicBehaviorLookAtBallAndClosestLandmark;
  GT2004BasicBehaviorLookAroundAtSeenBall basicBehaviorLookAroundAtSeenBall;
  GT2004BasicBehaviorSearchForBallLeft basicBehaviorSearchForBallLeft;
  GT2004BasicBehaviorSearchForBallRight basicBehaviorSearchForBallRight;
  //!@}

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;
  
  /** A reference to the GT2004HeadControl */
  GT2004HeadControl& headControl;
};

#endif // __GT2004HeadControlBasicBehaviors_h_

/*
* Change Log
* 
* $Log: GT2004HeadControlBasicBehaviors.h,v $
* Revision 1.18  2004/07/01 18:21:16  dassler
* Added BasicBehavior and HeadControlMode:
* search-for-ball-left
* search-for-ball-right
* Test is needed
*
* Revision 1.17  2004/06/29 17:47:40  schmitt
* lookAtCloseLandmark updated, if no landmark is seen
*
* Revision 1.16  2004/06/28 14:00:04  jhoffman
* - scan back to ball slower
* - directed scan briefly stops at landmarks
* - input symbol set-joints-is-close-to-destination implemented and added to behavior
* - merged marcs changes into track-ball-strict
*
* Revision 1.15  2004/06/19 12:32:12  jhoffman
* - directed scan briefly stops before looking at the next landmark
* - comments added
*
* Revision 1.14  2004/06/18 18:28:39  dassler
* introduced basic-behavior:
* BeginBallSearchAtBallPositionSeen
* BeginBallSearchAtBallPositionCommunicated
* BeginBallSearchAtBallPositionPropagated
*
* track-ball modified and build in ball-just-lost
*
* Revision 1.13  2004/06/17 15:26:45  jhoffman
* various improvements to headcontrol
*
* Revision 1.12  2004/06/16 18:14:00  jhoffman
* - search-for-landmarks-bug removed (setjointsdirect now sets "headPathPlanner.last...")
*
* Revision 1.11  2004/06/14 20:12:10  jhoffman
* - numerous changes and additions to headcontrol
* - cameraInfo default constructor now creates ERS7 info
* - debug drawing "headcontrolfield" added
*
* Revision 1.10  2004/06/11 16:28:46  juengel
* Added basic behavior release-ball.
*
* Revision 1.9  2004/06/05 16:02:49  jhoffman
* - added look-at-ball-and-closest-landmark (basicbahavior)
* - found and removed bug in simple-look-at-point
*
* Revision 1.8  2004/05/29 18:18:56  dueffert
* walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
*
* Revision 1.7  2004/05/27 17:13:37  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.6  2004/05/25 19:26:38  tim
* renamed basic behavior
*
* Revision 1.5  2004/05/25 18:30:12  jhoffman
* landmarks-array in field dimensions
* headcontrol looks at close landmark
*
* Revision 1.4  2004/05/25 17:41:27  tim
* added look-straight-ahead BasicBehavior
*
* Revision 1.3  2004/05/25 13:35:27  tim
* added grab-ball BasicBehavior
*
* Revision 1.2  2004/05/23 22:49:14  loetzsch
* simplified basic behaviors
*
* Revision 1.1.1.1  2004/05/22 17:19:24  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/18 13:40:00  loetzsch
* registered symbols and basic behaviors for GT2004HeadControl,
* renamed some states and basic behaviors
*
*/

