/**
* @file Xabsl2BehaviorControl.h
* 
* Implementation of class Xabsl2BehaviorControl.
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#ifndef __Xabsl2BehaviorControl_h_
#define __Xabsl2BehaviorControl_h_

#include "BehaviorControl.h"
#include "Tools/Xabsl2/GT/GTXabsl2EngineExecutor.h"


/**
* @class Xabsl2BehaviorControl
*
* Base class for all BehaviorControl solutions that use a Xabsl2Engine
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/ 
class Xabsl2BehaviorControl : public BehaviorControl, public GTXabsl2EngineExecutor
{
public:
/** 
* Constructor.
* @param interfaces The paramters of the BehaviorControl module.
* @param id The id of the Xabsl2Engine.
  */
  Xabsl2BehaviorControl(const BehaviorControlInterfaces& interfaces, 
    SolutionRequest::xabsl2EngineID id);
  
  /** destructor */
  ~Xabsl2BehaviorControl() {};

  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the messag was read
  */
  virtual bool handleMessage(InMessage& message);
  
protected:

  /** Is called if the engine could not be created */
  virtual void executeIfEngineCouldNotBeCreated();

  /** 
  * Prints the main action that was generated by the execution of the engine to a string
  * @param buf the string where to print the action
  */
  virtual void printGeneratedMainActionToString(char* buf);

};

/**
* Allows to register the same Xabsl2BehaviorControl instance more than once
* at a ModuleHandler to be able to switch between different agents using the
* GT Module mechanism.
*/
class Xabsl2BehaviorControlAgentInterface : public BehaviorControl, public Xabsl2NamedItem
{
public:
/**
* Constructor
* @param name The name of the agent
* @param behaviorControl A newly created Xabsl2BehaviorControl can contains the agent.
* @param interfaces The paramters of the BehaviorControl module.
  */
  Xabsl2BehaviorControlAgentInterface(const char* name,
    Xabsl2BehaviorControl* behaviorControl,
    const BehaviorControlInterfaces& interfaces);
  
  /** Destructor. Deletes the Xabsl2BehaviorControl */
  ~Xabsl2BehaviorControlAgentInterface();

  /** Executes the agent */
  virtual void execute();
  
  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the messag was read
  */
  virtual bool handleMessage(InMessage& message);
  
private:
  /** A pointer to the Xabsl2BehaviorControl can contains the agent. */
  Xabsl2BehaviorControl* behaviorControl;
};

#endif// __Xabsl2BehaviorControl_h_

/*
* Change log :
* 
* $Log: Xabsl2BehaviorControl.h,v $
* Revision 1.1.1.1  2004/05/22 17:15:31  cvsadm
* created new repository GT2004_WM
*
* Revision 1.7  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.6  2004/03/08 00:58:53  roefer
* Interfaces should be const
*
* Revision 1.5  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.4  2003/11/17 15:14:49  dueffert
* doxygen docu corrected
*
* Revision 1.3  2003/10/26 22:49:34  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
* Revision 1.2  2003/09/30 10:51:13  dueffert
* typos fixed
*
* Revision 1.1  2003/07/21 19:18:05  loetzsch
* - Xabsl2 AND and OR operators now can contain more than 2 operands
* - speed improvements and cleanup
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/01/28 18:07:47  loetzsch
* no message
*
* Revision 1.8  2003/01/22 18:09:23  loetzsch
* continued implementation of sending of intermediate code from xabsl2 dialog
* to a xabsl2 behavior control
*
* Revision 1.7  2003/01/21 20:36:14  loetzsch
* xabsl2 intermediate code now can be handled by Xabsl2BehaviorControl
*
* Revision 1.6  2003/01/20 08:14:16  loetzsch
* added xabsl2BehaviorControlID to the solutionRequest to distinct between the
* different Xabsl2BehaviorControls
*
* Revision 1.5  2003/01/19 13:04:51  loetzsch
* xabsl2 agents now can be changed by using the Module and SolutionRequest
* mechanism
*
* Revision 1.4  2003/01/12 14:54:04  loetzsch
* continued creation of option tree: Xabsl2Statement and derivates added
*
* Revision 1.3  2003/01/09 10:01:20  loetzsch
* added interfaces to the Xabsl2 Dialog in the RobotControl application
*
* Revision 1.2  2003/01/08 15:22:33  loetzsch
* - started implementation of the option tree
* - started the reading of the intermediate code
*
* Revision 1.1  2002/12/18 11:02:02  loetzsch
* - moved common parts of Humboldt2003BehaviorControl to new base class Xabsl2BehaviorControl
* - Xabsl2BehaviorControl shall become the base class for all BehaviorControl solutions based on the Xabsl2Engine
*
*/
