/**
* @file KickLogger.h
* 
* Definition of class KickLogger.
*
* @author Martin Ltzsch
*/

#ifndef __KickLogger_h_
#define __KickLogger_h_

#include "BehaviorControl.h"
#include "Tools/KickSelectionTable.h"

/**
* @class KickLogger
*
* Runs in parallel to a xabsl behavior control and logs the preconditions
* and results of each kick.
*
* @author Martin Ltzsch
*/ 
class KickLogger : public BehaviorControlInterfaces
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the BehaviorControl module.
  */
  KickLogger(BehaviorControlInterfaces& interfaces);

  /** destructor */
  ~KickLogger();

  /** Executes the logger */
  void execute();

private:
  /** Different states of the shoot logger */
  enum State { initial, kickRequested, kickStarted, kickFinished, 
    waitingForBall, wait };

  /** The state of the logger */
  State state;

  /** Changes the state of the logger */
  void changeState(State newState);

  /** The time of the last state change */
  unsigned long timeOfLastStateChange;

  /** 
  * Writes a collected record if requested by 
  * a debug key to the debug message queue.
  */
  void record();

  /** The kick record */
  KickCase kick;

  /** The requested kick */
  SpecialActionRequest::SpecialActionID kickID;

  /** The odometry data when the kick was started */
  OdometryData odometryDataWhenKickWasStarted;

  /** Returns the relative x position of a ball position */
  double getBallX(Pose2D from);

  /** Returns the relative y position of a ball position */
  double getBallY(Pose2D from);
};

#endif// __KickLogger_h_

/*
 * Change log :
 * 
 * $Log: KickLogger.h,v $
 * Revision 1.3  2004/06/20 15:22:35  risler
 * increased kick editor usability:
 * KickLogger now sends kick record via debug key instead of writing to file
 * KickEditor automatically displays sent kick records
 * KickCase moved to KickSelectionTable
 *
 * Revision 1.2  2004/06/02 17:18:24  spranger
 * MotionRequest cleanup
 *
 * Revision 1.1  2004/05/27 16:48:08  loetzsch
 * added the kick logger
 *
 * Revision 1.3  2004/03/20 00:20:18  kerdels
 * removed some errors
 *
 * Revision 1.1  2004/03/16 14:00:17  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.1  2004/03/06 12:52:11  loetzsch
 * cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
 *
 * Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
 * created local cvs for Gnne
 *
 * Revision 1.4  2004/02/04 14:09:56  loetzsch
 * improved
 *
 * Revision 1.3  2004/02/03 23:16:46  loetzsch
 * added debug key sendShootRecords
 *
 * Revision 1.2  2004/01/30 13:25:28  loetzsch
 * using odometry data in shoot logger,
 * first draft of visualization
 *
 * Revision 1.1  2004/01/28 21:16:56  loetzsch
 * added ATH2004ShootLogger
 *
 */
