/** 
* @file GT2004ConfigurationSymbols.cpp
*
* Implementation of class GT2004ConfigurationSymbols.
*
* @author Max Risler
*/

#include "GT2004ConfigurationSymbols.h"

GT2004ConfigurationSymbols::GT2004ConfigurationSymbols(
                           BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces)
{
  currentKickoff = 0;
    // start position for goalie (both kickoffs)
    goalie.x = -2000.0;
    goalie.y = 50;
    // start position for defensive supporter (both kickoffs)
    defensiveSupporter.x = -1000.0;
    defensiveSupporter.y = -200.0;

    // start position for offensive supporter (opponent kickoffs)
    offensiveSupporterOpponentKickoff.x = -1000.0;
    offensiveSupporterOpponentKickoff.y = -800.0;
    // start position for striker (opponent kickoffs)
    strikerOpponentKickoff.x = -1000.0;
    strikerOpponentKickoff.y = 500.0;

    // long kickoff to the right
    offensiveSupporterOwnKickoff[0].x = -100.0;
    offensiveSupporterOwnKickoff[0].y = -800.0;
    offensiveSupporterOwnKickoffAngle[0] = 0.0;
    strikerOwnKickoff[0].x = -130.0;
    strikerOwnKickoff[0].y = 130.0;
    strikerOwnKickoffAngle[0] = -45.0;

    // long kickoff to the left
    offensiveSupporterOwnKickoff[1].x = -100.0;
    offensiveSupporterOwnKickoff[1].y = 800.0;
    offensiveSupporterOwnKickoffAngle[1] = 0.0;
    strikerOwnKickoff[1].x = -130.0;
    strikerOwnKickoff[1].y = -130.0;
    strikerOwnKickoffAngle[1] = 45.0;

}

void GT2004ConfigurationSymbols::registerSymbols(Xabsl2Engine& engine)
{
  engine.registerDecimalInputSymbol("current-kickoff-variant",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getCurrentKickoff);

  engine.registerDecimalInputSymbol("own-kickoff.angle",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getOwnKickoffAngle);
  engine.registerDecimalInputSymbol("own-kickoff.x",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getOwnKickoffX);
  engine.registerDecimalInputSymbol("own-kickoff.y",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getOwnKickoffY);

  engine.registerDecimalInputSymbol("opponent-kickoff.angle",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getOpponentKickoffAngle);
  engine.registerDecimalInputSymbol("opponent-kickoff.x",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getOpponentKickoffX);
  engine.registerDecimalInputSymbol("opponent-kickoff.y",this,
    (double (Xabsl2FunctionProvider::*)())&GT2004ConfigurationSymbols::getOpponentKickoffY);
}

void GT2004ConfigurationSymbols::update()
{
  currentKickoff = gameControlData.data.opponentScore % numOfKickoff;

  // calculate kickoff position of striker based on obstacle-model
  double radius = 250.0;

  double leftAbs = fromDegrees(10.0);
  double rightAbs = fromDegrees(55.0);
  double freeWidth = fromDegrees(15.0);

  double step = fromDegrees(5.0);
  double bestAngle = 0.0;
  double bestDist = 0.0;
  
  double start, end;
  enum { left, right } direction;

  switch (currentKickoff % 2)
  {
    case 0: // right
      start = -1 * leftAbs;
      end = -1 * rightAbs;
      step *= -1;
      direction = right;
      break;
    case 1: // left
      start = leftAbs;
      end = rightAbs;
      step = step;
      direction = left;
      break;
  }

  for(
    double ang = start;
    (direction == left && ang <= end) || (direction == right && ang >= end);
    ang += step
  )
  {
    double currDist = obstaclesModel.getDistanceInCorridor(ang, freeWidth);
    double angAbs = ang - robotPose.getAngle();
    currDist = obstaclesModel.getDistanceInDirection(angAbs, freeWidth);
    if(currDist > bestDist)
    {
      bestDist = currDist;
      bestAngle = ang;
    }
  }

  double posX = - cos ( bestAngle ) * radius;
  double posY = - sin ( bestAngle ) * radius;

  obstacleKickoff.translation.x = posX;
  obstacleKickoff.translation.y = posY;
  obstacleKickoff.fromAngle(toDegrees(bestAngle));
}

double GT2004ConfigurationSymbols::getCurrentKickoff()
{
  return currentKickoff;
}

double GT2004ConfigurationSymbols::getOwnKickoffAngle()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    case Player::two: // robot number 2
    default:
      return 0;
    case Player::three: // robot number 3
      return offensiveSupporterOwnKickoffAngle[currentKickoff];
    case Player::four: // robot number 4
      return getOwnKickoffAngleByObstacles();
      //return strikerOwnKickoffAngle[currentKickoff];
  }
}

double GT2004ConfigurationSymbols::getOwnKickoffX()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return goalie.x;
    case Player::two: // robot number 2
      return defensiveSupporter.x;
    case Player::three: // robot number 3
      return offensiveSupporterOwnKickoff[currentKickoff].x;
    case Player::four: // robot number 4
      return getOwnKickoffXByObstacles();
      //return strikerOwnKickoff[currentKickoff].x;
  }
}

double GT2004ConfigurationSymbols::getOwnKickoffY()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return goalie.y;
    case Player::two: // robot number 2
      return defensiveSupporter.y;
    case Player::three: // robot number 3
      return offensiveSupporterOwnKickoff[currentKickoff].y;
    case Player::four: // robot number 4
      return getOwnKickoffYByObstacles();
      //return strikerOwnKickoff[currentKickoff].y;
  }
}

double GT2004ConfigurationSymbols::getOwnKickoffXByObstacles()
{
  return obstacleKickoff.translation.x;
}

double GT2004ConfigurationSymbols::getOwnKickoffYByObstacles()
{
  return obstacleKickoff.translation.y;
}

double GT2004ConfigurationSymbols::getOwnKickoffAngleByObstacles()
{
  return obstacleKickoff.getAngle();
}

double GT2004ConfigurationSymbols::getOpponentKickoffAngle()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    case Player::two: // robot number 2
    case Player::three: // robot number 3
    case Player::four: // robot number 4
    default:
      return 0;
  }
}

double GT2004ConfigurationSymbols::getOpponentKickoffX()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return goalie.x;
    case Player::two: // robot number 2
      return defensiveSupporter.x;
    case Player::three: // robot number 3
      return offensiveSupporterOpponentKickoff.x;
    case Player::four: // robot number 4
      return strikerOpponentKickoff.x;
  }
}

double GT2004ConfigurationSymbols::getOpponentKickoffY()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return goalie.y;
    case Player::two: // robot number 2
      return defensiveSupporter.y;
    case Player::three: // robot number 3
      return offensiveSupporterOpponentKickoff.y;
    case Player::four: // robot number 4
      return strikerOpponentKickoff.y;
  }
}

/*
* Change Log
* 
* $Log: GT2004ConfigurationSymbols.cpp,v $
* Revision 1.2  2004/07/02 16:52:55  thomas
* modified kickoff
* - striker positioning to angle based on obstacle-model
* - supporter not so near to side-border and a bit more away from middle-line
*
* Revision 1.1.1.1  2004/05/22 17:18:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/02 13:23:45  juengel
* Added GT2004BehaviorControl.
*
*/

