/** 
* @file EvolutionSymbols.cpp
*
* Implementation of class EvolutionSymbols.
*
* @author Martin Ltzsch
*/

#include "EvolutionSymbols.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Math/Pose2D.h"

EvolutionSymbols::EvolutionSymbols(const BehaviorControlInterfaces& interfaces,
                                 UDParametersSet& udParametersSet,
                                 Pose2D* udParametersCalibration,
                                 int& udCurrentIndex,
                                 Pose2D& measurementRequest,
                                 int& evolutionMode):
BehaviorControlInterfaces(interfaces),
udParametersSet(udParametersSet),
udParametersCalibration(udParametersCalibration),
udCurrentIndex(udCurrentIndex),
measurementRequest(measurementRequest),
evolutionMode(evolutionMode),
timeWhenBallWasStartedToCatch(0), timeUntilBallWasCaught(0),
role(BehaviorTeamMessage::goalie),
estimatedTimeToReachBall(0.0),
timeOfLastGoodSelfLocalization(SystemCall::getCurrentSystemTime())
{
}


void EvolutionSymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "duration-of-bad-self-localization"
  engine.registerDecimalInputSymbol("duration-of-bad-self-localization", &durationOfBadSelfLocalization);
  
  // "current-ud-parameters-start-position.*"
  engine.registerDecimalInputSymbol("current-ud-parameters-start-position.x",  &currentUDParametersStartPosition.x);
  engine.registerDecimalInputSymbol("current-ud-parameters-start-position.y",  &currentUDParametersStartPosition.y);
  engine.registerDecimalInputSymbol("current-ud-parameters-start-position.angle",  &currentUDParametersStartDirection);
  // "current-ud-parameters-measure-blind"
  engine.registerBooleanInputSymbol("current-ud-parameters-measure-blind",
    this,(bool(Xabsl2FunctionProvider::*)())&EvolutionSymbols::getCurrentUDParametersMeasureBlind);
  // "quality-of-current-ud-parameters"
  engine.registerDecimalInputSymbol("quality-of-current-ud-parameters",this,
    (double (Xabsl2FunctionProvider::*)())&EvolutionSymbols::getQualityOfCurrentUDParameters);
}

void EvolutionSymbols::update()
{
  if (robotPose.getValidity()>0.5)
  {
    timeOfLastGoodSelfLocalization=SystemCall::getCurrentSystemTime();
  }
  durationOfBadSelfLocalization=0.001*SystemCall::getTimeSince(timeOfLastGoodSelfLocalization);
  
  //calculate how much we will walk in N s (see MotionRequestResult.nb).
  //we use a mixture of theory and last measurement:
  Pose2D motReq=measurementRequest;
  if (udParametersCalibration[udCurrentIndex].translation.abs()!=0)
  {
    motReq.translation += udParametersCalibration[udCurrentIndex].translation;
    motReq.translation /= 2;
    motReq.rotation += udParametersCalibration[udCurrentIndex].rotation;
    motReq.rotation /= 2;
  }
  double N=4.75;
  Pose2D afterNs;
  Pose2D afterN2s;
  afterNs.rotation=motReq.rotation*N;
  afterN2s.rotation=afterNs.rotation/2;
  if (fabs(afterNs.rotation)<0.0001)
  {
    afterNs.translation=motReq.translation*N;
    afterN2s.translation=afterNs.translation/2;
  }
  else
  {
    double dx=motReq.translation.x;
    double dy=motReq.translation.y;
    double dr=motReq.rotation;
    afterNs.translation.x=(dy*(cos(dr*N)-1)+dx*sin(dr*N))/dr;
    afterNs.translation.y=(dx*(1-cos(dr*N))+dy*sin(dr*N))/dr;
    afterN2s.translation.x=(dy*(cos(dr*N/2)-1)+dx*sin(dr*N/2))/dr;
    afterN2s.translation.y=(dx*(1-cos(dr*N/2))+dy*sin(dr*N/2))/dr;
  }
  //cool heuristics with afterNs for start point...
  if (fabs(afterNs.rotation)>2.8)
  {
    currentUDParametersStartPosition=Vector2<double>(-1400,0);
    //currentUDParametersStartDirection=afterNs.rotation<0?80:-80;
    currentUDParametersStartDirection=afterNs.rotation<0?80:-80;
    if (fabs(motReq.rotation)>1.5)
    {
      currentUDParametersStartDirection=0;
    }
  }
  else if (fabs(afterNs.rotation)<1.0)
  {
    if (afterNs.translation.x>=fabs(afterNs.translation.y))
    {
      Pose2D start=Pose2D(0,-700,0).minusDiff(afterNs);
      currentUDParametersStartPosition=start.translation;
      currentUDParametersStartDirection=start.rotation*180/pi;
      //currentUDParametersStartDirection += (atan2(-start.translation.y,-start.translation.x)-start.rotation)/2.9*180/pi;
      if (currentUDParametersStartPosition.y>1350)
      {
        currentUDParametersStartPosition.y=1350;
        currentUDParametersStartDirection+=15;
      }
      else if (currentUDParametersStartPosition.y<-1350)
      {
        currentUDParametersStartPosition.y=-1350;
        currentUDParametersStartDirection-=15;
      }
    }
    else if (afterNs.translation.x<=-fabs(afterNs.translation.y))
    {
      currentUDParametersStartPosition=Vector2<double>(-700,0);
      currentUDParametersStartDirection=0;
      if (afterNs.translation.y>1350)
      {
        currentUDParametersStartDirection=20;
      }
      else if (afterNs.translation.y<-1350)
      {
        currentUDParametersStartDirection=20;
      }
    }
    else
    {
      Pose2D start=Pose2D(0,-1500,0).minusDiff(afterN2s);
      currentUDParametersStartPosition=start.translation;
      currentUDParametersStartDirection=start.rotation*180/pi;
    }
  }
  else
  {
    double rat=fabs(UDParametersSet::getRatio(motReq));
    if ((afterNs.translation.x>=fabs(afterNs.translation.y))&&(rat<0.2))
    {
      Pose2D start=Pose2D(afterNs.rotation/2.1,-900,0).minusDiff(afterNs);
      currentUDParametersStartPosition=start.translation;
      currentUDParametersStartDirection=start.rotation*180/pi;
    }
    else if ((afterNs.translation.x<=-fabs(afterNs.translation.y))&&(rat<0.2))
    {
      currentUDParametersStartPosition=Vector2<double>(-800,0);
      currentUDParametersStartDirection=-afterNs.rotation/2*180/pi;
    }
    else
    {
      Pose2D start=Pose2D(0,-1500,0).minusDiff(afterN2s);
      currentUDParametersStartPosition=start.translation;
      currentUDParametersStartDirection=start.rotation*180/pi;
    }
  }
}

bool EvolutionSymbols::getCurrentUDParametersMeasureBlind()
{
  return (udParametersCalibration[udCurrentIndex].translation.abs()==0)?
    fabs(measurementRequest.rotation)>0.55:
  fabs(measurementRequest.rotation+udParametersCalibration[udCurrentIndex].rotation)>1.1;
}

double EvolutionSymbols::getQualityOfCurrentUDParameters()
{
  if (udCurrentIndex>=0)
  {
    Pose2D fiction=measurementRequest;
    Pose2D reality=udParametersCalibration[udCurrentIndex];
    //2do: more usefull implementation
    return 20000.0/(200+(fiction-reality).translation.abs());
  }
  else
  {
    return 0;
  }
}

/*
* Change Log
* 
* $Log: EvolutionSymbols.cpp,v $
* Revision 1.1  2004/05/29 18:18:20  dueffert
* walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
*
* Revision 1.1  2004/05/27 13:31:27  dueffert
* walking evolution stuff separated
*
* Revision 1.22  2004/05/23 13:30:31  dueffert
* some start positions corrected
*
* Revision 1.21  2004/05/20 17:18:51  dueffert
* automatic measurement (hopefully) finalized
*
* Revision 1.20  2004/05/19 08:00:27  dueffert
* automatic walk speed measurement significantly improved
*
* Revision 1.19  2004/05/17 11:16:51  dueffert
* blind measurement improved
*
* Revision 1.18  2004/04/19 10:34:55  dueffert
* parameters tuned to allow measurement oft *fast* walking
*
* Revision 1.17  2004/04/16 14:56:37  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.16  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.15  2004/03/19 09:21:23  dueffert
* ability to measure freely choosen motion request added
*
* Revision 1.14  2004/03/15 12:37:54  dueffert
* start position improved
*
* Revision 1.13  2004/03/10 15:00:00  dueffert
* copy'n'past bug fixed; starting position improved
*
* Revision 1.12  2004/03/10 10:00:52  dueffert
* copy'n'paste bug fixed; start position tweaking
*
* Revision 1.11  2004/03/09 08:45:16  dueffert
* second measurement behavior added; calculation of tsart position improved
*
* Revision 1.10  2004/03/08 01:06:59  roefer
* Interfaces should be const
*
* Revision 1.9  2004/03/03 08:37:51  dueffert
* measurement start position is now calculated instead of predefined; file beautified
*
* Revision 1.8  2004/03/01 15:00:47  dueffert
* small start position improvements
*
* Revision 1.7  2004/02/29 13:42:58  dueffert
* UDParametersSet symmetries handled
*
* Revision 1.6  2004/02/27 16:43:21  dueffert
* small improvements
*
* Revision 1.5  2004/02/23 16:48:11  dueffert
* several improvements for measurement of walking
*
* Revision 1.4  2004/02/18 14:49:20  dueffert
* behavior control can now change walking parameters
*
* Revision 1.3  2004/02/10 11:13:18  dueffert
* symbol for evolution added
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/06 13:39:29  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.11  2003/07/07 21:15:39  risler
* search-ball.x added
* search-for-ball works without wlan
*
* Revision 1.10  2003/07/07 15:05:57  risler
* no wlan fallback :
* 5 seconds after wlan failure
* striker1 goes slightly into own half as well
*
* Revision 1.9  2003/07/06 21:58:25  risler
* role change : orientation influence reduced
*
* Revision 1.8  2003/07/05 21:27:19  risler
* bugfix: no wlan fallback, time for out of role striker increased
*
* Revision 1.7  2003/07/05 10:15:58  risler
* no wlan fallback for role change improved
*
* Revision 1.6  2003/07/04 16:30:34  loetzsch
* bonus for the current Striker in computeRole()
*
* Revision 1.5  2003/07/03 16:08:17  loetzsch
* renamed "robot-is-not-stuck-but-obstacles-are-close" to "obstacles-are-close"
*
* Revision 1.4  2003/07/03 11:04:42  juengel
* obstaclesAreClose is calculated
*
* Revision 1.3  2003/07/03 10:15:22  loetzsch
* added symbol "not-stuck-but..."
*
* Revision 1.2  2003/07/02 19:15:44  loetzsch
* ::computeRole()  does not use BehaviorTeamMessage::whoIsOnline() anymore
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.22  2003/06/26 16:20:29  loetzsch
* some role assignment experiments
*
* Revision 1.21  2003/06/23 19:13:21  loetzsch
* shifted the position of  "the-striker-is-playing-near-the-opponent-goal" for 20cm to the opponent goal
*
* Revision 1.20  2003/06/22 11:48:54  juengel
* new robot is stuck condition: > 20 percent of range is closer than 300mm
*
* Revision 1.19  2003/06/21 15:26:46  juengel
* obstaclesModel.getPercentageOfLowDistanceObstaclesInRange is used to calculate robot-is-stuck
*
* Revision 1.18  2003/06/20 20:15:24  juengel
* Renamed some methods in ObstaclesModel.
*
* Revision 1.17  2003/06/20 13:58:31  juengel
* Added calculation of robot is stuck.
*
* Revision 1.16  2003/06/19 21:31:45  juengel
* Added robot-is-stuck.
*
* Revision 1.15  2003/06/19 10:17:17  risler
* symbol another-teammate-is-preparing-a-kick added
*
* Revision 1.14  2003/06/17 19:54:59  risler
* ball caught symbols moved to strategy
*
* Revision 1.13  2003/06/02 14:10:20  loetzsch
* added some hysteresises for the supporter position options
*
* Revision 1.12  2003/05/31 14:38:48  loetzsch
* changed the symbols for the intercept
*
* Revision 1.11  2003/05/30 11:14:49  dueffert
* temporary bugfix
*
* Revision 1.10  2003/05/28 17:48:35  loetzsch
* better positioning for the supporters
*
* Revision 1.9  2003/05/27 16:52:53  loetzsch
* first passing experiments
*
* Revision 1.8  2003/05/26 13:45:02  loetzsch
* some improvements
*
* Revision 1.7  2003/05/25 22:37:02  loetzsch
* finished the game state options of GT2003
*
* Revision 1.6  2003/05/14 09:24:49  risler
* current role bonus hyteresis no longer in estimateTimeToReachBall
* added current role bonus for offensive supporter
*
* Revision 1.5  2003/05/08 14:25:30  risler
* some bugfixes
*
* Revision 1.4  2003/05/08 13:20:37  loetzsch
* some cleanup
*
* Revision 1.3  2003/05/08 10:20:35  dueffert
* bugs fixed
*
* Revision 1.2  2003/05/08 00:22:46  risler
* added strategy symbols
*
* Revision 1.1  2003/05/06 16:28:19  loetzsch
* added class StrategySymbols
*
*/

