/**
* @file GT2004BehaviorControl.h
* 
* Implementation of class GT2004BehaviorControl.
*
* @author Martin Ltzsch
* @author Matthias Jngel
*/

#ifndef __GT2004BehaviorControl_h_
#define __GT2004BehaviorControl_h_

#include "../Xabsl2BehaviorControl.h"

#include "GT2004BasicBehaviors/GT2004SimpleBasicBehaviors.h"
#include "GT2004BasicBehaviors/GT2004EvolutionBasicBehaviors.h"
#include "GT2004BasicBehaviors/GT2004ContinuousBasicBehaviors.h"
#include "GT2004BasicBehaviors/GT2004PotentialFieldBasicBehaviors.h"
#include "../CommonXabsl2BasicBehaviors/CommonBasicBehaviors.h"

#include "../CommonXabsl2Symbols/CalibrationSymbols.h"
#include "../CommonXabsl2Symbols/AngleSymbols.h"
#include "../CommonXabsl2Symbols/BallSymbols.h"
#include "../CommonXabsl2Symbols/HeadAndTailSymbols.h"
#include "../CommonXabsl2Symbols/JoystickSymbols.h"
#include "../CommonXabsl2Symbols/LEDAndSoundSymbols.h"
#include "../CommonXabsl2Symbols/KickSelectionSymbols.h"
#include "../CommonXabsl2Symbols/MathFunctions.h"
#include "../CommonXabsl2Symbols/MotionRequestSymbols.h"
#include "../CommonXabsl2Symbols/ObstaclesSymbols.h"
#include "../CommonXabsl2Symbols/RoboCupGameManagerSymbols.h"
#include "../CommonXabsl2Symbols/RobotPoseSymbols.h"
#include "../CommonXabsl2Symbols/RobotStateSymbols.h"
#include "../CommonXabsl2Symbols/SpecialVisionSymbols.h"
#include "../CommonXabsl2Symbols/ChallengeSymbols.h"
#include "GT2004Symbols/GT2004StrategySymbols.h"
#include "GT2004Symbols/GT2004ConfigurationSymbols.h"
#include "GT2004Symbols/EvolutionSymbols.h"
#include "GT2004Symbols/OpenChallengeSymbols.h"

#include "../KickLogger.h"

/**
* @class GT2004BehaviorControl
*
* A Behavior based on the Xabsl2Engine that is used by GermanTeam
* for the RoboCup 2004.
*
* @author Martin Ltzsch
* @author Matthias Jngel
*/ 
class GT2004BehaviorControl : public Xabsl2BehaviorControl
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BehaviorControl(BehaviorControlInterfaces& interfaces);

  /** destructor */
  ~GT2004BehaviorControl();

  /** Registers symbols and basic behaviors at the engine */
  virtual void registerSymbolsAndBasicBehaviors();
  
  /** Sets some varibles for leds and sound before the execution of the engine */
  void preExecute();

  /** Executes the module */
  virtual void execute();

  /** Sets some variables for head motion and sound after the execution of the engine */
  void postExecute();

  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the messag was read
  */
  virtual bool handleMessage(InMessage& message);

private:
  //!@name Symbols
  //!@{
  CalibrationSymbols calibrationSymbols;
  AngleSymbols angleSymbols;
  BallSymbols ballSymbols;
  GT2004ConfigurationSymbols configurationSymbols;
  HeadAndTailSymbols headAndTailSymbols;
  JoystickSymbols joystickSymbols;
  LEDAndSoundSymbols ledAndSoundSymbols;
  KickSelectionSymbols kickSelectionSymbols;
  MathFunctions mathFunctions;
  MotionRequestSymbols motionRequestSymbols;
  ObstaclesSymbols obstaclesSymbols;
  RoboCupGameManagerSymbols roboCupGameManagerSymbols;
  RobotPoseSymbols robotPoseSymbols;
  RobotStateSymbols robotStateSymbols;
  SpecialVisionSymbols specialVisionSymbols;
  GT2004StrategySymbols strategySymbols;
  EvolutionSymbols evolutionSymbols;
  ChallengeSymbols challengeSymbols;
  OpenChallengeSymbols openchallengeSymbols;
  //!@}

  //!@name Basic Behaviors
  //!@{
  GT2004SimpleBasicBehaviors simpleBasicBehaviors;
  GT2004EvolutionBasicBehaviors evolutionBasicBehaviors;
  GT2004ContinuousBasicBehaviors continuousBasicBehaviors;
  GT2004PotentialFieldBasicBehaviors potentialFieldBasicBehaviors;
  CommonBasicBehaviors commonBasicBehaviors;
  //!@}

  //!@name UD Evolution stuff
  //!@{
  UDParametersSet udParametersSet;
  UDParameters udExtraParameters;
  Pose2D udParametersCalibration[128];
  Pose2D measurementRequest;
  int evolutionMode;
  int udCurrentIndex;
  //!@}

  /*
  * Runs in parallel to the behavior control and logs the preconditions
  * and results of each kick.
  */
  KickLogger kickLogger;

  /** If quickStop is true, the robot just stands */
  bool quickStop;

  /** Indicates which HeadControlMode is set, when the robot is demobilized (quick stop)*/
  HeadControlMode debugHeadControlMode;
};

#endif// __GT2004BehaviorControl_h_

/* * Change log :
 * 
 * $Log: GT2004BehaviorControl.h,v $
 * Revision 1.11  2004/06/22 18:48:56  juengel
 * kickAngles clean up
 *
 * Revision 1.10  2004/06/16 14:43:51  risler
 * added obstacles symbols
 * added symbol obstacles.opponent-close-to-ball
 *
 * Revision 1.9  2004/06/08 18:36:12  juengel
 * Added  "debugHeadControlMode".
 *
 * Revision 1.8  2004/06/02 10:56:25  juengel
 * quickStop added
 *
 * Revision 1.7  2004/05/29 18:18:19  dueffert
 * walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
 *
 * Revision 1.6  2004/05/27 16:48:08  loetzsch
 * added the kick logger
 *
 * Revision 1.5  2004/05/25 12:58:17  tim
 * added missing potential field files and example behavior
 *
 * Revision 1.4  2004/05/24 16:35:20  spranger
 * removed doProfiling
 *
 * Revision 1.3  2004/05/22 21:33:33  loetzsch
 * adopted kick selection table to GT2004
 *
 * Revision 1.2  2004/05/22 21:31:04  loetzsch
 * adopted kick selection table to GT2004
 *
 * Revision 1.1.1.1  2004/05/22 17:17:59  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/05/08 16:18:13  hamerla
 * Open Challenge
 *
 * Revision 1.2  2004/05/04 10:48:58  loetzsch
 * replaced all enums
 * xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
 * by
 * behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
 * (this mechanism was neither fully implemented nor used)
 *
 * Revision 1.1  2004/05/02 13:26:42  juengel
 * Added GT2004BehaviorControl.
 *
 */
