/**
* @file GT2004BehaviorControl.cpp
* 
* Implementation of class GT2004BehaviorControl.
*
* @author Martin Ltzsch
* @author Matthias Jngel
*/

#include "GT2004BehaviorControl.h"
#include "Tools/Player.h"

GT2004BehaviorControl::GT2004BehaviorControl(BehaviorControlInterfaces& interfaces)
: Xabsl2BehaviorControl(interfaces,SolutionRequest::gt2004),
calibrationSymbols(interfaces),
angleSymbols(interfaces),
ballSymbols(interfaces),
configurationSymbols(interfaces),
headAndTailSymbols(interfaces),
joystickSymbols(interfaces),
kickSelectionSymbols(interfaces,"GT2004.kst"),
ledAndSoundSymbols(interfaces),
mathFunctions(interfaces),
motionRequestSymbols(interfaces),
obstaclesSymbols(interfaces),
roboCupGameManagerSymbols(interfaces),
robotPoseSymbols(interfaces),
robotStateSymbols(interfaces),
specialVisionSymbols(interfaces),
strategySymbols(interfaces),
evolutionSymbols(interfaces, udParametersSet, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
openchallengeSymbols(interfaces),
challengeSymbols(interfaces),
simpleBasicBehaviors(interfaces,errorHandler),
evolutionBasicBehaviors(interfaces,errorHandler, udParametersSet, udExtraParameters, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
continuousBasicBehaviors(interfaces,errorHandler),
potentialFieldBasicBehaviors(interfaces,errorHandler),
commonBasicBehaviors(interfaces,errorHandler),
kickLogger(interfaces),
udCurrentIndex((int)UDParametersSet::no_turn_0_fast),
evolutionMode(1)

{
  Xabsl2FileInputSource file("Xabsl2/gt04-ic.dat");
  init(file);
  ASSERT(pEngine);

  measurementRequest = udParametersSet.getParameters(udCurrentIndex)->requestedMotion;
  for (int i=0;i<=127;i++)
  {
    udParametersCalibration[i]=Pose2D(0,0,0);
  }
  udExtraParameters.index=UDParametersSet::numberOfParameters;
  quickStop = false;
}

GT2004BehaviorControl::~GT2004BehaviorControl()
{
}

void GT2004BehaviorControl::registerSymbolsAndBasicBehaviors()
{
  simpleBasicBehaviors.registerBasicBehaviors(*pEngine);
  evolutionBasicBehaviors.registerBasicBehaviors(*pEngine);
  continuousBasicBehaviors.registerBasicBehaviors(*pEngine);
  potentialFieldBasicBehaviors.registerBasicBehaviors(*pEngine);
  commonBasicBehaviors.registerBasicBehaviors(*pEngine);
  calibrationSymbols.registerSymbols(*pEngine);
  angleSymbols.registerSymbols(*pEngine);
  ballSymbols.registerSymbols(*pEngine);
  configurationSymbols.registerSymbols(*pEngine);
  headAndTailSymbols.registerSymbols(*pEngine);
  joystickSymbols.registerSymbols(*pEngine);
  ledAndSoundSymbols.registerSymbols(*pEngine);
  kickSelectionSymbols.registerSymbols(*pEngine);
  mathFunctions.registerSymbols(*pEngine);
  motionRequestSymbols.registerSymbols(*pEngine);
  obstaclesSymbols.registerSymbols(*pEngine);
  roboCupGameManagerSymbols.registerSymbols(*pEngine);
  robotPoseSymbols.registerSymbols(*pEngine);
  robotStateSymbols.registerSymbols(*pEngine);
  specialVisionSymbols.registerSymbols(*pEngine);
  strategySymbols.registerSymbols(*pEngine);
  evolutionSymbols.registerSymbols(*pEngine);
  openchallengeSymbols.registerSymbols(*pEngine);
  challengeSymbols.registerSymbols(*pEngine);
}

void GT2004BehaviorControl::preExecute()
{
  for(int i = 0; i < 14; i++) ledRequest.faceLED[i] = LEDRequest::oooo;
  ledRequest.backFrontWhiteLED = LEDRequest::oooo;
  ledRequest.backMiddleWhiteLED = LEDRequest::oooo;
  ledRequest.backRearWhiteLED = LEDRequest::oooo;
  ledRequest.backMiddleOrangeLED = LEDRequest::oooo;

  ledRequest.headWhiteLED = LEDRequest::oooo;
  ledRequest.headOrangeLED = LEDRequest::oooo;

  // set the outgoing behavior team message to none
  outgoingBehaviorTeamMessage.message = BehaviorTeamMessage::none;
  
  // Set the tail request depending on whether the ball was seen
  if (ballModel.ballWasSeen)
  {
    // ball seen
    ledRequest.headWhiteLED = LEDRequest::llll;
    motionRequest.tailRequest.tailRequestID = TailRequest::noTailWag;
  }
  else if (SystemCall::getTimeSince(ballModel.seen.timeWhenLastSeen) 
    > BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
    && SystemCall::getTimeSince(ballModel.communicated.timeWhenLastObserved) < 3000)
  {
    // ball known
    motionRequest.tailRequest.tailRequestID = TailRequest::wagHorizontalFast;
  }
  else
  {
    // ball not known
    motionRequest.tailRequest.tailRequestID = TailRequest::noTailWag;
  }
  
  // set head control mode lookToStars to detect missing settings
  headControlMode.headControlMode = HeadControlMode::lookToStars;

  soundRequest.soundID = SoundRequest::none;

  angleSymbols.angleShownByLEDs = AngleSymbols::undefined;
}

void GT2004BehaviorControl::execute()
{
  angleSymbols.update();
  ballSymbols.update();
  configurationSymbols.update();
  obstaclesSymbols.update();
  roboCupGameManagerSymbols.update();
  robotPoseSymbols.update();
  strategySymbols.update();
  evolutionSymbols.update();
  openchallengeSymbols.update();
  
  continuousBasicBehaviors.update();
  potentialFieldBasicBehaviors.update();
 
  // set some leds and and other variables
  preExecute();

  // execute the engine
  executeEngine();

  // play some sounds if necessary
  postExecute();

  // execute the kick logger, but not during a game
  kickLogger.execute();
}

void GT2004BehaviorControl::postExecute()
{
  // handle fall down
  if(outgoingBehaviorTeamMessage.gameState != BehaviorTeamMessage::penalized &&
     ((robotState.getState() == RobotState::crashed)||
     (robotState.getState() == RobotState::rollLeft)||
     (robotState.getState() == RobotState::rollRight)))
  {
    if ((motionInfo.executedMotionRequest.motionType == MotionRequest::walk &&
         motionInfo.executedMotionRequest.walkRequest.walkType != WalkRequest::upsideDown)||
        (motionInfo.executedMotionRequest.motionType == MotionRequest::stand))
    {
      motionRequest.motionType = MotionRequest::getup;
    }
  }

  // display team color
  if (getPlayer().getTeamColor() == Player::blue)
  {
    ledRequest.backFrontBlueLED = LEDRequest::llll;
    if(
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::sleep ||
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::initial ||
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::ready ||
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::set)
    {
      ledRequest.backFrontWhiteLED = LEDRequest::oooo;
    }
    ledRequest.backRearRedLED = LEDRequest::oooo;
  } 
  else if (getPlayer().getTeamColor() == Player::red) 
  {
    ledRequest.backFrontBlueLED = LEDRequest::oooo;
    ledRequest.backRearRedLED = LEDRequest::llll;
    if(
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::sleep ||
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::initial ||
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::ready ||
      outgoingBehaviorTeamMessage.gameState == BehaviorTeamMessage::set)
    {
      ledRequest.backRearWhiteLED = LEDRequest::oooo;
    }
  } 
  else 
  {
    ledRequest.backFrontBlueLED = LEDRequest::lloo;
    ledRequest.backRearRedLED = LEDRequest::lloo;
  }

  if(
    outgoingBehaviorTeamMessage.gameState != BehaviorTeamMessage::sleep &&
    outgoingBehaviorTeamMessage.gameState != BehaviorTeamMessage::initial)
  {
    // set the purple face leds depending on the dynamic role.
    switch (strategySymbols.role)
    {
    case BehaviorTeamMessage::striker:
      ledRequest.faceLED[13] = LEDRequest::llll;
      break;
    case BehaviorTeamMessage::offensiveSupporter:
      ledRequest.faceLED[13] = LEDRequest::oooo;
      ledRequest.faceLED[12] = LEDRequest::lloo;
      break;
    case BehaviorTeamMessage::defensiveSupporter:
      ledRequest.faceLED[13] = LEDRequest::oooo;
      break;
    default:
      ledRequest.faceLED[13] = LEDRequest::oooo;
    }

    if(getPlayer().getPlayerNumber() == Player::one)
    {
      if (ballModel.ballState.ballRollsByLeft)
        ledRequest.faceLED[1] = LEDRequest::llll;	
      if (ballModel.ballState.ballRollsByRight)
        ledRequest.faceLED[0] = LEDRequest::llll;	
      if (ballModel.ballState.ballRollsFast)
      {
        ledRequest.faceLED[2] = LEDRequest::llll;	
        ledRequest.faceLED[3] = LEDRequest::llll;	
      }
      if (ballModel.ballState.ballRollsTowardsRobot)
      {
        ledRequest.faceLED[1] = LEDRequest::llll;	
        ledRequest.faceLED[0] = LEDRequest::llll;	
      }
    }

    if(angleSymbols.angleShownByLEDs == AngleSymbols::undefined)
    {
      ledRequest.faceLED[10] = LEDRequest::lolo;
    }
    else
    {
      // visualize the angle the robot wants to play to with the top white leds.
      double bestAngle;
      bestAngle = toDegrees(normalize(angleSymbols.getAngle(angleSymbols.angleShownByLEDs)));

      if     (bestAngle >  75) ledRequest.faceLED[ 7] = LEDRequest::lolo; //?_____   
      else if(bestAngle >  45) ledRequest.faceLED[ 7] = LEDRequest::llll; // x____
      else if(bestAngle >  15) ledRequest.faceLED[ 9] = LEDRequest::llll; // _x___      
      else if(bestAngle > -15) ledRequest.faceLED[10] = LEDRequest::llll; // __x__
      else if(bestAngle > -45) ledRequest.faceLED[ 8] = LEDRequest::llll; // ___x_
      else if(bestAngle > -75) ledRequest.faceLED[ 6] = LEDRequest::llll; // ____x
      else                     ledRequest.faceLED[ 6] = LEDRequest::lolo; // _____?
    }

    // display stuck/obstacles
    if (obstaclesSymbols.robotIsStuck)
    {
      ledRequest.faceLED[4] = LEDRequest::llll;
      ledRequest.faceLED[5] = LEDRequest::llll;
    }
    else if (obstaclesSymbols.obstaclesAreClose)
    {
      ledRequest.faceLED[11] = LEDRequest::llll;
    }
  }

  // if the head control mode was not set, play a warning sound
  if (headControlMode.headControlMode == HeadControlMode::lookToStars)
  {
    soundRequest.soundID = SoundRequest::rob101;
  }

  if(sensorDataBuffer.frame[0].data[SensorData::chin] == 1)
    soundRequest.soundID = SoundRequest::bark2;


  if(sensorDataBuffer.lastFrame().data[SensorData::chin] == 1)
  {
    if(
      robotState.getButtonPressed(BodyPercept::backBack) &&
      robotState.getButtonDuration(BodyPercept::backBack) > 1000 &&
      profiler.profilerCollectMode == GTXabsl2Profiler::collectProfiles
      )
    {
      profiler.profilerCollectMode = GTXabsl2Profiler::dontCollectProfiles;
      profiler.profilerWriteMode = GTXabsl2Profiler::writeCompleteProfiles;
      soundRequest.soundID = SoundRequest::bing01;
    }

    if(
      robotState.getButtonPressed(BodyPercept::backFront) &&
      robotState.getButtonDuration(BodyPercept::backFront) > 1000
      )
    {
      profiler.profilerCollectMode = GTXabsl2Profiler::collectProfiles;
      soundRequest.soundID = SoundRequest::bing03;
    }
  }

  if(!quickStop)
  {
    if(
      robotState.getButtonPressed(BodyPercept::chin) &&
      robotState.getButtonDuration(BodyPercept::chin) > 500 &&
      
      robotState.getButtonPressed(BodyPercept::backMiddle) &&
      robotState.getButtonDuration(BodyPercept::backMiddle) > 500
      )
      quickStop = true;
  }
  else
  {
    if(
      robotState.getButtonPressed(BodyPercept::head) &&
      robotState.getButtonDuration(BodyPercept::head) > 100 &&
      robotState.getButtonDuration(BodyPercept::head) < 700
      )
      quickStop = false;
  }

  if(quickStop)
  {
    if(robotState.getButtonPressed(BodyPercept::backFront)) debugHeadControlMode.headControlMode = HeadControlMode::none;
    if(robotState.getButtonPressed(BodyPercept::backMiddle)) debugHeadControlMode.headControlMode = HeadControlMode::searchForBall;
    if(robotState.getButtonPressed(BodyPercept::backBack)) debugHeadControlMode.headControlMode = HeadControlMode::stayAsForced;

    if(debugHeadControlMode.headControlMode != HeadControlMode::none)
    {
      headControlMode = debugHeadControlMode;
    }

    motionRequest.motionType = MotionRequest::stand;
    ledRequest.backFrontWhiteLED = LEDRequest::oooo;
    ledRequest.backMiddleWhiteLED = LEDRequest::oooo;
    ledRequest.backRearWhiteLED = LEDRequest::oooo;

    ledRequest.backFrontBlueLED = LEDRequest::llll;
    ledRequest.backMiddleOrangeLED = LEDRequest::llll;
    ledRequest.backRearRedLED = LEDRequest::llll;

    if(debugHeadControlMode.headControlMode == HeadControlMode::none) ledRequest.backFrontWhiteLED = LEDRequest::llll;
    if(debugHeadControlMode.headControlMode == HeadControlMode::searchForBall) ledRequest.backMiddleWhiteLED = LEDRequest::llll;
    if(debugHeadControlMode.headControlMode == HeadControlMode::stayAsForced) ledRequest.backRearWhiteLED = LEDRequest::llll;
  }
}

bool GT2004BehaviorControl::handleMessage(InMessage& message)
{
  switch (message.getMessageID())
  {
  case idUDEvolutionRequest:
    {
      int mode;
      message.bin >> mode;
      switch (mode)
      {
      case 1: // measure UDParameters #... or its mirror if that has higher index or a self choosen motionRequest
        {
          message.bin >> udCurrentIndex;
          message.bin >> evolutionMode;
          if (udCurrentIndex==(int)UDParametersSet::numberOfParameters)
          {
            message.bin >> measurementRequest;
            if (UDParametersSet::getSpeed(measurementRequest)<0.05)
            {
              UDParametersSet::setSpeed(measurementRequest,0.05);
            }
            udExtraParameters.requestedMotion=measurementRequest;
            OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'measure (" << measurementRequest.translation.x << ", " << measurementRequest.translation.y << ", " << measurementRequest.rotation << ")' received");
          }
          else
          {
            int mirror=UDParametersSet::getIndexOfMirror(udCurrentIndex);
            //use higher mirror number = lturn
            if ((mirror>=0)&&(mirror>udCurrentIndex))
            {
              udCurrentIndex=mirror;
            }
            measurementRequest=udParametersSet.getParameters(udCurrentIndex)->requestedMotion;
            OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'measure " << UDParametersSet::getIndexString(udCurrentIndex) << "' received (" << measurementRequest.translation.x << ", " << measurementRequest.translation.y << ", " << measurementRequest.rotation << ")");
          }
          break;
        }
      case 2: // save parametersSet
        udParametersSet.save();
        OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'save parametersSet' executed");
        break;
      case 3: // load parametersSet
        {
          udParametersSet.load();
          for (int i=0;i<(int)UDParametersSet::numberOfParameters;i++)
          {
            udParametersCalibration[i]=Pose2D(0,0,0);
          }
          OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'load parametersSet' executed and calibration resetted");
          break;
        }
      default:
        OUTPUT(idText,text,"GT2003BehavContr: unknown UDEvolutionRequest received");
        break;
      }
      return true;
    }
  case idUDParameters:
    {
      UDParameters param;
      message.bin >> param;
      if (param.index<UDParametersSet::numberOfParameters)
      {
        *udParametersSet.getParameters(param.index) = param;
        OUTPUT(idText,text,"GT2003BehavContr: UDParameters " << UDParametersSet::getIndexString(param.index) << " received (" << param.requestedMotion.translation.x << ", " << param.requestedMotion.translation.y << ", " << param.requestedMotion.rotation << ")");
      }
      else
      {
        udExtraParameters = param;
        OUTPUT(idText,text,"GT2003BehavContr: extra UDParameters received (" << param.requestedMotion.translation.x << ", " << param.requestedMotion.translation.y << ", " << param.requestedMotion.rotation << ")");
      }
      udParametersCalibration[param.index]=Pose2D(0,0,0);
      //and send it to Motion immediately:
      udParameters = param;
      walkParameterTimeStamp = SystemCall::getCurrentSystemTime();
      return true;
    }
  default:
    
    return 
      (Xabsl2BehaviorControl::handleMessage(message) ||
      kickSelectionSymbols.handleMessage(message)
      );
  }
}


/*
* Change log :
* 
* $Log: GT2004BehaviorControl.cpp,v $
* Revision 1.27  2004/07/01 13:06:43  risler
* check executedMotionRequest instead of MotionRequest for getup
*
* Revision 1.26  2004/06/29 16:28:39  risler
* removed sleep state
* display robot number in initial
*
* Revision 1.25  2004/06/24 11:25:45  juengel
* Changed LEDs.
*
* Revision 1.24  2004/06/23 19:31:42  juengel
* Angle "undefined" is shown with a blinking middle led.
*
* Revision 1.23  2004/06/22 18:48:56  juengel
* kickAngles clean up
*
* Revision 1.22  2004/06/20 12:02:05  juengel
* Increased time needed to release a quick-stop.
*
* Revision 1.21  2004/06/20 10:40:46  juengel
* Quick stop + back button -> hcm stay as forced.
*
* Revision 1.20  2004/06/17 11:13:03  risler
* removed annoying debug sound
*
* Revision 1.19  2004/06/16 14:43:51  risler
* added obstacles symbols
* added symbol obstacles.opponent-close-to-ball
*
* Revision 1.18  2004/06/15 16:18:21  risler
* dont disable leds quickstop etc while getup
*
* Revision 1.17  2004/06/14 21:40:15  risler
* improved led settings
*
* Revision 1.16  2004/06/14 17:53:25  risler
* gamestate led display unified
*
* Revision 1.15  2004/06/09 08:19:48  dueffert
* missing initializations added
*
* Revision 1.14  2004/06/08 18:36:12  juengel
* Added  "debugHeadControlMode".
*
* Revision 1.13  2004/06/02 17:18:23  spranger
* MotionRequest cleanup
*
* Revision 1.12  2004/06/02 10:56:25  juengel
* quickStop added
*
* Revision 1.11  2004/05/29 18:18:19  dueffert
* walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
*
* Revision 1.10  2004/05/27 18:49:17  kerdels
* added a small 5 frames sliding average for the relative ballspeed,
* added new ballState Representation and adjusted the apropriate files
*
* Revision 1.9  2004/05/27 16:48:08  loetzsch
* added the kick logger
*
* Revision 1.8  2004/05/27 15:48:09  juengel
* Added profiler hot keys.
*
* Revision 1.7  2004/05/27 14:36:49  loetzsch
* readded message handling for kick selection table
*
* Revision 1.6  2004/05/25 12:58:17  tim
* added missing potential field files and example behavior
*
* Revision 1.5  2004/05/24 16:35:20  spranger
* removed doProfiling
*
* Revision 1.4  2004/05/22 21:48:42  loetzsch
* renamed ATHERS7.kst to GT2004.kst
*
* Revision 1.3  2004/05/22 21:31:04  loetzsch
* adopted kick selection table to GT2004
*
* Revision 1.2  2004/05/22 20:45:17  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:17:59  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.3  2004/05/08 16:18:13  hamerla
* Open Challenge
*
* Revision 1.2  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.1  2004/05/02 13:26:42  juengel
* Added GT2004BehaviorControl.
*
*/

