/** 
* @file GT2004SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __GT2004SimpleBasicBehaviors_h_
#define __GT2004SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"
#include "Tools/Math/PIDsmoothedValue.h"



/**
* A Basic Behavior for going to the ball
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Jan Hoffmann
*/
class GT2004BasicBehaviorGoToBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorGoToBall(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball.distance",distanceAtEnd);
    registerParameter("go-to-ball.max-speed", maxSpeed);
    registerParameter("go-to-ball.max-speed.y", maxSpeedY);
    registerParameter("go-to-ball.max-turn-speed", maxTurnSpeed);
    registerParameter("go-to-ball.target-angle-to-ball", targetAngleToBall);
    registerParameter("go-to-ball.walk-type", walkType);
    registerParameter("go-to-ball.y-offset", yOffset);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "go-to-ball.max-speed.y" */
  double maxSpeedY;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  /** parameter "go-to-ball.target-angle-to-ball" */
  double targetAngleToBall;
  
  /** parameter "go-to-ball.y-offset" */
  double yOffset;
  
  /** parameter "walk-type" */
  double walkType;
  
  /** parameter "max-turn-speed" */
  double maxTurnSpeed;
};

/**
* A Basic Behavior for clearing the ball
* @author Michael Spranger
* @author Matthias Jngel
*/
class GT2004BasicBehaviorGoToBallWithoutTurning: public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorGoToBallWithoutTurning(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-without-turning", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball-without-turning.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "go-to-ball.max-speed.y" */
  double maxSpeedY;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  /** parameter "go-to-ball.target-angle-to-ball" */
  double targetAngleToBall;
  
  /** parameter "go-to-ball.y-offset" */
  double yOffset;
  
  /** parameter "walk-type" */
  double walkType;
  
  /** parameter "max-turn-speed" */
  double maxTurnSpeed;
};


/**
* A Basic Behavior for going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class GT2004BasicBehaviorGoToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorGoToPoint(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point.x", x);
    registerParameter("go-to-point.y", y);
    registerParameter("go-to-point.destination-angle", destinationAngle);
    registerParameter("go-to-point.distance-remain", distanceRemain);
    registerParameter("go-to-point.angle-remain", angleRemain);
    registerParameter("go-to-point.max-speed", maxSpeed);
    registerParameter("go-to-point.max-speed.y", maxSpeedY);
    registerParameter("go-to-point.walk-type", walkType);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point.x" */
  double x;
  
  /** parameter "go-to-point.y" */
  double y;
  
  /** parameter "go-to-point.destination-angle" */
  double destinationAngle;

  /** parameter "distance-remain" */
  double distanceRemain;

  /** parameter "angle-remain" */
  double angleRemain;

  /** parameter "go-to-point.max-speed" */
  double maxSpeed;
  
  /** parameter "go-to-point.max-speed.y" */
  double maxSpeedY;
  
  /** parameter "walk-type" */
  double walkType;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for positioning the goalie inside the own goal
* @author Martin Ltzsch
*/
class GT2004BasicBehaviorGoaliePosition : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorGoaliePosition(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("goalie-position", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("goalie-position.x", x);
    registerParameter("goalie-position.y", y);
    registerParameter("goalie-position.destination-angle", destinationAngle);
    registerParameter("goalie-position.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "goalie-position.x" */
  double x;
  
  /** parameter "goalie-position.y" */
  double y;
  
  /** parameter "goalie-position.destination-angle" */
  double destinationAngle;
  
  /** parameter "goalie-position.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going turning around a point on the field
* @author Jan Hoffmann
*/
class GT2004BasicBehaviorTurnAroundPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorTurnAroundPoint(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("turn-around-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("turn-around-point.x", x);
    registerParameter("turn-around-point.y", y);
    registerParameter("turn-around-point.radius", radius);
    registerParameter("turn-around-point.left-right", leftRight);
    registerParameter("turn-around-point.forward-component", forwardComponent);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "turn-around-point.x" */
  double x;
  
  /** parameter "turn-around-point.y" */
  double y;
  
  /** parameter "turn-around-point.radius" */
  double radius;
  
  /** parameter "go-to-point.left-right" */
  double leftRight;
  
  /** parameter "go-to-point.forward-component" */
  double forwardComponent;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/** A goalie positioning approach from Benjamin */
class GT2004ERS7BasicBehaviorGoaliePositionBenjamin : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004ERS7BasicBehaviorGoaliePositionBenjamin(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("goalie-position-benjamin", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
	registerParameter("goalie-position-benjamin.max-speed", maxSpeed);
	registerParameter("goalie-position-benjamin.min-x-trans", minXTrans);
	registerParameter("goalie-position-benjamin.min-y-trans", minYTrans);
	registerParameter("goalie-position-benjamin.weight-pose", weightPose);
	registerParameter("goalie-position-benjamin.weight-odo", weightOdo);
	registerParameter("goalie-position-benjamin.cut-y", cutY);
	registerParameter("goalie-position-benjamin.guard-direct-to-goal", guardDirectToGoal);
	registerParameter("goalie-position-benjamin.guard-line", guardLine);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  
	double maxSpeed;
	double minXTrans;
	double minYTrans;
	double weightPose;
	double weightOdo;
	double cutY;
	double guardDirectToGoal;
	double guardLine;
	
	Pose2D lastRobotPose;
	Pose2D lastOdometry;
	Pose2D goaliePose;
	Pose2D lastGoaliePose;
	
  
};

/**
* A Basic Behavior for going to a point on the field facing forward
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class GT2004BasicBehaviorGoForwardToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorGoForwardToPoint(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-forward-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-forward-to-point.x", x);
    registerParameter("go-forward-to-point.y", y);
    registerParameter("go-forward-to-point.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-forward-to-point.x" */
  double x;
  
  /** parameter "go-forward-to-point.y" */
  double y;
  
  /** parameter "go-forward-to-point.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author Jan Hoffmann
*/
class GT2004BasicBehaviorGoToPointAndAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorGoToPointAndAvoidObstacles(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-and-avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces),
    turnSpeed(0, .5, 0, 0, -pi, pi, pi/2),
    speedX(0, .4, 0, 0, -80, 350, 100),
    speedY(0, .4, 0, 0, -100, 100, 100)
  {
    registerParameter("go-to-point-and-avoid-obstacles.x", x);
    registerParameter("go-to-point-and-avoid-obstacles.y", y);
    registerParameter("go-to-point-and-avoid-obstacles.max-speed", maxSpeed);
    registerParameter("go-to-point-and-avoid-obstacles.avoidance-level", avoidanceLevel);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-point-and-avoid-obstacles.x" */
  double x;
  
  /** parameter "go-to-point-and-avoid-obstacles.y" */
  double y;
  
  /** parameter "go-to-point-and-avoid-obstacles.max-speed" */
  double maxSpeed;
  
  /** parameter "go-to-point-and-avoid-obstacles.avoid-penalty-areas" */
  double avoidanceLevel;
  
  /** smoothes the rotation */  
  PIDsmoothedValue turnSpeed;
  
  /** smoothes the speed in forward direction */  
  PIDsmoothedValue speedX;
  
  /** smoothes the speed in sideways direction */  
  PIDsmoothedValue speedY;
  
  /** last time behavior was executed */
  long lastTimeExecuted;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author 
*/
class GT2004BasicBehaviorMoveTheBridge : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorMoveTheBridge(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("move-the-bridge", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("move-the-bridge.type", type);
    registerParameter("move-the-bridge.x", x);
    registerParameter("move-the-bridge.y", y);
    registerParameter("move-the-bridge.rotation", rotation);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "move-the-bridge.x" */
  double x;
  
  /** parameter "move-the-bridge.y" */
  double y;
  
  /** parameter "move-the-bridge.type" */
  double type;
  
  /** parameter "move-the-bridge.rotation" */
  double rotation;
  
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author 
*/
class GT2004BasicBehaviorDogAsJoysick : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorDogAsJoysick(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("dog-as-joystick", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
};


/**
* A Basic Behavior for intercepting the ball on a specific x-height.
* @author Jochen Kerdels
*/
class GT2004BasicBehaviorInterceptAtX : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorInterceptAtX(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("intercept-at-x", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("intercept-at-x.x", x);
    registerParameter("intercept-at-x.max-speed", maxSpeed);
    registerParameter("intercept-at-x.test-thres", testThres);
    registerParameter("intercept-at-x.test-thres2", testThres2);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "intercept-at-x.x" */
  double x;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "intercept-at-x.test-thres" */
  double testThres;
  
  /** parameter "intercept-at-x.test-thres2" */
  double testThres2;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
};


/**
* The Basic Behavior "intercept-test-batchman".
* @author Mathias Hlsbusch
*/
class GT2004BasicBehaviorInterceptTestBatchman : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorInterceptTestBatchman(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("intercept-test-batchman", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
};

/**
* @class SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class GT2004SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
/**
* Constructor
  */
  GT2004SimpleBasicBehaviors(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
    errorHandler(errorHandler),
    basicBehaviorGoToBallWoTurning(interfaces,errorHandler),
    basicBehaviorGoToBall(interfaces,errorHandler),
    basicBehaviorGoToPoint(interfaces,errorHandler),
    basicBehaviorGoForwardToPoint(interfaces,errorHandler),
    basicBehaviorTurnAroundPoint(interfaces,errorHandler),
    basicBehaviorGoToPointAndAvoidObstacles(interfaces,errorHandler),
    basicBehaviorMoveTheBridge(interfaces,errorHandler),
    basicBehaviorInterceptAtX(interfaces,errorHandler),
    basicBehaviorDogAsJoysick(interfaces,errorHandler),
    basicBehaviorGoaliePositionBenjamin(interfaces,errorHandler),
    basicBehaviorGoaliePosition(interfaces,errorHandler),
    basicBehaviorInterceptTestBatchman(interfaces,errorHandler)
  {}
  
  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine)
  {
    engine.registerBasicBehavior(basicBehaviorGoToBallWoTurning);
    engine.registerBasicBehavior(basicBehaviorGoToBall);
    engine.registerBasicBehavior(basicBehaviorGoToPoint);
    engine.registerBasicBehavior(basicBehaviorGoForwardToPoint);
    engine.registerBasicBehavior(basicBehaviorGoToPointAndAvoidObstacles);
    engine.registerBasicBehavior(basicBehaviorTurnAroundPoint);
    engine.registerBasicBehavior(basicBehaviorMoveTheBridge);
    engine.registerBasicBehavior(basicBehaviorInterceptAtX);
    engine.registerBasicBehavior(basicBehaviorDogAsJoysick);
    engine.registerBasicBehavior(basicBehaviorInterceptTestBatchman);
    engine.registerBasicBehavior(basicBehaviorGoaliePositionBenjamin);
    engine.registerBasicBehavior(basicBehaviorGoaliePosition);
  }
  
private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;
  
  //!@name Basic Behaviors
  //!@{
  GT2004BasicBehaviorGoToBallWithoutTurning
    basicBehaviorGoToBallWoTurning;
  GT2004BasicBehaviorGoToBall  basicBehaviorGoToBall;
  GT2004BasicBehaviorGoToPoint  basicBehaviorGoToPoint;  
  GT2004BasicBehaviorGoForwardToPoint  basicBehaviorGoForwardToPoint;  
  GT2004BasicBehaviorTurnAroundPoint basicBehaviorTurnAroundPoint;  
  GT2004BasicBehaviorGoToPointAndAvoidObstacles  basicBehaviorGoToPointAndAvoidObstacles;
  GT2004BasicBehaviorMoveTheBridge basicBehaviorMoveTheBridge;
  GT2004BasicBehaviorDogAsJoysick basicBehaviorDogAsJoysick;
  GT2004BasicBehaviorInterceptAtX basicBehaviorInterceptAtX;
  GT2004ERS7BasicBehaviorGoaliePositionBenjamin  basicBehaviorGoaliePositionBenjamin;
  GT2004BasicBehaviorGoaliePosition  basicBehaviorGoaliePosition;
  GT2004BasicBehaviorInterceptTestBatchman basicBehaviorInterceptTestBatchman;
  //!@}
};

#endif // __GT2004SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: GT2004SimpleBasicBehaviors.h,v $
* Revision 1.17  2004/06/30 10:42:28  spranger
* renamed goalie-clear-ball to tun-without-ball
*
* Revision 1.16  2004/06/27 16:36:01  altmeyer
* improved GoaliePositionBenjamin
*
* Revision 1.15  2004/06/27 14:59:23  goehring
* goToPoint has parameter distanceRemain and angleRemain
*
* Revision 1.14  2004/06/27 14:53:51  altmeyer
* new GoaliePositionBenjamin and new debug key send_goaliePositionField
*
* Revision 1.13  2004/06/23 20:59:04  juengel
* Added basicBehaviorGoalieClearBall.
*
* Revision 1.12  2004/06/23 16:19:28  loetzsch
* reactivated and improved old ath goalie
*
* Revision 1.11  2004/06/18 10:56:33  altmeyer
* improved goalie-position
*
* Revision 1.10  2004/06/18 00:32:16  risler
* added walk-type to go-to-point
* added max-speed.y to go-to-ball and go-to-point
*
* Revision 1.9  2004/06/17 15:30:04  altmeyer
* new GoaliePosition (Parameters, Motion-Gltting ;-)
*
* Revision 1.8  2004/06/16 17:35:04  altmeyer
* added GoaliePosition (author: Jan Hoffmann)
*
* Revision 1.7  2004/06/15 16:15:40  risler
* added go-to-ball parameters max-turn-speed and walk-type
*
* Revision 1.6  2004/05/27 18:49:17  kerdels
* added a small 5 frames sliding average for the relative ballspeed,
* added new ballState Representation and adjusted the apropriate files
*
* Revision 1.5  2004/05/26 14:36:44  juengel
* Added y-offset to go-to-ball
*
* Revision 1.4  2004/05/25 14:19:19  kerdels
* adjusted intercept-at-x
*
* Revision 1.3  2004/05/24 16:58:25  kerdels
* intercept at x added
*
* Revision 1.2  2004/05/24 16:31:25  loetzsch
* added basic behavior intercept-batch-man
*
* Revision 1.1.1.1  2004/05/22 17:18:01  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/05/13 22:11:39  hamerla
* OpenChallenge use a dog as joystick for the bridgemove
*
* Revision 1.3  2004/05/08 16:18:13  hamerla
* Open Challenge
*
* Revision 1.2  2004/05/03 14:33:20  loetzsch
* some cleanup
*
* Revision 1.1  2004/05/02 13:24:39  juengel
* Added GT2004BehaviorControl.
*
*/

