/** 
* @file GT2004PotentialFieldBasicBehaviors.h
*
* Declaration of basic behaviors defined in potential-field-basic-behaviors.xml.
*
* @author <a href="mailto:timlaue@tzi.de">Tim Laue</a>
*/

#ifndef __GT2004PotentialFieldBasicBehaviors_h_
#define __GT2004PotentialFieldBasicBehaviors_h_

#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/PotentialFields/GTStandardConverter.h"
#include "Modules/BehaviorControl/BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"


/**
* An abstract base class for all potential field based basic behaviors
*
* @author Tim Laue
*/
class GT2004PotentialFieldBasicBehavior : public Xabsl2BasicBehavior, 
                                          public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004PotentialFieldBasicBehavior(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler, const char* behaviorName, const char* pfcFileName):
            Xabsl2BasicBehavior(behaviorName, errorHandler),
            BehaviorControlInterfaces(interfaces)
  {
    potentialfields.load(pfcFileName);
  };
    
  /**  Executes the basic behavior. */
  virtual void execute() = 0;

protected:
  /** An instance of the potential field architecture*/
  GTStandardConverter potentialfields;
  /** A standard instance of a result structure*/
  PotentialfieldResult result;
};


class GT2004PotentialFieldBasicBehaviorGoToPose : public GT2004PotentialFieldBasicBehavior
{
public:
  /*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004PotentialFieldBasicBehaviorGoToPose(const BehaviorControlInterfaces& interfaces,
                                            Xabsl2ErrorHandler& errorHandler);
    
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter potential-field-go-to-pose.x*/
  double x;
  /** Parameter potential-field-go-to-pose.y*/
  double y;
  /** Parameter potential-field-go-to-pose.destination-angle*/
  double destinationAngle;
  /** Parameter potential-field-go-to-pose.max-speed*/
  double maxSpeed;
};


class GT2004PotentialFieldBasicBehaviorSupport : public GT2004PotentialFieldBasicBehavior
{
public:
  /*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004PotentialFieldBasicBehaviorSupport(const BehaviorControlInterfaces& interfaces,
                                           Xabsl2ErrorHandler& errorHandler);
    
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter potential-field-support.x*/
  double x;
  /** Parameter potential-field-support.y*/
  double y;
  /** Parameter potential-field-support.max-speed*/
  double maxSpeed;
};


class GT2004PotentialFieldBasicBehaviorOffensiveSupport : public GT2004PotentialFieldBasicBehavior
{
public:
  /*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004PotentialFieldBasicBehaviorOffensiveSupport(const BehaviorControlInterfaces& interfaces,
                                                    Xabsl2ErrorHandler& errorHandler);
    
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter potential-field-offensive-support.x*/
  double x;
  /** Parameter potential-field-offensive-support.y*/
  double y;
  /** Parameter potential-field-offensive-support.max-speed*/
  double maxSpeed;
};


/**
* @class PotentialFieldBasicBehaviors
*
* Creates and registers potential field basic behaviors
*/
class GT2004PotentialFieldBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  GT2004PotentialFieldBasicBehaviors(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler):
  BehaviorControlInterfaces(interfaces),
  errorHandler(errorHandler),
  potentialFieldGoToPose(interfaces, errorHandler),
  potentialFieldSupport(interfaces, errorHandler),
  potentialFieldOffensiveSupport(interfaces, errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

  /** updates some variables */
  void update();

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  GT2004PotentialFieldBasicBehaviorGoToPose potentialFieldGoToPose;
  GT2004PotentialFieldBasicBehaviorSupport potentialFieldSupport;
  GT2004PotentialFieldBasicBehaviorOffensiveSupport potentialFieldOffensiveSupport;
  //!@}
};

#endif // __GT2004PotentialFieldBasicBehaviors_h_


/*
* $Log: GT2004PotentialFieldBasicBehaviors.h,v $
* Revision 1.3  2004/06/17 16:18:38  tim
* added potential field support behaviors
*
* Revision 1.2  2004/06/16 15:09:58  tim
* added potential field support behavior
*
* Revision 1.1  2004/05/25 12:59:39  tim
* added potential-field-go-to-pose
*
*/
