/**
* @file GT2004EvolutionBasicBehaviors.h
*
* Declaration of basic behaviors defined in evolution-basic-behaviors.xml.
*
* @author Uwe Dffert
*/

#ifndef __GT2004EvolutionBasicBehaviors_h_
#define __GT2004EvolutionBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Evolution/Population.h"
#include "Tools/Evolution/Parcour.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Modules/WalkingEngine/UDParameterSet.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"

/**
* A BasicBehavior for measuring and calibrating an the current UDParameters
* @author Uwe Dffert
*/
class GT2004BasicBehaviorMeasureUDParameters : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorMeasureUDParameters(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler,
    UDParametersSet& udParametersSet,
    UDParameters& udExtraParameters,
    Pose2D* udParametersCalibration,
    int& udCurrentIndex,
    Pose2D& measurementRequest,
    int& evolutionMode):
  Xabsl2BasicBehavior("measure-ud-parameters", errorHandler),
    BehaviorControlInterfaces(interfaces),
    udParametersSet(udParametersSet),
    udExtraParameters(udExtraParameters),
    udParametersCalibration(udParametersCalibration),
    udCurrentIndex(udCurrentIndex),
    measurementRequest(measurementRequest),
    evolutionMode(evolutionMode),
    startFrameNumber(0),lastFrameNumber(0)
  {
  }
  /** Executes the basic behavior. */
  virtual void execute();
private:
  /** robotPose frame number of first call of this beasic behavior in current measurement */
  unsigned long startFrameNumber;
  /** robotPose frame number of last call of this beasic behavior in current measurement */
  unsigned long lastFrameNumber;
  /** local variables to store measurement start position */
  double x0,y0,r0,t0;

  /** UD Evolution stuff */
  UDParametersSet& udParametersSet;
  UDParameters& udExtraParameters;
  Pose2D* udParametersCalibration;
  int& udCurrentIndex;
  Pose2D& measurementRequest;
  int& evolutionMode;
};

/**
* A BasicBehavior for measuring and calibrating an the current UDParameters while being partially blind (while turning)
* @author Uwe Dffert
*/
class GT2004BasicBehaviorMeasureUDParametersBlind : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorMeasureUDParametersBlind(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler,
    UDParametersSet& udParametersSet,
    UDParameters& udExtraParameters,
    Pose2D* udParametersCalibration,
    int& udCurrentIndex,
    Pose2D& measurementRequest,
    int& evolutionMode):
  Xabsl2BasicBehavior("measure-ud-parameters-blind", errorHandler),
    BehaviorControlInterfaces(interfaces),
    udParametersSet(udParametersSet),
    udExtraParameters(udExtraParameters),
    udParametersCalibration(udParametersCalibration),
    udCurrentIndex(udCurrentIndex),
    measurementRequest(measurementRequest),
    evolutionMode(evolutionMode),
    startFrameNumber(0),lastFrameNumber(0),clusterFrames(0),goodFrames(0)
  {
  }
  /** Executes the basic behavior. */
  virtual void execute();
private:
  /** robotPose frame number of first call of this beasic behavior in current measurement */
  unsigned long startFrameNumber;
  /** robotPose frame number of last call of this beasic behavior in current measurement */
  unsigned long lastFrameNumber;
  /** number of valid frames in current cluster*/
  unsigned long clusterFrames;
  /** local variables to store measurement start position */
  double x0,y0,r0,t0;
  double lastRot;
  double lastTime;
  double firstNullTime;
  double lastNullTime;
  double nullTime;
  Pose2D speed;
  int clusterNumber;
  int goodFrames;
  double dr,dx,dy;

  /** UD Evolution stuff */
  UDParametersSet& udParametersSet;
  UDParameters& udExtraParameters;
  Pose2D* udParametersCalibration;
  int& udCurrentIndex;
  Pose2D& measurementRequest;
  int& evolutionMode;
};

/**
* A BasicBehavior for walking forward or backward following a certain parcour as well as
* changing walking parameters according to the fitness measured in the last parcour walk
* This Basic Behavior needs a walking engine that does not switch parameters according
* to the MotionRequest (like InvKin:GT2003 and UD), but a non-switching one (like InvKin:FF)
* @author Uwe Dffert
*/
class GT2004BasicBehaviorEvolveOmniParameters : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorEvolveOmniParameters(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler,
    UDParametersSet& udParametersSet,
    Pose2D* udParametersCalibration,
    int& udCurrentIndex,
    Pose2D& measurementRequest,
    int& evolutionMode):
  Xabsl2BasicBehavior("evolve-walking-parameters", errorHandler),
    BehaviorControlInterfaces(interfaces),
    udParametersSet(udParametersSet),
    udParametersCalibration(udParametersCalibration),
    udCurrentIndex(udCurrentIndex),
    measurementRequest(measurementRequest),
    evolutionMode(evolutionMode),
    lastMode(0),
    motionRequestX(0.15, 0.25, 0.0, 0, -300, 300, 50),
    motionRequestY(0.15, 0.25, 0.0, 0, -300, 300, 50),
    motionRequestR(0.15, 0.25, 0.0, 0, -pi, pi, 0.5)
  {
    registerParameter("execution-mode", mode);
    //initialize Population
    delete(ftPopulation.individual[0]);
    delete(bPopulation.individual[0]);
    ERS7EvolveWalkingParameters ers7;

    InvKinWalkingParameters msh7backward("", 0, 88.5057, 79.127, 43.2139, 101.905, 77.6874, -49.6692, 9.76973, 28.5799, +0.0724927, -0.0922574, 1.20455, 0.843727, 1.44822, 34.8207, 38.8411, 100, 100, 1, 0.06, 61, 0.461105, 0.0361205, 0.0391534, 0.508829, 0.0532571, 0.0469326, 0, 0.5, 0.5, 0, 0, 0, 0.0337565);
    UDParameters msh7(127, Pose2D(0, 0, 0), 77.5836, 80.2702, 50.3536, 105.174, 81.5432, -52.0932, 5, 24, -0.25, 0.016, 64, 0, 0.49, 0.06, 0.06, 0.49, 0.06, 0.06, 0, 0.5, 0.5, 0);

    UDParameters unsw(127, Pose2D(0, 0, 0), Pose2D(0, 34, 44), 76.9, 83.1, 55.9, 108.6, 81.9, -56.1, 5, 24, -0.06, 0.05, 80, 0, 0.5, 0.06, 0.06, 0.5, 0.06, 0.06, 0, 0.5, 0.5, 0);

    ftPopulation.individual[0]= new UDParameters(msh7); //ers7
    bPopulation.individual[0]= new UDParameters(msh7backward); //ers7

    for (int i=1;i<10;i++)
    {
      ftPopulation.individual[i]->mutationOf(ftPopulation.individual[0], 0.3, 0.12, true);
      ftPopulation.individual[i]->index=ftPopulation.individual[0]->index;
      bPopulation.individual[i]->mutationOf(bPopulation.individual[0], 0.3, 0.12, true);
      bPopulation.individual[i]->index=bPopulation.individual[0]->index;
    }
    InTextFile inFTPopul("ft_pop.dat");
    if (inFTPopul.exists())
    {
      inFTPopul >> ftPopulation;
    }
    InTextFile inBPopul("b_pop.dat");
    if (inBPopul.exists())
    {
      inBPopul >> bPopulation;
    }
  }

  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** a Polpulation of WalkingParameters */
  Population<UDParameters,10> ftPopulation;
  Population<UDParameters,10> bPopulation;

  /** specifies whether the robot should walk forward or backward or parameters should be changed */
  double mode;
  int lastMode;

  ForwardTurningParcour forwardTurningParcour;
  SimpleBackwardParcour simpleBackwardParcour;
  PIDsmoothedValue motionRequestX, motionRequestY, motionRequestR;

  /** UD Evolution stuff */
  UDParametersSet& udParametersSet;
  Pose2D* udParametersCalibration;
  int& udCurrentIndex;
  Pose2D& measurementRequest;
  int& evolutionMode;
};


/**
* A BasicBehavior for sending the current UDParameters via PackageCognitionMotion to the
* walking engine and choose the next Parameters to be optimized
*
* @author Uwe Dffert
*/
class GT2004BasicBehaviorSendCurrentUDParametersAndChooseNext : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorSendCurrentUDParametersAndChooseNext(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler,
    UDParametersSet& udParametersSet,
    Pose2D* udParametersCalibration,
    int& udCurrentIndex,
    Pose2D& measurementRequest,
    int& evolutionMode):
  Xabsl2BasicBehavior("send-current-ud-parameters-and-choose-next", errorHandler),
    BehaviorControlInterfaces(interfaces),
    udParametersSet(udParametersSet),
    udParametersCalibration(udParametersCalibration),
    udCurrentIndex(udCurrentIndex),
    measurementRequest(measurementRequest),
    evolutionMode(evolutionMode)
  {
  }

  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** UD Evolution stuff */
  UDParametersSet& udParametersSet;
  Pose2D* udParametersCalibration;
  int& udCurrentIndex;
  Pose2D& measurementRequest;
  int& evolutionMode;
};


/**
* A BasicBehavior for generating the next UDParameters to be measured exactly
*
* @author Uwe Dffert
*/
class GT2004BasicBehaviorNextUDParametersToBeMeasured : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2004BasicBehaviorNextUDParametersToBeMeasured(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler,
    UDParametersSet& udParametersSet,
    UDParameters& udExtraParameters,
    Pose2D* udParametersCalibration,
    int& udCurrentIndex,
    Pose2D& measurementRequest,
    int& evolutionMode):
  Xabsl2BasicBehavior("next-ud-parameters-to-measure", errorHandler),
    BehaviorControlInterfaces(interfaces),
    udParametersSet(udParametersSet),
    udExtraParameters(udExtraParameters),
    udParametersCalibration(udParametersCalibration),
    udCurrentIndex(udCurrentIndex),
    measurementRequest(measurementRequest),
    evolutionMode(evolutionMode),tableIndex(0),trials(0),maxSpeed(0),
    maxSpeedRatio(0),maxYRatio(0),maxRRatio(0)
  {
  }

  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** UD Evolution stuff */
  UDParametersSet& udParametersSet;
  UDParameters& udExtraParameters;
  Pose2D* udParametersCalibration;
  int& udCurrentIndex;
  Pose2D& measurementRequest;
  int& evolutionMode;

  Pose2D requestTable[40];
  Pose2D measureTable[40];
  int tableIndex;
  int trials;
  double maxSpeed;
  double maxSpeedRatio;
  double maxYRatio;
  double maxRRatio;
};

/**
* @class GT2004EvolutionBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class GT2004EvolutionBasicBehaviors : public BehaviorControlInterfaces
{
public:
/**
* Constructor
  */
  GT2004EvolutionBasicBehaviors(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler,
    UDParametersSet& udParametersSet,
    UDParameters& udExtraParameters,
    Pose2D* udParametersCalibration,
    int& udCurrentIndex,
    Pose2D& measurementRequest,
    int& evolutionMode)
    : BehaviorControlInterfaces(interfaces),
    errorHandler(errorHandler),
    basicBehaviorEvolveOmniParameters(interfaces,errorHandler, udParametersSet, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
    basicBehaviorMeasureUDParameters(interfaces,errorHandler, udParametersSet, udExtraParameters, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
    basicBehaviorMeasureUDParametersBlind(interfaces,errorHandler, udParametersSet, udExtraParameters, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
    basicBehaviorSendCurrentUDParametersAndChooseNext(interfaces,errorHandler, udParametersSet, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
    basicBehaviorNextUDParametersToBeMeasured(interfaces,errorHandler, udParametersSet, udExtraParameters, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  GT2004BasicBehaviorEvolveOmniParameters  basicBehaviorEvolveOmniParameters;
  GT2004BasicBehaviorMeasureUDParameters  basicBehaviorMeasureUDParameters;
  GT2004BasicBehaviorMeasureUDParametersBlind  basicBehaviorMeasureUDParametersBlind;
  GT2004BasicBehaviorSendCurrentUDParametersAndChooseNext  basicBehaviorSendCurrentUDParametersAndChooseNext;
  GT2004BasicBehaviorNextUDParametersToBeMeasured  basicBehaviorNextUDParametersToBeMeasured;
  //!@}
};

#endif // __GT2004EvolutionBasicBehaviors_h_

/*
* Change Log
*
* $Log: GT2004EvolutionBasicBehaviors.h,v $
* Revision 1.3  2004/07/07 15:13:00  dueffert
* evolution uses two populations now
*
* Revision 1.2  2004/06/14 16:54:56  juengel
* Removed some WalkingEngineParameterSets.
*
* Revision 1.1  2004/05/29 18:18:19  dueffert
* walk parameter evolution, measurement and calibration stuff ported to GT2004_WM
*
* Revision 1.1  2004/05/27 13:31:26  dueffert
* walking evolution stuff separated
*
* Revision 1.23  2004/05/20 17:18:35  dueffert
* automatic measurement (hopefully) finalized
*
* Revision 1.22  2004/05/19 07:59:44  dueffert
* automatic walk speed measurement significantly improved
*
* Revision 1.21  2004/05/14 10:15:49  dueffert
* exact measurement behavior with logging implemented
*
* Revision 1.20  2004/05/12 14:23:32  dueffert
* measurement basic behavior added
*
* Revision 1.19  2004/04/29 15:15:01  dueffert
* measurement is now frame instead of time based
*
* Revision 1.18  2004/04/16 14:56:36  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.17  2004/03/24 13:44:49  dueffert
* support for uniform noise mutation readded
*
* Revision 1.16  2004/03/19 16:37:27  dueffert
* blind measurement works for the first time
*
* Revision 1.15  2004/03/12 17:11:38  dueffert
* omni-evo uses parameters in package now; blind speed calculation improved
*
* Revision 1.14  2004/03/10 15:01:08  dueffert
* first simple fast turn (blind) measurement implemented
*
* Revision 1.13  2004/03/10 10:02:22  dueffert
* population size tweaked; statistics output added
*
* Revision 1.12  2004/03/09 10:28:10  dueffert
* now using const interface too
*
* Revision 1.11  2004/03/09 08:46:30  dueffert
* second measurement behavior added
*
* Revision 1.10  2004/03/08 01:06:49  roefer
* Interfaces should be const
*
* Revision 1.9  2004/03/01 15:06:27  dueffert
* measurement output beautified
*
* Revision 1.8  2004/02/23 16:48:11  dueffert
* several improvements for measurement of walking
*
* Revision 1.7  2004/02/18 14:49:20  dueffert
* behavior control can now change walking parameters
*
* Revision 1.6  2004/02/10 11:19:19  dueffert
* simple basic behavior for evolution added, another improved, beautified
*
* Revision 1.5  2003/12/09 15:15:20  dueffert
* WalkAccelerationRestrictor replaced by PIDsmoothedValue for evolving
*
* Revision 1.4  2003/11/19 13:49:39  dueffert
* better positioning
*
* Revision 1.3  2003/10/31 08:32:49  dueffert
* doxygen bugs fixed
*
* Revision 1.2  2003/10/28 13:27:22  dueffert
* spelling and evolution logic improved
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.11  2003/09/30 10:51:13  dueffert
* typos fixed
*
* Revision 1.10  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.9  2003/08/09 14:57:06  dueffert
* some evolution improvements
*
* Revision 1.8  2003/08/08 15:43:25  dueffert
* some evolution implementation added
*
* Revision 1.7  2003/08/08 14:30:04  dueffert
* some evolution implementation added
*
* Revision 1.6  2003/08/04 07:46:39  roefer
* Challenge 2 modifications
*
* Revision 1.5  2003/07/30 14:52:03  dueffert
* walk evolution agent added
*
* Revision 1.4  2003/07/09 11:44:26  jhoffman
* obstacle avoidance simple behavior mode as used in challenge 3
* known bug: sometimes stops in front of obstacle (but eventually the situation is resolved)
*
* Revision 1.3  2003/07/06 20:34:30  dueffert
* GT2004BasicBehaviorGoToBallWithDirection added
*
* Revision 1.2  2003/07/04 16:24:24  jhoffman
* updated obstacle avoider (simple behavior) to be used in challenge
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/06/04 00:43:45  loetzsch
* put the GT2003WalkAccelerationRestrictor into two extra files in /Src/Tools
*
* Revision 1.2  2003/05/05 19:22:03  risler
* added classes Simple/ContinuousBasicBehaviors for registering basic behaviors
*
* Revision 1.1  2003/05/03 15:14:02  loetzsch
* GT2003BehaviorControl first draft
*
*
*/
