/**
* @file GT2004ContinuousRules/TurnToPoint.h
*
* @author Max Risler
*/

#ifndef __GT2004TurnToPoint_h_
#define __GT2004TurnToPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace GT2004ContinuousRules
{

/**@class TurnToPoint
 */
class TurnToPoint: public ContinuousRule{
private:
  /** reference to the point to turn to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;
  
public:

  /** constructor
   */
  TurnToPoint(
    BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false)
    :
    ContinuousRule(interfaces,"TurnToPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative)
  {};

  /** constructor
   */
  TurnToPoint(
    BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false)
    :
    ContinuousRule(interfaces, "TurnToPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    walk.x = walk.y = 0;

    Vector2<double> destination(pointX, pointY);

    if (relative)
    {
      Vector2<double> destination(pointX, pointY);
      double distance = destination.abs();
      double factor = (distance - 100) / 500;
      if (factor > 1) factor = 1;
      if (factor < 0) factor = 0;
      factor *= 3;
      factor += 1;
      //factor = 1;
      ra = factor*destination.angle();
      if (ra > 3) ra = 3;
      else if (ra < -3) ra = -3;
    }
    else
    {
      Vector2<double> destination(pointX, pointY);
      double distance = Geometry::distanceTo(robotPose, destination);
      double factor = (distance - 100) / 500;
      if (factor > 1) factor = 1;
      if (factor < 0) factor = 0;
      factor *= 3;
      factor += 1;
      //factor = 1;
      ra = factor*destination.angle();
      if (ra > 3) ra = 3;
      else if (ra < -3) ra = -3;
    }

    rweight=1.0;
  }
};

}


#endif //__GT2004TurnToBall_h_


/*
* Change log:
*
* $Log: TurnToPoint.h,v $
* Revision 1.1.1.1  2004/05/22 17:18:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/02 13:19:03  juengel
* Added GT2004BehaviorControl.
*
*/
