/**
* @file GT2004ContinuousRules/GoToPoint.h
*
* @author Dirk Thomas
*/

#ifndef __GT2004GoToPoint_h_
#define __GT2004GoToPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace GT2004ContinuousRules
{

/**@class GoToPoint
 */
class GoToPoint: public ContinuousRule{
private:
  /** reference to the point to go to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;

  /** if the rule not constructed via a reference to the point
   * these variables are used to store the coordinates */
  double localPointX;
  double localPointY;

	/** the distance above which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance below which the result vector is 0 in mm*/
  double minDistance;
  /** the maximum attracting force */
  double maxForce;


public:

  /** constructor
   */
  GoToPoint(
    BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces,"GoToPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {};

  /** constructor
   */
  GoToPoint(
    BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces, "GoToPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {};

  /** constructor
   */
  GoToPoint(
    BehaviorControlInterfaces& interfaces,
    const Vector2<int> point,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces,"GoToPoint"),
    pointX(localPointX),
    pointY(localPointY),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {
      localPointX = (double)point.x;
      localPointY = (double)point.y;
  };

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    walk = Vector2<double>(pointX, pointY);

    if (!relative) 
      walk -= robotPose.translation;

    double distance = walk.abs();
    walk.normalize();

    walk *= getAttractingForce(distance, maxForce, minDistance, maxForceLength);

    rweight=0;
  }
};

}


#endif //__GoToPoint_h_


/*
* Change log:
*
* $Log: GoToPoint.h,v $
* Revision 1.1.1.1  2004/05/22 17:18:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/02 13:19:03  juengel
* Added GT2004BehaviorControl.
*
*/
