/** 
* @file GT2004ContinuousRules/AvoidTeammates.h
*
* 
* @author Max Risler
*/

#ifndef __GT2004AvoidTeammates_h_
#define __GT2004AvoidTeammates_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace GT2004ContinuousRules
{

/**@class AvoidTeammates
 * rule which avoids collision with the teammates
 */
class AvoidTeammates: public ContinuousRule{
public:

  /** constructor
   */
  AvoidTeammates(BehaviorControlInterfaces& interfaces,
                 double maxForce = 3.0, 
                 double maxForceLength = 150.0, 
                 double influenceLength = 250.0)
      :ContinuousRule(interfaces, "AvoidTeammates"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength)
  {};
 
	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    Vector2<double> fromTeammate;
    double distance;

    walk.x=walk.y=0;

    Geometry::Line robot(robotPose.translation,
                         robotPose*Vector2<double>(-bodyLength,0)-robotPose.translation);

    for (int i=0;i<playerPoseCollection.numberOfOwnPlayers;i++)
    {
      distance = Geometry::getDistanceToEdge(robot,playerPoseCollection.getOwnPlayerPose(i).getPose().translation);

      fromTeammate = robotPose.translation - playerPoseCollection.getOwnPlayerPose(i).getPose().translation;
      fromTeammate.normalize();

      walk += fromTeammate * 
              getRepellingForce(distance, maxForce, maxForceLength, influenceLength) * 
              playerPoseCollection.getOwnPlayerPose(i).getValidity();
    }

    ra=0;
    rweight=0;
  };

protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is max_force (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;
};

}

#endif 


/*
* Change log:
*
* $Log: AvoidTeammates.h,v $
* Revision 1.1.1.1  2004/05/22 17:18:01  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/02 13:19:03  juengel
* Added GT2004BehaviorControl.
*
*/
