/** 
* @file GT2004ContinuousRules/AvoidOwnPenaltyArea.h
*
* 
* @author Martin Kallnik
*/

#ifndef __GT2004AvoidOwnPenaltyArea_h_
#define __GT2004AvoidOwnPenaltyArea_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace GT2004ContinuousRules
{

/**@class AvoidOwnPenaltyArea
 * rule which avoids the own penalty area
 */
class AvoidOwnPenaltyArea: public ContinuousRule{
private:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;

public:

  /** constructor
   */
  AvoidOwnPenaltyArea(BehaviorControlInterfaces& interfaces,
                 double maxForce = 3.0, 
                 double maxForceLength = 100.0, 
                 double influenceLength = 300.0)
      :ContinuousRule(interfaces, "AvoidOwnPenaltyArea"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength)
  {};
 
 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    if (getPlayer().getPlayerNumber() != Player::one)
    {
    
      double distance = min (
        FieldDimensions::distanceToOwnPenaltyArea ( robotPose.translation ),
        FieldDimensions::distanceToOwnPenaltyArea ( robotPose*Vector2<double>(-bodyLength,0) ) );
    
      walk = robotPose.translation;
      walk.x -= xPosOwnGroundline;
      walk.normalize();
    
      walk *= getRepellingForce(distance, maxForce, maxForceLength, influenceLength);
    } else {
      // no avoid own penalty area for goalie
      walk.x = walk.y = 0;
    }
    
    ra=0;
    rweight=0;
  };
};

}


#endif 


/*
* Change log:
*
* $Log: AvoidOwnPenaltyArea.h,v $
* Revision 1.1.1.1  2004/05/22 17:18:01  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/05/02 13:19:03  juengel
* Added GT2004BehaviorControl.
*
*/
