/** 
* @file GT2004ContinuousBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in continuous-basic-behaviors.xml.
*
* @author Martin Kallnik
* @author Max Risler
*/

#include "GT2004ContinuousRules/AvoidBorder.h"
#include "GT2004ContinuousRules/AvoidObstacles.h"
#include "GT2004ContinuousRules/AvoidPoint.h"
#include "GT2004ContinuousRules/AvoidTeammates.h"
#include "GT2004ContinuousRules/AvoidOwnPenaltyArea.h"
#include "GT2004ContinuousRules/GoToPoint.h"
#include "GT2004ContinuousRules/TurnToAngle.h"
#include "GT2004ContinuousRules/TurnToPoint.h"

#include "GT2004ContinuousBasicBehaviors.h"

using namespace GT2004ContinuousRules;

GT2004ContinuousBasicBehaviors::GT2004ContinuousBasicBehaviors(
                       BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler) : 
  BehaviorControlInterfaces(interfaces),
  errorHandler(errorHandler),
  turn(interfaces,errorHandler,"cont-turn"),
  goalieReturn(interfaces, errorHandler,"cont-goalie-return"),
  supporterPosition(interfaces,errorHandler,"cont-supporter-position"),
  offensiveSupporterPosition(interfaces, errorHandler, "cont-offensive-supporter-position")
{

  turn.addRule(new TurnToAngle(interfaces,turnRotationSpeed,true));
  turn.addRule(new AvoidBorder(interfaces));
  turn.addRule(new AvoidOwnPenaltyArea(interfaces));
  turn.addRule(new AvoidTeammates(interfaces));
  turn.addRule(new AvoidObstacles(interfaces));

  turn.registerParameter("cont-turn.rotation-speed",turnRotationSpeed);

  supporterPosition.setMaxRotationSpeed(fromDegrees(120.0));
  // go to destination but no need to reach it exactly
  supporterPosition.addRule(new GoToPoint(interfaces,supporterPositionX,supporterPositionY,false,1.0,0.0,500.0));
  // turn to ball
  supporterPosition.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  // avoid running into opponent goal area
  supporterPosition.addRule(new AvoidPoint(interfaces,opponentGoalRearPanelMiddle,false,3.0,600.0,850.0));
  // avoid running over ball
  supporterPosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  supporterPosition.addRule(new AvoidBorder(interfaces));
  supporterPosition.addRule(new AvoidOwnPenaltyArea(interfaces));
  supporterPosition.addRule(new AvoidTeammates(interfaces));
  supporterPosition.addRule(new AvoidObstacles(interfaces));

  supporterPosition.registerParameter("cont-supporter-position.x",supporterPositionX);
  supporterPosition.registerParameter("cont-supporter-position.y",supporterPositionY);
  supporterPosition.registerParameter("cont-supporter-position.max-speed",supporterPositionMaxSpeed);

  offensiveSupporterPosition.setMaxRotationSpeed(fromDegrees(120.0));
  // go to destination but no need to reach it exactly
  offensiveSupporterPosition.addRule(new GoToPoint(interfaces,offensiveSupporterPositionX,offensiveSupporterPositionY,false,1.0,0.0,500.0));
  // turn to ball
  offensiveSupporterPosition.addRule(new TurnToPoint(interfaces,knownPointBetweenBallandGoal,true));
  // avoid running into opponent goal area
  offensiveSupporterPosition.addRule(new AvoidPoint(interfaces,opponentGoalRearPanelMiddle,false,3.0,600.0,850.0));
  // avoid running over ball
  offensiveSupporterPosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  offensiveSupporterPosition.addRule(new AvoidBorder(interfaces));
  offensiveSupporterPosition.addRule(new AvoidOwnPenaltyArea(interfaces));
  offensiveSupporterPosition.addRule(new AvoidTeammates(interfaces));
  offensiveSupporterPosition.addRule(new AvoidObstacles(interfaces));

  offensiveSupporterPosition.registerParameter("cont-offensive-supporter-position.x",offensiveSupporterPositionX);
  offensiveSupporterPosition.registerParameter("cont-offensive-supporter-position.y",offensiveSupporterPositionY);
  offensiveSupporterPosition.registerParameter("cont-offensive-supporter-position.max-speed",offensiveSupporterPositionMaxSpeed);

  goalieReturn.registerParameter("cont-goalie-return.x",goalieReturnX);
  goalieReturn.registerParameter("cont-goalie-return.y",goalieReturnY);
  goalieReturn.registerParameter("cont-goalie-return.max-speed",goalieReturnMaxSpeed);
  
  goalieReturn.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  goalieReturn.addRule(new GoToPoint(interfaces,goalieReturnX,goalieReturnY));
  goalieReturn.addRule (new AvoidBorder(interfaces));

}


void GT2004ContinuousBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(turn);
  engine.registerBasicBehavior(supporterPosition);
  engine.registerBasicBehavior(offensiveSupporterPosition);
  engine.registerBasicBehavior(goalieReturn);
}

void GT2004ContinuousBasicBehaviors::update()
{
 
  knownBallPos = ballModel.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);

  knownBallPosRobotRelative = Geometry::vectorTo(robotPose, knownBallPos);
  
    
  /* Point between ball and goal calculation*/
  Vector2<double>rP(robotPose.translation);
  double relAngleToBall = Vector2<double>(knownBallPos -rP).angle();
  double relAngleToGoal = Vector2<double>(Vector2<double>(xPosOpponentGoal,0)-rP).angle();
  if(fabs( relAngleToBall - relAngleToGoal) < pi/2)
  {
    double desAngle = ((relAngleToBall + relAngleToGoal) /2) - (fabs(relAngleToBall) < fabs(relAngleToGoal) ? relAngleToBall:relAngleToGoal);
    knownPointBetweenBallandGoal = Geometry::vectorTo(robotPose, Vector2<double>(rP.x+cos(desAngle),rP.y+sin(desAngle)));
  }

  else
  {
     knownPointBetweenBallandGoal = knownBallPosRobotRelative;
  }
  
  supporterPosition.setMaxSpeed(supporterPositionMaxSpeed);
  offensiveSupporterPosition.setMaxSpeed(offensiveSupporterPositionMaxSpeed);
  goalieReturn.setMaxSpeed(goalieReturnMaxSpeed);
  
  // set the evaluation mode
  // this is only done for localization dependent behaviors, others keep discrete mode
  turn.setEvaluationMode(ContinuousBasicBehavior::discrete);
  supporterPosition.setEvaluationMode(ContinuousBasicBehavior::discrete);
  offensiveSupporterPosition.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goalieReturn.setEvaluationMode(ContinuousBasicBehavior::discrete);
  
}

/*
* Change Log
* 
* $Log: GT2004ContinuousBasicBehaviors.cpp,v $
* Revision 1.3  2004/07/01 14:19:07  altmeyer
* changed  transistions, replaced go-to-poin-behavouir through count-goalie-return
*
* Revision 1.2  2004/05/22 20:45:17  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:18:00  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.1  2004/05/02 13:24:39  juengel
* Added GT2004BehaviorControl.
*
*
*/

