/** 
* @file RobotStateSymbols.cpp
*
* Implementation of class RobotStateSymbols.
*
* @author Matthias Jngel
*/

#include "RobotStateSymbols.h"

RobotStateSymbols::RobotStateSymbols(const BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces)
{
}


void RobotStateSymbols::registerSymbols(Xabsl2Engine& engine)
{
  // robot-state.buttons
  engine.registerBooleanInputSymbol("robot-state.any-back-button-pressed-for-one-second",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getAnyBackButtonPressedForOneSecond);

  engine.registerBooleanInputSymbol("robot-state.any-back-button-pressed-short-and-released",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getAnyBackButtonPressedShortAndReleased);

  engine.registerBooleanInputSymbol("robot-state.fore-back-button-pressed-short-and-released",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getForeBackButtonPressedShortAndReleased);

  engine.registerBooleanInputSymbol("robot-state.middle-back-button-pressed-short-and-released",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getMiddleBackButtonPressedShortAndReleased);

  engine.registerBooleanInputSymbol("robot-state.hind-back-button-pressed-short-and-released",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getHindBackButtonPressedShortAndReleased);

  engine.registerBooleanInputSymbol("robot-state.fore-back-button-pressed-short",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getForeBackButtonPressedShort);

  engine.registerBooleanInputSymbol("robot-state.middle-back-button-pressed-short",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getMiddleBackButtonPressedShort);

  engine.registerBooleanInputSymbol("robot-state.hind-back-button-pressed-short",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getHindBackButtonPressedShort);

  engine.registerBooleanInputSymbol("robot-state.head-button-pressed-for-three-seconds",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getHeadButtonPressedForThreeSeconds);

  engine.registerBooleanInputSymbol("robot-state.head-button-pressed-short",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getHeadButtonPressedShort);

  engine.registerBooleanInputSymbol("robot-state.head-button-pressed-short-and-released",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getHeadButtonPressedShortAndReleased);

  // "robot-state.mouth-state"
  engine.registerEnumeratedInputSymbol("robot-state.mouth-state",this,
    (int (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getMouthState);
  engine.registerEnumeratedInputSymbolEnumElement("robot-state.mouth-state","mouth-open",RobotState::mouthOpen);
  engine.registerEnumeratedInputSymbolEnumElement("robot-state.mouth-state","mouth-closed",RobotState::mouthClosed);

  engine.registerBooleanInputSymbol("robot-state.ball-under-head",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getBallUnderHead);

  engine.registerBooleanInputSymbol("robot-state.something-in-front-of-chest",this,
    (bool (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getSomethingInFrontOfChest);

  engine.registerDecimalInputSymbol("robot-state.time-since-something-was-in-front-of-chest",this,
    (double (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getTimeSinceSomethingWasInFrontOfChest);

  engine.registerDecimalInputSymbol("robot-state.distance-to-something-in-front-of-chest",this,
    (double (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getDistanceToSIFOC);

  // "robot-state.system-time"
  engine.registerDecimalInputSymbol("robot-state.system-time",this,
    (double (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getCurrentSystemTime);
  // "robot-state.remaining-capacity"
  engine.registerDecimalInputSymbol("robot-state.remaining-capacity",this,
    (double (Xabsl2FunctionProvider::*)())&RobotStateSymbols::getRemainingCapacity);
}

void RobotStateSymbols::update()
{
}

int RobotStateSymbols::getMouthState()
{
  return robotState.getMouthState();
}

double RobotStateSymbols::getCurrentSystemTime()
{
  return SystemCall::getCurrentSystemTime();
}

double RobotStateSymbols::getRemainingCapacity()
{
  return SystemCall::getRemainingPower();
}

bool RobotStateSymbols::getBallUnderHead()
{
//  cout << sensorDataBuffer.lastFrame().data[SensorData::neckTilt]<<"|"<<sensorDataBuffer.lastFrame().data[SensorData::neckTilt]<<"\n";
//  cout << flush;
  return sensorDataBuffer.lastFrame().data[SensorData::headTilt]>270000;
}

bool RobotStateSymbols::getSomethingInFrontOfChest()
{
  return robotState.getSomethingInFrontOfChest();
}

double RobotStateSymbols::getTimeSinceSomethingWasInFrontOfChest()
{
  return SystemCall::getTimeSince(robotState.getTimeWhenSomethingWasInFrontOfChestLast());
}

double RobotStateSymbols::getDistanceToSIFOC()
{
  return robotState.getDistanceToSIFOC();
}


bool RobotStateSymbols::getAnyBackButtonPressedForOneSecond()
{
  return (
    (
    robotState.getAnyBackButtonPressed() &&
    robotState.getAnyBackButtonDuration() > 1000
    )
    ||
    (
    !robotState.getAnyBackButtonPressed() &&
    robotState.getAnyBackButtonPressedTime() > 1000 &&
    SystemCall::getTimeSince(robotState.getAnyBackButtonTime()) < 700
    )
    );
}

bool RobotStateSymbols::getAnyBackButtonPressedShortAndReleased()
{
  return (
    !robotState.getAnyBackButtonPressed() &&
    robotState.getAnyBackButtonPressedTime() < 1000 &&
    SystemCall::getTimeSince(robotState.getAnyBackButtonTime()) < 700
    );
}

bool RobotStateSymbols::getForeBackButtonPressedShort()
{
  return (
    (
    robotState.getButtonPressed(BodyPercept::backFront) &&
    robotState.getButtonDuration(BodyPercept::backFront) < 800
    )
    );
}

bool RobotStateSymbols::getForeBackButtonPressedShortAndReleased()
{
  return (
    !robotState.getButtonPressed(BodyPercept::backFront) &&
    robotState.getButtonPressedTime(BodyPercept::backFront) < 1000 &&
    SystemCall::getTimeSince(robotState.getButtonTime(BodyPercept::backFront)) < 700
    );
}

bool RobotStateSymbols::getMiddleBackButtonPressedShort()
{
  return (
    (
    robotState.getButtonPressed(BodyPercept::backMiddle) &&
    robotState.getButtonDuration(BodyPercept::backMiddle) < 800
    )
    );
}

bool RobotStateSymbols::getMiddleBackButtonPressedShortAndReleased()
{
  return (
    !robotState.getButtonPressed(BodyPercept::backMiddle) &&
    robotState.getButtonPressedTime(BodyPercept::backMiddle) < 1000 &&
    SystemCall::getTimeSince(robotState.getButtonTime(BodyPercept::backMiddle)) < 700
    );
}

bool RobotStateSymbols::getHindBackButtonPressedShort()
{
  return (
    (
    robotState.getButtonPressed(BodyPercept::backBack) &&
    robotState.getButtonDuration(BodyPercept::backBack) < 800
    )
    );
}

bool RobotStateSymbols::getHindBackButtonPressedShortAndReleased()
{
  return (
    !robotState.getButtonPressed(BodyPercept::backBack) &&
    robotState.getButtonPressedTime(BodyPercept::backBack) < 1000 &&
    SystemCall::getTimeSince(robotState.getButtonTime(BodyPercept::backBack)) < 700
    );
}

bool RobotStateSymbols::getHeadButtonPressedForThreeSeconds()
{
  return (
    (
    robotState.getButtonPressed(BodyPercept::head) &&
    robotState.getButtonDuration(BodyPercept::head) > 3000
    )
    ||
    (
    !robotState.getButtonPressed(BodyPercept::head) &&
    robotState.getButtonPressedTime(BodyPercept::head) > 3000 &&
    SystemCall::getTimeSince(robotState.getButtonTime(BodyPercept::head)) < 500
    )
    );
}

bool RobotStateSymbols::getHeadButtonPressedShort()
{
  return (
    (
    robotState.getButtonPressed(BodyPercept::head) &&
//    robotState.getButtonDuration(BodyPercept::head) > 100 &&
    robotState.getButtonDuration(BodyPercept::head) < 800
    )
    );
}

bool RobotStateSymbols::getHeadButtonPressedShortAndReleased()
{
  return (
    !robotState.getButtonPressed(BodyPercept::head) &&
//    robotState.getButtonPressedTime(BodyPercept::head) > 100 &&
    robotState.getButtonPressedTime(BodyPercept::head) < 1000 &&
    SystemCall::getTimeSince(robotState.getButtonTime(BodyPercept::head)) < 700
    );
}

/*
* Change Log
* 
* $Log: RobotStateSymbols.cpp,v $
* Revision 1.10  2004/06/29 15:13:59  risler
* removed touch sensor averaging and minimum press time, not necessary for ers7 buttons
*
* Revision 1.9  2004/06/17 15:55:35  cesarz
* some changes
*
* Revision 1.8  2004/06/16 17:07:33  cesarz
* Moved body PSD calculations
*
* Revision 1.7  2004/06/16 14:43:51  risler
* added obstacles symbols
* added symbol obstacles.opponent-close-to-ball
*
* Revision 1.6  2004/06/08 18:34:42  juengel
* Added robot-state.time-since-something-was-in-front-of-chest.
*
* Revision 1.5  2004/06/02 14:09:48  brueckne
* improved new button interface
*
* Revision 1.4  2004/05/27 17:13:36  jhoffman
* - renaming: tilt1 -> neckTilt,  pan -> headPan,  tilt2 -> headTilt
* - clipping included for setJoints
* - removed some microrad/rad-bugs
* - bodyPosture constructor and "=" operator fixed
*
* Revision 1.3  2004/05/25 20:14:30  tim
* added something-in-front-of-chest
*
* Revision 1.2  2004/05/24 14:13:51  juengel
* New button evaluation.
*
* Revision 1.1.1.1  2004/05/22 17:17:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/04/08 15:33:01  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.3  2004/03/08 00:58:58  roefer
* Interfaces should be const
*
* Revision 1.2  2004/01/24 11:40:13  juengel
* Added ERS7 switches (head, backFront, backBack).
*
* Revision 1.1  2003/10/22 22:18:44  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:29  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/08/09 14:56:09  dueffert
* symbol remaining-capacity added
*
* Revision 1.2  2003/07/06 12:33:05  dueffert
* robot.state-system.time added
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/06/21 13:06:51  goehring
* Debug Collisionsymbols
*
* Revision 1.4  2003/06/20 13:27:20  risler
* added tailLeft tailRight messages
*
* Revision 1.3  2003/06/20 10:32:27  goehring
* Added CollisionStateSymbols
*
* Revision 1.2  2003/06/10 15:09:07  goehring
* Collision-Detector Xabsl symbols added
*
* Revision 1.1  2003/05/07 17:54:46  juengel
* Added missing symbols to GT2003BehaviorControl.
*
*/

