/** 
* @file RoboCupGameManagerSymbols.cpp
*
* Implementation of class RoboCupGameManagerSymbols.
*
* @author Matthias Jngel
*/

#include "RoboCupGameManagerSymbols.h"


RoboCupGameManagerSymbols::RoboCupGameManagerSymbols(const BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces),
lastGameControlDataState(ROBOCUP_STATE_INITIAL),
timeWhenOwnTeamScored(0), lastOwnScore(0),
timeWhenOpponentTeamScored(0), lastOpponentScore(0), currentState(ROBOCUP_STATE_INITIAL)
{
}


void RoboCupGameManagerSymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "game.state"
  engine.registerEnumeratedInputSymbol("game.state",this,
    (int (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getState);
  engine.registerEnumeratedInputSymbolEnumElement("game.state","game.state.initial",ROBOCUP_STATE_INITIAL);
  engine.registerEnumeratedInputSymbolEnumElement("game.state","game.state.ready",ROBOCUP_STATE_READY);
  engine.registerEnumeratedInputSymbolEnumElement("game.state","game.state.set",ROBOCUP_STATE_SET);
  engine.registerEnumeratedInputSymbolEnumElement("game.state","game.state.playing",ROBOCUP_STATE_PLAYING);
  engine.registerEnumeratedInputSymbolEnumElement("game.state","game.state.penalized",ROBOCUP_STATE_PENALIZED);
  engine.registerEnumeratedInputSymbolEnumElement("game.state","game.state.finished",ROBOCUP_STATE_FINISHED);
  // "game.current-state"
  engine.registerEnumeratedOutputSymbol("game.current-state",(int*) &currentState);
  engine.registerEnumeratedOutputSymbolEnumElement("game.current-state","game.current-state.initial",ROBOCUP_STATE_INITIAL);
  engine.registerEnumeratedOutputSymbolEnumElement("game.current-state","game.current-state.ready",ROBOCUP_STATE_READY);
  engine.registerEnumeratedOutputSymbolEnumElement("game.current-state","game.current-state.set",ROBOCUP_STATE_SET);
  engine.registerEnumeratedOutputSymbolEnumElement("game.current-state","game.current-state.playing",ROBOCUP_STATE_PLAYING);
  engine.registerEnumeratedOutputSymbolEnumElement("game.current-state","game.current-state.penalized",ROBOCUP_STATE_PENALIZED);
  engine.registerEnumeratedOutputSymbolEnumElement("game.current-state","game.current-state.finished",ROBOCUP_STATE_FINISHED);
  // "game.kickoff"
  engine.registerEnumeratedInputSymbol("game.kickoff",this,
    (int (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getKickoff);
  engine.registerEnumeratedInputSymbolEnumElement("game.kickoff","game.kickoff.invalid",ROBOCUP_KICKOFF_INVALID);
  engine.registerEnumeratedInputSymbolEnumElement("game.kickoff","game.kickoff.own",ROBOCUP_KICKOFF_OWN);
  engine.registerEnumeratedInputSymbolEnumElement("game.kickoff","game.kickoff.opponent",ROBOCUP_KICKOFF_OPPONENT);
  // "game.team-color"
  engine.registerEnumeratedInputSymbol("game.team-color",this,
    (int (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getTeamColor);
  engine.registerEnumeratedInputSymbolEnumElement("game.team-color","red",Player::red);
  engine.registerEnumeratedInputSymbolEnumElement("game.team-color","blue",Player::blue);
  // "game.set-team-color"
  engine.registerEnumeratedOutputSymbol("game.set-team-color",this,
    (void (Xabsl2FunctionProvider::*)(int))&RoboCupGameManagerSymbols::setTeamColor);
  engine.registerEnumeratedOutputSymbolEnumElement("game.set-team-color","red",Player::red);
  engine.registerEnumeratedOutputSymbolEnumElement("game.set-team-color","blue",Player::blue);
  // "game.own-team-scored"
  engine.registerBooleanInputSymbol("game.own-team-scored",this,
    (bool (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getOwnTeamScored);
  // "game.opponent-team-scored"
  engine.registerBooleanInputSymbol("game.opponent-team-scored",this,
    (bool (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getOpponentTeamScored);
  // "game.own-score"
  engine.registerDecimalInputSymbol("game.own-score",this,
    (double (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getOwnScore);
  // "game.opponent-score"
  engine.registerDecimalInputSymbol("game.opponent-score",this,
    (double (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getOpponentScore);
  // "game.time-since-last-message"
  engine.registerDecimalInputSymbol("game.time-since-last-message",this,
    (double (Xabsl2FunctionProvider::*)())&RoboCupGameManagerSymbols::getTimeSinceLastMessage);
  // self symbols
}

void RoboCupGameManagerSymbols::update()
{
  if ((gameControlData.data.state == ROBOCUP_STATE_READY)&&
      ((lastGameControlDataState == ROBOCUP_STATE_PLAYING)||
       (lastGameControlDataState == ROBOCUP_STATE_PENALIZED)))
  {
    if (gameControlData.data.ownScore > lastOwnScore) 
    {
      timeWhenOwnTeamScored = SystemCall::getCurrentSystemTime();
    }
    if (gameControlData.data.opponentScore > lastOpponentScore)
    {
      timeWhenOpponentTeamScored = SystemCall::getCurrentSystemTime();
    }
  }
  lastOwnScore = gameControlData.data.ownScore;
  lastOpponentScore = gameControlData.data.opponentScore;
  lastGameControlDataState = gameControlData.data.state;

  if (getTimeSinceLastMessage() < 500)
  {
    getPlayer().setTeamColor(
      gameControlData.data.teamColor == ROBOCUP_TEAMCOLOR_RED ? Player::red : Player::blue);
  }
}

void RoboCupGameManagerSymbols::setTeamColor(int teamColor)
{
  getPlayer().setTeamColor((Player::teamColor)teamColor);
}

int RoboCupGameManagerSymbols::getState()
{
  return (int)gameControlData.data.state;
}

int RoboCupGameManagerSymbols::getKickoff()
{
  return (int)gameControlData.data.kickoff;
}

int RoboCupGameManagerSymbols::getTeamColor() 
{ 
  return (int)(getPlayer().getTeamColor()); 
}

bool RoboCupGameManagerSymbols::getOwnTeamScored()
{
  return (SystemCall::getTimeSince(timeWhenOwnTeamScored) < 5000);
}

bool RoboCupGameManagerSymbols::getOpponentTeamScored()
{
  return (SystemCall::getTimeSince(timeWhenOpponentTeamScored) < 5000);
}

double RoboCupGameManagerSymbols::getOwnScore()
{
  return (double)gameControlData.data.ownScore;
}

double RoboCupGameManagerSymbols::getOpponentScore()
{
  return (double)gameControlData.data.opponentScore;
}

double RoboCupGameManagerSymbols::getTimeSinceLastMessage()
{
  return (double)SystemCall::getTimeSince(gameControlData.timeStamp);
}



/*
* Change Log
* 
* $Log: RoboCupGameManagerSymbols.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:16:59  cvsadm
* created new repository GT2004_WM
*
* Revision 1.11  2004/04/09 14:06:45  tim
* integrated changes from GO2004
*
* Revision 1.10  2004/04/08 10:08:42  risler
* fixed and unified team color symbols
*
* Revision 1.9  2004/04/07 12:29:00  risler
* ddd checkin after go04 - first part
*
* Revision 1.3  2004/04/01 18:55:53  risler
* bugfix RoboCupGameManager team color interpertreted correctly
*
* Revision 1.2  2004/03/31 00:10:15  risler
* implemented new kickoff rules
*
* Revision 1.1.1.1  2004/03/29 08:28:49  Administrator
* initial transfer from tamara
*
* Revision 1.8  2004/03/27 22:18:55  loetzsch
* bug fix
*
* Revision 1.7  2004/03/27 15:19:33  loetzsch
* in ::update sets the teamColor as sent from the game manager
*
* Revision 1.6  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.5  2004/03/25 15:49:47  tim
* changed handling of current state
*
* Revision 1.4  2004/03/25 15:04:25  tim
* added initialisation of currentState
*
* Revision 1.3  2004/03/15 12:50:31  tim
* Adaptions to new GameController
*
* Revision 1.2  2004/03/08 00:58:57  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/22 22:18:44  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:28  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/07/06 09:34:58  dueffert
* *-team-scored improved!
*
* Revision 1.2  2003/07/05 17:34:27  dueffert
* scoring goals can only be signalized by sending ready
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2003/05/25 22:37:02  loetzsch
* finished the game state options of GT2003
*
* Revision 1.1  2003/05/07 17:54:46  juengel
* Added missing symbols to GT2003BehaviorControl.
*
*/

