/** 
* @file BallSymbols.cpp
*
* Implementation of class BallSymbols.
*
* @author Martin Ltzsch
*/

#include "BallSymbols.h"
#include "Tools/Math/Geometry.h"
#include "Tools/FieldDimensions.h"

BallSymbols::BallSymbols(const BehaviorControlInterfaces& interfaces) : 
BehaviorControlInterfaces(interfaces), 
ballIsHandledAtTheMoment(false),
ballWasHandledInLastFrame(false),
playBallPrecisely(false),
ballPrecision(0)
{
}


void BallSymbols::registerSymbols(Xabsl2Engine& engine)
{
  engine.registerDecimalInputSymbol("ball.seen.distance",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenDistance);
  engine.registerDecimalInputSymbol("ball.seen.angle",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenAngle);
  
  engine.registerDecimalInputSymbol("ball.number-of-images-with-ball-percept", this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getNumberOfImagesWithBallPercept);

  engine.registerDecimalInputSymbol("ball.number-of-images-without-ball-percept", this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getNumberOfImagesWithoutBallPercept);
  
  engine.registerBooleanInputSymbol("ball.ball-was-seen",&ballModel.ballWasSeen);

  engine.registerDecimalInputSymbol("ball.seen.x",&(ballModel.seen.x));
  engine.registerDecimalInputSymbol("ball.seen.y",&(ballModel.seen.y));
  engine.registerDecimalInputSymbol("ball.seen.speed",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenSpeed);
  engine.registerDecimalInputSymbol("ball.seen.relative-speed.x",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenRelativeSpeedX);
  engine.registerDecimalInputSymbol("ball.seen.relative-speed.y",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenRelativeSpeedY);
  engine.registerDecimalInputSymbol("ball.seen.distance-to-own-penalty-area",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getDistanceSeenBallToOwnPenaltyArea);
  engine.registerDecimalInputSymbol("ball.seen.distance-x",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenDistanceX);
  engine.registerDecimalInputSymbol("ball.seen.distance-y",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getSeenDistanceY);
  engine.registerDecimalInputSymbol("ball.time-since-last-seen",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getTimeSinceLastSeen);
  engine.registerDecimalInputSymbol("ball.consecutively-seen-time",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getConsecutivelySeenTime);
  engine.registerDecimalInputSymbol("ball.time-since-last-seen-consecutively",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getTimeSinceLastSeenConsecutively);
  engine.registerBooleanInputSymbol("ball.just-seen",this,
    (bool (Xabsl2FunctionProvider::*)())&BallSymbols::getJustSeen);
  engine.registerBooleanInputSymbol("ball.seen.ball-in-front-of-opponent-goal", this,
    (bool (Xabsl2FunctionProvider::*)())&BallSymbols::getBallSeenInFrontOfOpponentGoal);

  engine.registerDecimalInputSymbol("ball.known.distance",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getKnownDistance);
  engine.registerDecimalInputSymbol("ball.known.angle",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getKnownAngle);
  engine.registerDecimalInputSymbol("ball.known.x",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getKnownX);
  engine.registerDecimalInputSymbol("ball.known.y",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getKnownY);
  engine.registerDecimalInputSymbol("ball.motionValidity",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getMotionValidity);
  engine.registerDecimalInputSymbol("ball.time-since-last-known",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getTimeSinceLastKnown);
  engine.registerDecimalInputSymbol("ball.time-after-which-communicated-balls-are-accepted",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getTimeAfterWhichCommunicatedBallAreAccepted);
  engine.registerDecimalInputSymbol("ball.angle-to-opponent-goal",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getAngleToOpponentGoal);
  
  engine.registerDecimalInputSymbol("ball.projected-distance-on-y-axis",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getProjectedDistanceOnYAxis);
  engine.registerDecimalInputSymbol("ball.time-until-ball-crosses-y-axis",this,
    (double (Xabsl2FunctionProvider::*)())&BallSymbols::getTimeUntilBallCrossesYAxis);

  engine.registerBooleanInputSymbol("ball.is-handled-at-the-moment", &ballIsHandledAtTheMoment);
  engine.registerDecimalInputSymbol("ball.average-distance",&averageDistance);

  engine.registerEnumeratedOutputSymbol("ball.handling",&ballWasHandledInLastFrame);
  engine.registerEnumeratedOutputSymbolEnumElement("ball.handling","handling-the-ball",1);
  engine.registerEnumeratedOutputSymbolEnumElement("ball.handling","not-handling-the-ball",0);

  engine.registerBooleanInputSymbol("ball.rolls-by-left"       , &(ballModel.ballState.ballRollsByLeft));
  engine.registerBooleanInputSymbol("ball.rolls-by-right"      , &(ballModel.ballState.ballRollsByRight));
  engine.registerBooleanInputSymbol("ball.rolls-towards-robot" , &(ballModel.ballState.ballRollsTowardsRobot));
  engine.registerBooleanInputSymbol("ball.rolls-fast"          , &(ballModel.ballState.ballRollsFast));

  engine.registerBooleanInputSymbol("ball.play-ball-precisely", &playBallPrecisely);
  engine.registerEnumeratedOutputSymbol("ball.precision",&ballPrecision);
  engine.registerEnumeratedOutputSymbolEnumElement("ball.precision","precision.player",0);
  engine.registerEnumeratedOutputSymbolEnumElement("ball.precision","precision.penalty-shooter",1);
  

}

void BallSymbols::update()
{
  ballDistanceRingBuffer.add(
    (int)Geometry::distanceTo(robotPose.getPose(), ballModel.seen) );
  
  if(ballDistanceRingBuffer.getNumberOfEntries() != 0)
  {
    averageDistance = ballDistanceRingBuffer.getSum() / ballDistanceRingBuffer.getNumberOfEntries();
  }
  else averageDistance = 1000;


  // use the information that was set by the options in the last frame
  ballIsHandledAtTheMoment = (ballWasHandledInLastFrame == 0 ? false : true);

  playBallPrecisely = (ballPrecision == 0 ? false : true);


  // set "ballWasHandledInLastFrame" to false. It will only become true when
  // an option sets it to true again
  ballWasHandledInLastFrame = 0;

}


double BallSymbols::getKnownDistance()
{
  return Geometry::distanceTo(robotPose.getPose(),ballModel.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted));
}

double BallSymbols::getSeenDistance()
{
  return Geometry::distanceTo(robotPose.getPose(),ballModel.seen);
}

double BallSymbols::getSeenDistanceX()
{
  return getSeenDistance() * 
    cos(Geometry::angleTo(robotPose.getPose(),ballModel.seen));
}

double BallSymbols::getSeenDistanceY()
{
  return getSeenDistance() * 
    sin(Geometry::angleTo(robotPose.getPose(),ballModel.seen));
}

double BallSymbols::getKnownAngle()
{
  return toDegrees(Geometry::angleTo(robotPose.getPose(),
    ballModel.getKnownPosition(BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted)));
}

double BallSymbols::getTimeAfterWhichCommunicatedBallAreAccepted()
{
  return (double)BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted;
}

double BallSymbols::getSeenAngle()
{
  return toDegrees(Geometry::angleTo(robotPose.getPose(),
    ballModel.seen));
}

double BallSymbols::getSeenSpeed()
{
  return ballModel.seen.speed.abs();
}

double BallSymbols::getSeenRelativeSpeedX()
{
  return 
    (ballModel.seen.speed.x * cos(robotPose.rotation) -
     ballModel.seen.speed.y * sin(robotPose.rotation));
}

double BallSymbols::getSeenRelativeSpeedY()
{
  return 
    (ballModel.seen.speed.x * sin(robotPose.rotation) +
     ballModel.seen.speed.y * cos(robotPose.rotation));
}

double BallSymbols::getDistanceSeenBallToOwnPenaltyArea()
{
  return FieldDimensions::distanceToOwnPenaltyArea(ballModel.seen);
}

double BallSymbols::getTimeUntilBallCrossesYAxis()
{
  return (double) ballModel.ballState.timeBallCrossesYAxis;
}

double BallSymbols::getProjectedDistanceOnYAxis()
{
  return ballModel.ballState.projectedDistanceOnYAxis;
}

double BallSymbols::getKnownX()
{
  return ballModel.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted).x;
}

double BallSymbols::getKnownY()
{
  return ballModel.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted).y;
}

double BallSymbols::getMotionValidity()
{
  if (ballModel.motionValidity<0){
    return 0.0;
  }
  return ballModel.motionValidity;
}

double BallSymbols::getTimeSinceLastKnown()
{
  return ballModel.getTimeSinceLastKnown(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
}

double BallSymbols::getTimeSinceLastSeen()
{
  return SystemCall::getTimeSince(ballModel.seen.timeWhenLastSeen);
}

double BallSymbols::getConsecutivelySeenTime()
{
  return ballModel.seen.timeUntilSeenConsecutively 
    - ballModel.seen.timeWhenFirstSeenConsecutively;
}

double BallSymbols::getTimeSinceLastSeenConsecutively()
{
  return SystemCall::getTimeSince(ballModel.seen.timeUntilSeenConsecutively);
}

bool BallSymbols::getBallSeenInFrontOfOpponentGoal(){
  return ballModel.seen.ballInFrontOfOpponentGoal;
}

bool BallSymbols::getJustSeen()
{
  return SystemCall::getTimeSince(ballModel.seen.timeWhenLastSeen) < 50;
}

double BallSymbols::getAngleToOpponentGoal()
{
  return toDegrees(normalize(Geometry::angleTo(ballModel.seen, Vector2<double>(xPosOpponentGroundline,yPosCenterGoal))));
}

double BallSymbols::getNumberOfImagesWithBallPercept()
{
  return ballModel.numberOfImagesWithBallPercept;
}

double BallSymbols::getNumberOfImagesWithoutBallPercept()
{
  return ballModel.numberOfImagesWithoutBallPercept;
}

/*
* Change Log
* 
* $Log: BallSymbols.cpp,v $
* Revision 1.8  2004/06/28 00:35:37  juengel
* bug fixed
*
* Revision 1.7  2004/06/27 17:41:01  goetzke
* Added penalty-shooter.
*
* Revision 1.6  2004/06/23 11:37:43  dassler
* New Ballsymbols introduced
* ball.projected-distance-on-y-axis" description="Projected Distance of the ball to the y axis of the robot
* ball.time-until-ball-crosses-y-axis" description="Time until the ball crosses the y axis of the robot
*
* Revision 1.5  2004/06/15 17:50:24  juengel
* Added symbols:
* numberOfImagesWithBallPercept,
* numberOfImagesWithoutBallPercept
*
* Revision 1.4  2004/05/27 18:49:17  kerdels
* added a small 5 frames sliding average for the relative ballspeed,
* added new ballState Representation and adjusted the apropriate files
*
* Revision 1.3  2004/05/22 22:52:02  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.2  2004/05/22 20:47:58  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:16:56  cvsadm
* created new repository GT2004_WM
*
* Revision 1.18  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.17  2004/05/03 15:32:25  uhrig
* Added additional ball state and disabled sending of Kalman filter process model states.
*
* Revision 1.16  2004/04/15 13:12:27  loetzsch
* added setting of the symbol "ball.handling"
*
* Revision 1.15  2004/04/14 20:35:28  loetzsch
* i.m.p.r.o.v.e.d.
*
* Revision 1.14  2004/04/08 15:33:01  wachter
* GT04 checkin of Microsoft-Hellounds
*

* Revision 1.13  2004/04/05 17:56:47  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.3  2004/04/02 11:44:30  jumped
* addded ball.seen.in.front-of-opponent-goal (and supporting functions)
*
* Revision 1.2  2004/04/01 22:08:37  juengel
* Added ball.average-distance.
*
* Revision 1.1.1.1  2004/03/31 11:16:41  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.12  2004/03/28 17:54:11  juengel
* Added ballState "ballRollsFast".
*
* Revision 1.11  2004/03/28 16:33:58  loetzsch
* added input symbol ball.consecutively-seen-time and ball.time-since-last-consecutively-seen
*
* Revision 1.10  2004/03/27 21:11:29  loetzsch
* changed ball.seen-consecutively
*
* Revision 1.9  2004/03/27 14:45:41  loetzsch
* added boolean input symbol "ball.seen-consecutively"
*
* Revision 1.8  2004/03/21 12:44:49  juengel
* Added "wait-for-feedback" to ballState
*
* Revision 1.7  2004/03/08 00:58:54  roefer
* Interfaces should be const
*
* Revision 1.6  2004/02/05 16:56:00  spranger
* fixed ball-state enum
*
* Revision 1.5  2004/02/05 14:21:03  spranger
* added ballState enum
*
* Revision 1.4  2004/02/03 13:20:47  spranger
* renamed all references to  class BallP_osition to BallModel (possibly changed include files)
*
* Revision 1.3  2004/01/29 11:00:51  schumann
* added filtering of ball motion
*
* Revision 1.2  2004/01/29 09:52:43  schumann
* added motion validity symbol to xabsl
*
* Revision 1.1  2003/10/22 22:18:44  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:28  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/06/24 14:31:19  schumann
* added primitive behavior for ball challenge
*
* Revision 1.9  2003/06/22 14:54:12  risler
* ball.seen.relative-speed.x/y added
*
* Revision 1.8  2003/06/19 14:49:04  risler
* symbol ball.getting-closer added
*
* Revision 1.7  2003/06/18 18:31:29  risler
* added relative ball position
*
* Revision 1.6  2003/06/17 19:54:59  risler
* ball caught symbols moved to strategy
*
* Revision 1.5  2003/06/04 11:17:14  risler
* added symbol ball.seen.distance-to-own-penalty-area
*
* Revision 1.4  2003/06/02 14:45:33  loetzsch
* added symbols ball.caught and ball.catch-time
*
* Revision 1.3  2003/06/02 03:57:52  loetzsch
* copied a simple goalie behavior from humboldt2003
*
* Revision 1.2  2003/05/09 15:28:51  loetzsch
* added time-after-which-communicated-ball-are-accepted
*
* Revision 1.1  2003/05/04 11:41:40  loetzsch
* added class BallSymbols
*
*/

