/** 
* @file AngleSymbols.h
*
* Declaration of class AngleSymbols.
*
* @author Matthias Jngel
*/

#ifndef __AngleSymbols_h_
#define __AngleSymbols_h_

#include "../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Actorics/RobotDimensions.h"

/**
* The Xabsl2 symbols that are defined in "angle-symbols.xml"
*
* @author Matthias Jngel
*/ 
class AngleSymbols : public Xabsl2FunctionProvider, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the BehaviorControl module.
  */
  AngleSymbols(const BehaviorControlInterfaces& interfaces);
  
  /** registers the symbols at an engine */
  void registerSymbols(Xabsl2Engine& engine);
  
    /** updates the symbols */
  void update();

  //!@name Input symbols
  //!@{

  //localisation based:
  double getAngleToCenterOfField(); /*!< calculates the value for the symbol "robot-pose.best-angle-away-from-own-goal" */
  double getAngleToOpponentGoal(); /*!< A function for the symbol "robot-pose.angle-to-opponent-goal" */
  double getAngleToPointBehindOpponentGoal(); /*!< calculates the value for the symbol "robot-pose.angle-to-point-behind-opponent-goal" */

  // combined: vision + localisation
  double getBestAngleToOpponentGoal(); /*!< calculates the value for the symbol "robot-pose.best-angle-to-opponent-goal" */
  double getBestAngleToOpponentGoalNoObstacles(); /*!< calculates the value for the symbol "robot-pose.best-angle-to-opponent-goal-no-obstacles" */
  double getBestAngleAwayFromOwnGoal(); /*!< calculates the value for the symbol "robot-pose.best-angle-away-from-own-goal" */
  double getBestAngleAwayFromOwnGoalNoObstacles(); /*!< calculates the value for the symbol "robot-pose.best-angle-away-from-own-goal-no-obstacles" */
  double getGoalieGoalKickAngle(); /*!< calculates the value for the symbol "goalie.goal-kick-angle" */

  //!@}

  /** The angle that is shown by the top white LEDs */
  enum Angles {
    undefined,
    //localisation based:
    angleToCenterOfField,                //input symbol
    angleToOpponentGoal,                 //input symbol
    angleToLeftOpponentGoalPost,
    angleToRightOpponentGoalPost,        
    angleToLeftOpponentGoalCorner,
    angleToRightOpponentGoalCorner,        
    angleToPointBehindOpponentGoal,      //input symbol

    // vision based
    angleToFreePartOfOpponentGoal,   

    // combined: vision + localisation
    bestAngleToOpponentGoal,             //input symbol
    bestAngleToOpponentGoalNoObstacles,  //input symbol
    bestAngleAwayFromOwnGoal,            //input symbol
    bestAngleAwayFromOwnGoalNoObstacles, //input symbol
    goalieGoalKickAngle,                 //input symbol

    numberOfAngles
  };

  double getAngle(Angles id);
  
  static const char* getAngleName(Angles id)
  {
		switch(id)
		{
    case undefined: return "undefined";
    //localisation based:
    case angleToCenterOfField: return "angleToCenterOfField";
    case angleToOpponentGoal: return "angleToOpponentGoal";
    case angleToLeftOpponentGoalPost: return "angleToLeftOpponentGoalPost";
    case angleToRightOpponentGoalPost: return "angleToRightOpponentGoalPost";
    case angleToLeftOpponentGoalCorner: return "angleToLeftOpponentGoalCorner";
    case angleToRightOpponentGoalCorner: return "angleToRightOpponentGoalCorner";
    case angleToPointBehindOpponentGoal: return "angleToPointBehindOpponentGoal";

    // vision based
    case angleToFreePartOfOpponentGoal: return "angleToFreePartOfOpponentGoal";

    // combined: vision + localisation
    case bestAngleToOpponentGoal: return "bestAngleToOpponentGoal";
    case bestAngleToOpponentGoalNoObstacles: return "bestAngleToOpponentGoalNoObstacles";
    case bestAngleAwayFromOwnGoal: return "bestAngleAwayFromOwnGoal";
    case bestAngleAwayFromOwnGoalNoObstacles: return "bestAngleAwayFromOwnGoalNoObstacles";
    case goalieGoalKickAngle: return "goalieGoalKickAngle";
    default: return "please-edit-angle-name";
    }
  }

  Angles angleShownByLEDs;
  void drawAngleShownByLeds();

private:
  void calculateVisionBasedAngles();
  void calculateLocalisationBasedAngles();
  void calculateCombinedAngles();
  double angles[numberOfAngles];
};


#endif // __AngleSymbols_h_

/*
* Change Log
* 
* $Log: AngleSymbols.h,v $
* Revision 1.2  2004/06/23 19:31:10  juengel
* Added angle "undefined"
*
* Revision 1.1  2004/06/22 18:48:56  juengel
* kickAngles clean up
*
*/

