/**
 * @file ValidityBasedBallLocator.h
 * 
 * This file contains a default class for ball-localization.
 *
 * @author <a href="mailto:goehring@informatik.hu-berlin.de">Daniel Goehring</a>
 */

#ifndef __ValidityBasedBallLocator_h_
#define __ValidityBasedBallLocator_h_

#include "BallLocator.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/Math/PIDsmoothedValue.h"

/**
* @class ValidityBasedBallLocator
*
* A solution of the BallLocator module.
*
*
 * @author <a href="mailto:goehring@informatik.hu-berlin.de">Daniel Goehring</a>
*/
class ValidityBasedBallLocator : public BallLocator
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the BallLocator module.
  */
  ValidityBasedBallLocator(const BallLocatorInterfaces& interfaces);

  /** Executes the module */
  virtual void execute();

private:
	  enum {original = 0,
          bearing,
          size,
          combined,
          numOfBallPerceptProperties
    };

    enum {seen = 0,
          speed,
          numOfBallStateProperties};

    enum {historySize = 100};
    enum {maxRegressionFrames = 20};
    
    Vector2<double> offset[numOfBallPerceptProperties][historySize];

    Pose2D          robotOdo[historySize];

    Vector2<double> ballPose[numOfBallStateProperties][historySize];

    double          validity[historySize];
    unsigned long   frameNumber[historySize];
    unsigned long   step;

    Vector2<double> ballOffset;
    Vector2<double> ballOffsetByBearing;
    Vector2<double> ballOffsetBySize;
    Vector2<double> ballOffsetCartesian;

    Vector2<double> lastPropagated;

    Pose2D lastBallPercept,
           lastRobotOdometry,
           distOdometry;

    
    double avgValidity[maxRegressionFrames + 2];
    
    double avgX;
    double avgY;
    double sumUp;
    double sumDn;
    double quot;
    double dX;
    double dY;
    double distance;
    double angle;

    bool set;
    unsigned long dFrameNumber;


    long consecutiveFramesBallSeen;
    long consecutiveFramesBallNotSeen;
    
    unsigned long lastTimeBallSeen;

    Vector2<double> relative2FieldCoord(RobotPose rp, double x, double y);
    void compensateOdometry();

    bool handleMessage(InMessage& message);
    unsigned long frameToImageNumber(unsigned long imageDiff);

    double validityStep;
    double maxDist;
    double minSpeedAbs;
};

#endif// __ValidityBasedBallLocator_h_

/*
 * Change log :
 * $Log: ValidityBasedBallLocator.h,v $
 * Revision 1.3  2004/05/26 00:09:31  goehring
 * ValidityBasedBallLocator added
 *
 * Revision 1.6  2004/05/17 14:11:44  goehring
 * ballPosition.propagated.speed implemented
 *
 * Revision 1.5  2004/05/05 21:19:53  goehring
 * no message
 *
 * Revision 1.4  2004/04/27 18:22:06  goehring
 * Speed clipping for small speeds implemented
 *
 * Revision 1.3  2004/04/27 13:04:44  goehring
 * Speed calculation improved
 *
 * Revision 1.2  2004/04/26 15:37:57  goehring
 * SpeedCalculation optimized
 *
 * Revision 1.1  2004/04/21 20:11:40  goehring
 * BallLocators added
 *
 * Revision 1.1.1.1  2004/03/31 11:16:37  loetzsch
 * created ATH repository for german open 2004
 *
 * Revision 1.23  2004/03/08 00:58:35  roefer
 * Interfaces should be const
 *
 * Revision 1.22  2004/02/08 22:05:29  goehring
 * Design change to ringbuffers
 *
 * Revision 1.21  2004/02/06 18:45:39  goehring
 * Concept Change
 *
 * Revision 1.3  2004/02/05 21:36:50  goehring
 * Validity tested
 *
 * Revision 1.2  2003/12/09 16:33:10  jhoffman
 * - some cleaning up
 *
 * Revision 1.1  2003/12/02 13:51:01  goehring
 * no message
 *
 * 
 *
 */
