/**
 * @file BallSideDetector.h
 * 
 * This file contains default class for ball-side-detector.
 *
 * @author <a href="mailto:heinze@informatik.hu-berlin.de">Andreas Heinze</a>
*/


#ifndef __BallSideDetctor_h_
#define __BallSideDetctor_h_

#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/CalibrationRequest.h"
#include "Representations/Cognition/BallModel.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/RingBuffer.h"

/**
* @class BallSideDetector
*
* Recognize a close fast ball based on percepts
*
* @author <a href="mailto:heinze@informatik.hu-berlin.de">Andreas Heinze</a>
*/
class BallSideDetector
{
public:
  /** 
  * constructor
  * @param ballPercept A reference to the last ball percept
  * @param ballModel Gives the class access to the BallModel
  */
  BallSideDetector(const BallPercept& ballPercept, 
    const RobotPose& robotPose,
    const CalibrationRequest& calibrationRequest,
    BallModel& ballModel);

  /** Executes the detector */
  void execute();

  /** A reference to the last ball percept */
  const BallPercept& ballPercept;

  /** A rference to the calibration request */
  const CalibrationRequest& calibrationRequest;

  /** A BallModel where to store the ball data */
  BallModel& ballModel;

  /** A RobotPose where to read the robots position */
  const RobotPose& robotPose;

private:
  void determineFastBall();
  void determineBallSide();
  void calculateLastAverageDistance();
  void calculateCurrentAverageDistance();
  bool isBallPerceptValid();

  void calibrate();
  enum {sizeOfBallPerceptBuffer = 10};
  RingBuffer<BallPercept, sizeOfBallPerceptBuffer> ballPerceptBuffer;
  RingBuffer<BallPercept, sizeOfBallPerceptBuffer> validBallPerceptBuffer;

  bool waitForFeedback;
  int currentFrame;
  int numberOfConsecutivePerceptsWithBall;
  int numberOfConsecutivePerceptsWithoutBall;

  double currentAverageDistance;
  double lastAverageDistance;
};

#endif

/*
 * Change log :
 * $Log: BallSideDetector.h,v $
 * Revision 1.1.1.1  2004/05/22 17:15:17  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.11  2004/04/05 17:56:45  loetzsch
 * merged the local German Open CVS of the aibo team humboldt with the tamara CVS
 *
 * Revision 1.2  2004/04/01 14:46:48  juengel
 * Improved.
 *
 * Revision 1.10  2004/03/29 15:03:51  juengel
 * Improved
 *
 * Revision 1.9  2004/03/29 11:06:34  juengel
 * Improved.
 *
 * Revision 1.8  2004/03/28 16:02:18  juengel
 * improved
 *
 * Revision 1.7  2004/03/21 19:11:47  juengel
 * improved
 *
 * Revision 1.6  2004/03/21 12:39:32  juengel
 * Added CalibrationRequest to BallLocatorInterfaces.
 *
 * Revision 1.5  2004/02/05 10:32:46  heinze
 * worked at ball-side-detector
 *
 * Revision 1.4  2004/02/03 13:19:49  spranger
 * renamed all references to  class BallPosition to BallModel
 *
 * Revision 1.3  2004/01/20 17:56:44  heinze
 * worked at ball-side-detector
 *
 * Revision 1.2  2004/01/20 16:45:52  loetzsch
 * The BallLocator has access to the complete BallPosition again
 *
 * Revision 1.1  2004/01/13 18:35:24  heinze
 * added ball-side-detector
 *
 *
 */

