/**
* @file BallSideDetector.cpp
* 
* This file contains a ball-side-detector class.
*
* @author <a href="mailto:heinze@informatik.hu-berlin.de">Andreas Heinze</a>
**/

#include "BallSideDetector.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Tools/MessageQueue/MessageQueue.h"

BallSideDetector::BallSideDetector(const BallPercept& ballPercept, 
                                   const RobotPose& robotPose, 
                                   const CalibrationRequest& calibrationRequest,
                                   BallModel& ballModel) :
ballPercept(ballPercept),
ballModel(ballModel),
calibrationRequest(calibrationRequest),
robotPose(robotPose),
waitForFeedback(false),
numberOfConsecutivePerceptsWithBall(0),
numberOfConsecutivePerceptsWithoutBall(0)
{
}

bool BallSideDetector::isBallPerceptValid()
{
  Vector2<double> ballPositionSizeBased, ballPositionBearingBased;
  ballPercept.getOffsetSizeBased(ballPositionSizeBased);
  ballPercept.getOffsetBearingBased(ballPositionBearingBased);

  double error = sqr(ballPositionSizeBased.abs()) / 15000 * 10;
  double distance = Geometry::distance(ballPositionSizeBased, ballPositionBearingBased);
//  OUTPUT(idText, text, "error: " << error);
//  OUTPUT(idText, text, "distance: " << distance);
  if(distance < error) 
  {
//    OUTPUT(idText, text, "valid");
    return true;
  }
  else 
    return false;
}

void BallSideDetector::execute()
{
  ballModel.ballState.reset();
  currentFrame = ballPercept.frameNumber;
 
  ballPerceptBuffer.add(ballPercept);

  determineFastBall();

  if(ballPercept.ballWasSeen && isBallPerceptValid())
  {
    numberOfConsecutivePerceptsWithoutBall = 0;
    numberOfConsecutivePerceptsWithBall++;

    validBallPerceptBuffer.add(ballPercept);

    calculateCurrentAverageDistance();
    calculateLastAverageDistance();

//    determineFastBall();
    determineBallSide();
  }
  else 
  {
    if(numberOfConsecutivePerceptsWithoutBall == 0)
      lastAverageDistance = currentAverageDistance;

    numberOfConsecutivePerceptsWithBall = 0;
    numberOfConsecutivePerceptsWithoutBall++;
  }
}  

void BallSideDetector::calculateCurrentAverageDistance()
{
  if(validBallPerceptBuffer.getNumberOfEntries() < 5) return;
  if(numberOfConsecutivePerceptsWithBall < 5) return;

  currentAverageDistance = 0;
  Vector2<double> ballPosition;
  for(int i = 0; i < 5; i++)
  {
    validBallPerceptBuffer[i].getOffsetSizeBased(ballPosition);
    currentAverageDistance += ballPosition.abs();
  }
  currentAverageDistance /= 5;
}


void BallSideDetector::calculateLastAverageDistance()
{
  if(validBallPerceptBuffer.getNumberOfEntries() < 10) return;
  if(numberOfConsecutivePerceptsWithBall < 10) return;

  lastAverageDistance = 0;
  Vector2<double> ballPosition;
  for(int i = 5; i < 10; i++)
  {
    validBallPerceptBuffer[i].getOffsetSizeBased(ballPosition);
    lastAverageDistance += ballPosition.abs();
  }
  lastAverageDistance /= 5;
}

void BallSideDetector::determineFastBall()
{
//  if(ballPerceptBuffer.getNumberOfEntries() < sizeOfBallPerceptBuffer) return;
//VIVI
//  Eingabe ist:
//  ballPercept.*

//  oder 
//  ballPerceptBuffer[n].*

 
  Vector2<double> ballPositionSizeBased, ballPositionBearingBased;

  ballPercept.getOffsetSizeBased(ballPositionSizeBased);
  ballPercept.getOffsetBearingBased(ballPositionBearingBased);

// OUTPUT(idText, text, ballPercept.x);
// niemals einchecken
  
//  ballPositionSizeBased.*
//  ballPositionSizeBased.x
//  ballPositionSizeBased.y
//  abstandZumBallS = ballPositionSizeBased.abs();
//  abstandZumBallB = ballPositionBearingBased.abs();

//  ballPercept.frameNumber; 

//  Abstand zw. zwei frames: 8 ms
//  ca. 30 Aufrufe pro Sekunde

  
//  if(?????)
//  {
//    Ausgabe
      ballModel.ballState.ballRollsFast = true;
//  }
  //sonst nix
}



/*  double error = max(200, (currentAverageDistance) / 5);

  //  OUTPUT(idText, text, "err: " << error << " dist: " << ballPosition[0].abs());
  if(
    (lastAverageDistance - currentAverageDistance) > error &&
    (numberOfConsecutivePerceptsWithoutBall < 30 ||
    numberOfConsecutivePerceptsWithBall > 10 )
    )
  {
    ballModel.ballState = BallModel::ballRollsFast;
  }
*/

/*
  if(validBallPerceptBuffer.getNumberOfEntries() < sizeOfBallPerceptBuffer) return;

  //Ballposition
  Vector2<double> ballPosition[sizeOfBallPerceptBuffer];
  double ballSpeed[sizeOfBallPerceptBuffer];
  
  for(int i = 0; i < sizeOfBallPerceptBuffer; i++)
  {
    validBallPerceptBuffer[i].getOffsetSizeBased(ballPosition[i]);
    if(i!=0)
    {
      ballSpeed[i-1] = 
        1000 * Geometry::distance(ballPosition[i], ballPosition[i-1]) / 
        ( (validBallPerceptBuffer[i-1].frameNumber - validBallPerceptBuffer[i].frameNumber) * 8 );
    }
  }

  double error = max(400, (ballPosition[0].abs() * ballPosition[1].abs()) / 15000 * 3);
//  OUTPUT(idText, text, "err: " << error << " dist: " << ballPosition[0].abs());
  if(
    Geometry::distance(ballPosition[0], ballPosition[1]) > error &&
    validBallPerceptBuffer[0].frameNumber - validBallPerceptBuffer[1].frameNumber < 125 &&
    currentFrame - validBallPerceptBuffer[0].frameNumber < 100
    )
  {
    ballModel.ballState = BallModel::ballRollsFast;
  }

*/

void BallSideDetector::determineBallSide()
{
  if(ballPerceptBuffer.getNumberOfEntries() < sizeOfBallPerceptBuffer) return;

  //Ballposition
  Vector2<double> ballPositionSizeBased[sizeOfBallPerceptBuffer];
  Vector2<double> ballPositionBearingBased[sizeOfBallPerceptBuffer];
  double ballSpeed[sizeOfBallPerceptBuffer];
  
  for(int i = 0; i < sizeOfBallPerceptBuffer; i++)
  {
    ballPerceptBuffer[i].getOffsetBearingBased(ballPositionBearingBased[i]);
    ballPerceptBuffer[i].getOffsetSizeBased(ballPositionSizeBased[i]);
    if(i!=0)
    {
      ballSpeed[i-1] = 
        1000 * Geometry::distance(ballPositionBearingBased[i], ballPositionBearingBased[i-1]) / 
        ( (ballPerceptBuffer[i-1].frameNumber - ballPerceptBuffer[i].frameNumber) * 8 );
    }
  }

//  OUTPUT(idText,text,"bs: " << ballSpeed[0] << "  fn:  " << (ballPerceptBuffer[0].frameNumber - ballPerceptBuffer[1].frameNumber));
  if ( 
    (ballPositionBearingBased[0].y < -100) && 
    (ballPositionBearingBased[0].y > -300) && 
    (ballPositionBearingBased[0].x < 350) &&
    (ballSpeed[0] + ballSpeed[1] + ballSpeed[2]) / 3 > 200 &&
    (ballPositionBearingBased[1].x > 350) //&&
//    (ballPositionBearingBased[2].x > ballPositionBearingBased[1].x + 100)
    )
  {
    ballModel.ballState.ballRollsByRight = true;
  }
  if (
    (ballPositionBearingBased[0].y > 100) && 
    (ballPositionBearingBased[0].y < 300) && 
    (ballPositionBearingBased[0].x < 350) &&
    (ballSpeed[0] + ballSpeed[1] + ballSpeed[2]) / 3 > 200 &&
    (ballPositionBearingBased[1].x > 350) //&&
//    (ballPositionBearingBased[2].x > ballPositionBearingBased[1].x + 100)
    )
  {
    ballModel.ballState.ballRollsByLeft = true;
  }
}

void BallSideDetector::calibrate()
{
  if(!waitForFeedback)
  {
//    if(ballModel.ballState != BallModel::none) waitForFeedback = true;
    if(numberOfConsecutivePerceptsWithoutBall > 10 && ballPerceptBuffer.getNumberOfEntries() >= sizeOfBallPerceptBuffer) waitForFeedback = true;
  }
  else
  {
    if(calibrationRequest.feedback != CalibrationRequest::noFeedback) 
    {
      waitForFeedback = false;
      
      OutTextFile f("ball.log",true);

      switch(calibrationRequest.feedback)
      {
      case CalibrationRequest::ballRolledByLeft:  f << "l"; break;
      case CalibrationRequest::ballRolledByRight: f << "r"; break;
      case CalibrationRequest::ballDidNotRollBy:  f << "n"; break;
      }

      f << ballPerceptBuffer.getNumberOfEntries();
      
      for(int i = 0; i < ballPerceptBuffer.getNumberOfEntries(); i++)
      {
        f << ballPerceptBuffer[i];
      }
      
      ballPerceptBuffer.init();
      ballModel.ballState.reset();
    }
  }
}

/*
void BallSideDetector::execute()
{
  ballModel.ballState = BallModel::none;
  currentFrame = ballPercept.frameNumber;

  if(ballPercept.ballWasSeen)
  {
    numberOfConsecutivePerceptsWithoutBall = 0;
    numberOfConsecutivePerceptsWithBall++;
    if(!waitForFeedback)
    {
      ballPerceptBuffer.add(ballPercept);
      Vector2<double> ballPositionSizeBased, ballPositionBearingBased;
      ballPercept.getOffsetSizeBased(ballPositionSizeBased);
      ballPercept.getOffsetBearingBased(ballPositionBearingBased);
      if(Geometry::distance(ballPositionSizeBased, ballPositionBearingBased) < 150)
      {
        validBallPerceptBuffer.add(ballPercept);
      }
    }
    calculateCurrentAverageDistance();
    if(numberOfConsecutivePerceptsWithBall % 5 == 0)
    {
      lastAverageDistance = currentAverageDistance;
    }
    determineFastBall();
    determineBallSide();
  }
  else 
  {
    if(numberOfConsecutivePerceptsWithoutBall == 0) calculateLastAverageDistance();

    numberOfConsecutivePerceptsWithBall = 0;
    numberOfConsecutivePerceptsWithoutBall++;
  }
  
  if(calibrationRequest.mode == CalibrationRequest::calibrateBallSideDetector)
    calibrate();
}  
*/



/*
* Change log :
* $Log: BallSideDetector.cpp,v $
* Revision 1.2  2004/05/27 18:49:17  kerdels
* added a small 5 frames sliding average for the relative ballspeed,
* added new ballState Representation and adjusted the apropriate files
*
* Revision 1.1.1.1  2004/05/22 17:15:17  cvsadm
* created new repository GT2004_WM
*
* Revision 1.22  2004/04/16 16:14:55  juengel
* Preparations for determineFastBall
*
* Revision 1.21  2004/04/05 17:56:45  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.2  2004/04/01 14:46:49  juengel
* Improved.
*
* Revision 1.20  2004/03/29 18:39:40  juengel
* Improved.
*
* Revision 1.19  2004/03/29 15:03:51  juengel
* Improved
*
* Revision 1.18  2004/03/29 11:06:34  juengel
* Improved.
*
* Revision 1.17  2004/03/28 17:54:11  juengel
* Added ballState "ballRollsFast".
*
* Revision 1.16  2004/03/28 16:02:18  juengel
* improved
*
* Revision 1.15  2004/03/23 14:41:27  juengel
* OUTPUTS removed
*
* Revision 1.14  2004/03/21 19:11:48  juengel
* improved
*
* Revision 1.13  2004/03/21 12:39:32  juengel
* Added CalibrationRequest to BallLocatorInterfaces.
*
* Revision 1.12  2004/03/16 14:00:17  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.2  2004/03/08 18:23:45  loetzsch
* removed useless inclusion of ATH2004SimpleBasicBehaviors.h
* Revision 1.11  2004/02/06 20:23:32  roefer
* Warnings removed
*
* Revision 1.10  2004/02/05 15:50:14  heinze
* ball-side-detector
* - worked at it
*
* Revision 1.9  2004/02/05 15:28:07  heinze
* ball-side-detector
* - comment out some OUTPUTS
*
* Revision 1.8  2004/02/05 14:19:46  heinze
* ball-side-detector
* - added dependings to determine, whether the ball rolls right or left
*
* Revision 1.7  2004/02/05 10:32:46  heinze
* worked at ball-side-detector
*
* Revision 1.6  2004/02/03 16:02:11  heinze
* - worked at ball-side-detector
*
* Revision 1.5  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.4  2004/01/23 16:32:54  roefer
* Warnings removed
*
* Revision 1.3  2004/01/20 17:56:44  heinze
* worked at ball-side-detector
*
* Revision 1.2  2004/01/20 16:45:52  loetzsch
* The BallLocator has access to the complete BallPosition again
*
* Revision 1.1  2004/01/13 18:35:23  heinze
* added ball-side-detector
*
*
*/


