/**
* @file BallLocatorSelector.h
* 
* Definition of class BallLocatorSelector
*
* @author Max Risler
* @author Martin Ltzsch
* @author Matthias Jngel
*/

#ifndef __BallLocatorSelector_h_
#define __BallLocatorSelector_h_

#include "Tools/Module/ModuleSelector.h"

#include "BallLocator.h"
#include "KalmanBallLocator.h"
#include "ValidityBasedBallLocator.h"
#include "KalmanComboBallLocator.h"
#include "PerceptBallLocator.h"

/**
* @class BallLocatorSelector
* 
* A module selector for ball locators.
*
* @author Martin Ltzsch
*/
class BallLocatorSelector : public ModuleSelector, public BallLocatorInterfaces
{
public:
/**
* Constructor.
* @param handler The module handler of the process.
* @param interfaces The paramters of the BallLocator module.
  */
  BallLocatorSelector(ModuleHandler &handler, const BallLocatorInterfaces& interfaces)
    : ModuleSelector(SolutionRequest::ballLocator),
    BallLocatorInterfaces(interfaces)
  {
    handler.setModuleSelector(SolutionRequest::ballLocator, this);
  }
  
  /** 
  * Is called on start and when the selected solution changes
  * to create a specific solution.
  * @param id The id of the solution to create
  * @return The created solution or 0
  */
  virtual Module* createSolution(SolutionRequest::ModuleSolutionID id)
  {
    switch(id)
    {
    case SolutionRequest::kalmanBallLocator:
      return new KalmanBallLocator(*this);
    case SolutionRequest::validityBasedBallLocator:
      return new ValidityBasedBallLocator(*this);
    case SolutionRequest::kalmanComboBallLocator:
      return new KalmanComboBallLocator(*this,
        new KalmanBallLocator(*this),
        new PerceptBallLocator(*this));
    default:
      return 0;
    }
  }
  
};

#endif //__BallLocatorSelector_h_

/*
* Change log:
*
* $Log: BallLocatorSelector.h,v $
* Revision 1.6  2004/07/02 09:31:27  jhoffman
* Kalman-Combo now uses the "PerceptBallLocator" for the position; odometry is used for propagation of position when ball is not seen
*
* Revision 1.5  2004/07/01 12:38:28  dassler
* introduced new Balllocator KalmanComboBallLocator
* ball.seen is set by the Validity and the rest by the Kalman Filter
*
* Revision 1.4  2004/05/26 00:08:58  goehring
* ValidityBasedBallLocator added
*
* Revision 1.3  2004/05/22 22:52:01  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.2  2004/05/22 22:01:06  loetzsch
* removed some ball locators
*
* Revision 1.1.1.1  2004/05/22 17:15:16  cvsadm
* created new repository GT2004_WM
*
* Revision 1.13  2004/04/21 20:11:40  goehring
* BallLocators added
*
* Revision 1.12  2004/04/09 11:35:51  roefer
* Bremen Byters German Open check-in
*
* Revision 1.11  2004/04/08 15:33:00  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.10  2004/04/07 13:25:43  thomas
* ddd checkin after go04 - second part fixed solutions
*
* Revision 1.9  2004/04/07 12:28:56  risler
* ddd checkin after go04 - first part
*
* Revision 1.8  2004/03/11 14:27:31  uhrig
* KalmanBallLocator was added
*
* Revision 1.7  2004/03/08 00:58:36  roefer
* Interfaces should be const
*
* Revision 1.6  2004/01/21 14:31:57  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.5  2003/12/03 13:29:27  schumann
* added MSH2004 Ball Locator
*
* Revision 1.4  2003/12/02 13:52:48  goehring
* PerceptBallLocator added
*
* Revision 1.3  2003/10/22 10:20:37  goehring
* VoteBasedBallLocator Added
*
* Revision 1.2  2003/10/15 13:37:44  goehring
* ATH2004BallLocator added
*
* Revision 1.1  2003/10/06 13:33:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/25 10:10:31  juengel
* Removed VoteBasedBallLocator.
*
* Revision 1.2  2003/07/05 12:00:10  kudlacik
* Added BallChallengeBallLocator
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/06/24 10:11:31  jhoffman
* added new ball locator
*
* Revision 1.7  2003/06/21 11:59:10  roefer
* SampleBallLocator added, does not work!
*
* Revision 1.6  2003/05/01 17:09:06  loetzsch
* Redesign of ball modeling:
* - Modularized class BallP_osition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.5  2003/03/05 10:34:05  jhoffman
* no message
*
* Revision 1.4  2003/02/19 16:22:12  engel
* SensorFusionBallLocator added
*
* Revision 1.3  2003/01/30 13:16:48  loetzsch
* Redesign of class BallP_osition
*
* Revision 1.2  2002/09/12 09:45:58  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.6  2002/06/17 18:36:32  kspiess
* sample based multi ball locator added
*
* Revision 1.5  2002/06/13 22:11:16  roefer
* StickyBallLocator added
*
* Revision 1.4  2002/06/06 10:51:47  kspiess
* distribution for multiple balls;
*
* Revision 1.3  2002/06/03 15:27:30  roefer
* BallLocator gets TeamMessageCollection
*
* Revision 1.2  2002/05/14 12:39:04  dueffert
* corrected some documentation mistakes
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/05/03 11:51:26  juengel
* JumpingBallLocator created.
*
* Revision 1.3  2002/04/02 13:10:18  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.2  2002/02/11 11:13:05  roefer
* BallPerceptor and BallLocator inserted
*
* Revision 1.1  2002/02/05 03:50:15  loetzsch
* added a few new module selectors
*
*/
