/**
* @file BallLocator.h
* 
* This file contains a generic class for Ball-Localization.
*/

#ifndef __BallLocator_h_
#define __BallLocator_h_

#include "Tools/Module/Module.h"

#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Perception/CalibrationRequest.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Motion/OdometryData.h"

/**
* @class BallLocatorInterfaces
* 
* The interfaces of the BallLocator module.
*/
class BallLocatorInterfaces
{
public:
  /** Constructor.*/
  BallLocatorInterfaces(
    const OdometryData& odometryData,
    const CameraMatrix& cameraMatrix,
    const BallPercept& ballPercept,
    const LandmarksPercept& landmarksPercept,
    const RobotPose& robotPose,
    const CalibrationRequest& calibrationRequest,
    const SensorDataBuffer& sensorDataBuffer,
    BallModel& ballModel)
    : 
  odometryData(odometryData),
    cameraMatrix(cameraMatrix),
    ballPercept(ballPercept),
    landmarksPercept(landmarksPercept),
    robotPose(robotPose),
    calibrationRequest(calibrationRequest),
    sensorDataBuffer(sensorDataBuffer),
    ballModel(ballModel)
  {}
  
  /** The odometry data provided by the motion modules */
  const OdometryData& odometryData;
  
  /** The offset and the rotation of the camera */
  const CameraMatrix& cameraMatrix;
  
  /** The ball percept that was generated last */
  const BallPercept& ballPercept;
  
  /** The landmarkspercept that was generated last */
  const LandmarksPercept& landmarksPercept;

  /** The robots position that was calculated last */
  const RobotPose& robotPose;
  
  /** The calibration request */
  const CalibrationRequest& calibrationRequest;

  /** The sensor data buffer*/
  const SensorDataBuffer& sensorDataBuffer;

  /** The ball position to be changed */
  BallModel& ballModel;
};

/**
* @class BallLocator 
*
* The Base class for ball locators
*
* It is the task of ball modeling to keep track of the location of 
* the ball even if the ball is currently not seen.
*/
class BallLocator : public Module, public BallLocatorInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the BallLocator module.
  */
  BallLocator(const BallLocatorInterfaces& interfaces)
    : BallLocatorInterfaces(interfaces)
  {}
  
  /** Destructor */
  virtual ~BallLocator() {}
};

#endif //__BallLocator_h_

/*
* Change log :
* 
* $Log: BallLocator.h,v $
* Revision 1.3  2004/05/27 09:53:33  loetzsch
* removed "timeOfImageProcessing"
*
* Revision 1.2  2004/05/22 22:52:01  juengel
* Renamed ballP_osition to ballModel.
*
* Revision 1.1.1.1  2004/05/22 17:15:16  cvsadm
* created new repository GT2004_WM
*
* Revision 1.8  2004/04/09 11:35:51  roefer
* Bremen Byters German Open check-in
*
* Revision 1.7  2004/04/05 17:56:45  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.2  2004/04/02 10:03:15  jumped
* added landmarkspercept to balllocator interface
*
* Revision 1.1.1.1  2004/03/31 11:16:37  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.6  2004/03/21 12:39:32  juengel
* Added CalibrationRequest to BallLocatorInterfaces.
*
* Revision 1.5  2004/03/08 00:58:35  roefer
* Interfaces should be const
*
* Revision 1.4  2004/02/03 13:19:48  spranger
* renamed all references to  class BallP_osition to BallModel
*
* Revision 1.3  2004/01/20 23:09:49  loetzsch
* bug fix
*
* Revision 1.2  2004/01/20 16:45:52  loetzsch
* The BallLocator has access to the complete BallP_osition again
*
* Revision 1.1  2003/10/06 13:33:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:48  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/05/01 17:09:06  loetzsch
* Redesign of ball modeling:
* - Modularized class BallP_osition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.4  2003/03/19 15:41:06  jhoffman
* added proper ball speed
*
* Revision 1.3  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.2  2002/09/12 09:45:58  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/08/22 14:41:04  risler
* added some doxygen comments
*
* Revision 1.2  2002/06/03 15:27:30  roefer
* BallLocator gets TeamMessageCollection
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.4  2002/04/02 13:10:18  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.3  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.2  2001/12/10 17:47:06  risler
* change log added
*
*/
