/**
* @file MakeStickDialog.cpp
* The dialog for the MakeStick tool. Includes all program code.
*
* @author Tim Riemenschneider
* @author Thomas Roefer
* @author Tim Laue
*/

#include "stdafx.h"
#include "MakeStick.h"
#include "MakeStickDlg.h"
#include <fstream>
#include <Shlwapi.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

using namespace std;


const char* configTypeNames[NumberOfConfigTypes] = 
{
  "GermanTeam 2003 (configure)", 
  "GermanTeam 2004 (configure)", 
  "GermanTeam current (ERS210)", 
  "GermanTeam current (ERS7)",
  "GermanTeam 2003 (from ZIP)",
  "GermanTeam 2004 (from ZIP)",
  "CMPack (configure)",
  "CMPack (from ZIP)"
};


const char* gt2003Roles[4] = 
{"goalie","defender","striker1","striker2"};


CMakeStickDlg::CMakeStickDlg(CWnd* pParent /*=NULL*/)
	           : CDialog(CMakeStickDlg::IDD, pParent)
{
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
  m_role = AfxGetApp()->GetProfileInt("Defaults", "ROLE", 0);
  m_color = AfxGetApp()->GetProfileInt("Defaults", "COLOR", 0);
  m_sndhalf = AfxGetApp()->GetProfileInt("Defaults", "SNDHALF", FALSE);
  m_ip4base = AfxGetApp()->GetProfileInt("Defaults", "IP4BASE", 0);
  m_ip1 = AfxGetApp()->GetProfileInt("Defaults", "IP1", 172);
  m_ip2 = AfxGetApp()->GetProfileInt("Defaults", "IP2", 21);
  m_ip3 = AfxGetApp()->GetProfileInt("Defaults", "IP3", 3);
  m_ip4 = m_ip4base + 1 + (4*m_color + m_role + m_sndhalf * 4)%8;
  m_name.Format("AIBO%d", m_ip4);
  m_netmask = AfxGetApp()->GetProfileString("Defaults", "Netmask", "255.255.0.0");
  m_essid = AfxGetApp()->GetProfileString("Defaults", "ESSID", "Uni-Bremen");
  m_stickdrive = AfxGetApp()->GetProfileString("Defaults", "STICKDRIVE", "E");
  m_wepkey = AfxGetApp()->GetProfileString("Defaults", "WEPKEY", "");
  m_apmode = AfxGetApp()->GetProfileInt("Defaults", "APMODE", 2);
  m_channel = AfxGetApp()->GetProfileInt("Defaults", "CHANNEL", 3);
  m_force = AfxGetApp()->GetProfileInt("Defaults", "FORCE", 0);
  m_close = AfxGetApp()->GetProfileInt("Defaults", "CLOSE", FALSE);
  m_configType = (ConfigType) AfxGetApp()->GetProfileInt("Defaults", "CONFIGTYPE", GERMAN_TEAM_2004_CONFIGURE);
  m_teamName = AfxGetApp()->GetProfileString("Defaults", "TEAMNAME", "Unknown");
  m_sLocation = AfxGetApp()->GetProfileString("Defaults", "LOCATION", "Bremen");
  m_zipFileName = AfxGetApp()->GetProfileString("Defaults", "ZIPFILE", "(none)");
  m_console = AfxGetApp()->GetProfileInt("Defaults", "CONSOLE", FALSE);
  m_unsafe = AfxGetApp()->GetProfileInt("Defaults", "UNSAFE", FALSE);
}


CMakeStickDlg::~CMakeStickDlg()
{
  AfxGetApp()->WriteProfileInt("Defaults", "IP1", m_ip1);
  AfxGetApp()->WriteProfileInt("Defaults", "IP2", m_ip2);
  AfxGetApp()->WriteProfileInt("Defaults", "IP3", m_ip3);
  AfxGetApp()->WriteProfileInt("Defaults", "IP4BASE", m_ip4base);
  AfxGetApp()->WriteProfileString("Defaults", "Netmask", m_netmask);
  AfxGetApp()->WriteProfileString("Defaults", "ESSID", m_essid);
  AfxGetApp()->WriteProfileString("Defaults", "STICKDRIVE", m_stickdrive);
  AfxGetApp()->WriteProfileString("Defaults", "WEPKEY", m_wepkey);
  AfxGetApp()->WriteProfileInt("Defaults", "APMODE", m_apmode);
  AfxGetApp()->WriteProfileInt("Defaults", "CHANNEL", m_channel);
  AfxGetApp()->WriteProfileInt("Defaults", "SNDHALF", m_sndhalf);
  AfxGetApp()->WriteProfileInt("Defaults", "FORCE", m_force);
  AfxGetApp()->WriteProfileInt("Defaults", "CLOSE", m_close);
  AfxGetApp()->WriteProfileInt("Defaults", "ROLE", m_role);
  AfxGetApp()->WriteProfileInt("Defaults", "COLOR", m_color);
  AfxGetApp()->WriteProfileInt("Defaults", "CONFIGTYPE", m_configType);
  AfxGetApp()->WriteProfileString("Defaults", "TEAMNAME", m_teamName);
  AfxGetApp()->WriteProfileString("Defaults", "LOCATION", m_sLocation);
  AfxGetApp()->WriteProfileString("Defaults", "ZIPFILE", m_zipFileName);
  AfxGetApp()->WriteProfileInt("Defaults", "CONSOLE", m_console);
  AfxGetApp()->WriteProfileInt("Defaults", "UNSAFE", m_unsafe);
}


void CMakeStickDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CMakeStickDlg)
	DDX_Control(pDX, IDC_CONFIGTYPE, m_configtype);
	DDX_Radio(pDX, IDC_GOALIE, m_role);
	DDX_Radio(pDX, IDC_RED, m_color);
	DDX_Text(pDX, IDC_IP1, m_ip1);
	DDV_MinMaxInt(pDX, m_ip1, 0, 255);
	DDX_Text(pDX, IDC_IP2, m_ip2);
	DDV_MinMaxInt(pDX, m_ip2, 0, 255);
	DDX_Text(pDX, IDC_IP3, m_ip3);
	DDV_MinMaxInt(pDX, m_ip3, 0, 255);
	DDX_Text(pDX, IDC_IP4, m_ip4);
	DDX_Text(pDX, IDC_NAME, m_name);
	DDX_Text(pDX, IDC_NETMASK, m_netmask);
	DDX_Text(pDX, IDC_ESSID, m_essid);
	DDX_Text(pDX, IDC_IP4BASE, m_ip4base);
	DDV_MinMaxInt(pDX, m_ip4base, 0, 245);
	DDX_Text(pDX, IDC_STICKDRIVE, m_stickdrive);
	DDV_MaxChars(pDX, m_stickdrive, 1);
	DDX_Check(pDX, IDC_SNDHALF, m_sndhalf);
	DDX_Text(pDX, IDC_WEPKEY, m_wepkey);
	DDX_Text(pDX, IDC_APMODE, m_apmode);
	DDV_MinMaxInt(pDX, m_apmode, 0, 2);
	DDX_Text(pDX, IDC_CHANNEL, m_channel);
	DDV_MinMaxInt(pDX, m_channel, 0, 13);
	DDX_Radio(pDX, IDC_COPY, m_force);
	DDX_Check(pDX, IDC_CLOSE, m_close);
	DDX_Text(pDX, IDC_TEAMNAME, m_teamName);
  DDV_MaxChars(pDX, m_teamName, 15);
  DDX_Text(pDX, IDC_ZIPNAME, m_zipFileName);
  DDV_MaxChars(pDX, m_zipFileName, 200);
	DDX_Control(pDX, IDC_LOCATION, m_location);
	DDX_Text(pDX, IDC_LOCATION, m_sLocation);
	DDX_Check(pDX, IDC_CONSOLE, m_console);
	DDX_Check(pDX, IDC_UNSAFE, m_unsafe);
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CMakeStickDlg, CDialog)
	//{{AFX_MSG_MAP(CMakeStickDlg)
	ON_WM_SYSCOMMAND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_BN_CLICKED(IDC_WRITESTICK, OnWriteStick)
	ON_BN_CLICKED(IDC_BROWSE, OnBrowse)
	ON_BN_CLICKED(IDC_ERROR, OnError)
	ON_BN_CLICKED(IDC_GOALIE, OnRoleChange)
	ON_BN_CLICKED(IDC_DEFENDER, OnRoleChange)
	ON_BN_CLICKED(IDC_STRIKER1, OnRoleChange)
	ON_BN_CLICKED(IDC_STRIKER2, OnRoleChange)
	ON_BN_CLICKED(IDC_BLUE, OnRoleChange)
	ON_BN_CLICKED(IDC_RED, OnRoleChange)
	ON_EN_CHANGE(IDC_IP4BASE, OnRoleChange)
	ON_BN_CLICKED(IDC_SNDHALF, OnRoleChange)
	ON_BN_CLICKED(IDC_COPY, OnSelchangeConfigtype)
	ON_BN_CLICKED(IDC_FORCE, OnSelchangeConfigtype)
	ON_BN_CLICKED(IDC_UPDATE, OnSelchangeConfigtype)
	ON_BN_CLICKED(IDC_CLOSE, OnRoleChange)
  ON_BN_CLICKED(IDC_BUTTONLOAD, OnLoadButton)
	ON_CBN_SELCHANGE(IDC_CONFIGTYPE, OnSelchangeConfigtype)
	ON_CBN_SELCHANGE(IDC_LOCATION, OnRoleChange)
	ON_BN_CLICKED(IDC_CONSOLE, OnRoleChange)
	ON_BN_CLICKED(IDC_UNSAFE, OnRoleChange)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CMakeStickDlg::OnInitDialog()
{
  CInstanceChecker& instanceChecker = ((CMakeStickApp*) AfxGetApp())->instanceChecker;
  if (!instanceChecker.PreviousInstanceRunning())
    instanceChecker.TrackFirstInstanceRunning();
  // Init dialog and set icon
	CDialog::OnInitDialog();
	SetIcon(m_hIcon, TRUE);			
	SetIcon(m_hIcon, FALSE);		
	
  // Set possible configurations
  m_configtype.ResetContent();
  m_configtype.AddString(configTypeNames[GERMAN_TEAM_2003_CONFIGURE]);
  m_configtype.AddString(configTypeNames[GERMAN_TEAM_2004_CONFIGURE]);
  m_configtype.AddString(configTypeNames[GERMAN_TEAM_COPY_ERS210]);
  m_configtype.AddString(configTypeNames[GERMAN_TEAM_COPY_ERS7]);
  m_configtype.AddString(configTypeNames[GERMAN_TEAM_2003_ZIP]);
  m_configtype.AddString(configTypeNames[GERMAN_TEAM_2004_ZIP]);
  m_configtype.AddString(configTypeNames[CMU]);
  m_configtype.AddString(configTypeNames[CMU_ZIP]);
  m_configtype.SetCurSel(m_configType);

  // Try to set current location
  m_location.ResetContent();
  WIN32_FIND_DATA findFileData;
  HANDLE hFind = FindFirstFile("..\\Config\\Location\\*", &findFileData);
  int selected = 0;
  while(hFind != INVALID_HANDLE_VALUE)
  {
    if(findFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY && 
       findFileData.cFileName[0] != '.' &&
       strcmp(findFileData.cFileName, "CVS"))
    {
      if(m_sLocation == findFileData.cFileName)
        selected = m_location.GetCount();
      m_location.AddString(findFileData.cFileName);
    }
    if(!FindNextFile(hFind, &findFileData))
      break;
  }
  FindClose(hFind);
  m_location.SetCurSel(selected);

  OnSelchangeConfigtype();
  OnRoleChange();
  CEdit* zipNameField = (CEdit*)GetDlgItem(IDC_ZIPNAME);
  zipNameField->SetWindowText(m_zipFileName);
  int endOfFileName(zipNameField->LineLength());
  zipNameField->SetSel(endOfFileName,endOfFileName);
  zipNameField->SetLimitText(200);

	return TRUE;
}


void CMakeStickDlg::OnPaint() 
{
	if (IsIconic())
	{
		CPaintDC dc(this); 
		SendMessage(WM_ICONERASEBKGND, (WPARAM) dc.GetSafeHdc(), 0);
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialog::OnPaint();
	}
}


HCURSOR CMakeStickDlg::OnQueryDragIcon()
{
	return (HCURSOR) m_hIcon;
}


void CMakeStickDlg::OnWriteStick() 
{
  bool result(true),zipResult(true);
  if(!UpdateData()) return;
  switch(m_configType) 
  {
    case GERMAN_TEAM_COPY_ERS210:
    case GERMAN_TEAM_COPY_ERS7:
      {
        ofstream lcfg("..\\Config\\Location.cfg");
        lcfg << (const char*) m_sLocation << endl;
        lcfg.close();
        char command[100];
        sprintf(command, "bash ../Make/copyfiles.bash -m %c:%s %s %s %s", m_stickdrive[0], 
                          m_force == 1 ? " --force" : m_force == 2 ? " --update" : "",
                          m_console ? "--wconsole" : "",
                          m_unsafe ? "--unsafe" : "",
                          m_configType == GERMAN_TEAM_COPY_ERS210 ? "ERS210" : "ERS7");
        system(command);
      }
    case GERMAN_TEAM_2003_CONFIGURE:
    case GERMAN_TEAM_2004_CONFIGURE:
      result = WriteConfigGT();
      break;
    case GERMAN_TEAM_2003_ZIP:
    case GERMAN_TEAM_2004_ZIP:
      zipResult = UnzipStick();
      result = WriteConfigGT();
      break;
    case CMU_ZIP:
      zipResult = UnzipStick();
    case CMU:
      result = WriteConfigCMU();
  }
  if(!zipResult)
  {
    MessageBox("The ZIP archive does not exist!", "Error");
    return;
  }
  if(!result) 
  {
    MessageBox("The configuration could not be written.\nMaybe, there was no Memorystick in the drive?", "Error");
  } 
  else 
  {
    if(m_close)
      PostMessage(WM_CLOSE);
    else
      MessageBox("The configuration was successfully written.");
  }
}


void CMakeStickDlg::OnBrowse()
{
  char command[20];
  sprintf(command, "explorer.exe %c:\\", m_stickdrive[0]);
  WinExec(command, SW_SHOWNORMAL);
  if(m_close)
    PostMessage(WM_CLOSE);
}


void CMakeStickDlg::OnError()
{
  char command[40];
  sprintf(command, "bash ./showError.bash %c:", m_stickdrive[0]);
  WinExec(command, SW_SHOWNORMAL);
  if(m_close)
    PostMessage(WM_CLOSE);
}


void CMakeStickDlg::OnRoleChange() 
{
  UpdateData();
  m_ip4 = m_ip4base + 1 + (4*m_color + m_role + m_sndhalf * 4)%8;
  m_name.Format("AIBO%d", m_ip4);
  UpdateData(FALSE);
}


void CMakeStickDlg::OnSelchangeConfigtype() 
{
  if(!UpdateData()) return;

  bool enable = false;
  switch(m_configType = (ConfigType) m_configtype.GetCurSel()) {
    case GERMAN_TEAM_2003_ZIP:
      GetDlgItem(IDC_ZIPNAME)->EnableWindow(TRUE);
      GetDlgItem(IDC_BUTTONLOAD)->EnableWindow(TRUE);
      GetDlgItem(IDC_GOALIE)->SetWindowText("Goalie (1)");
      GetDlgItem(IDC_DEFENDER)->SetWindowText("Defender (2)");
      GetDlgItem(IDC_STRIKER1)->SetWindowText("Striker 1 (3)");
      GetDlgItem(IDC_STRIKER2)->SetWindowText("Striker 2 (4)");
      GetDlgItem(IDC_TEAMNAME)->EnableWindow(FALSE);
      break;
    case GERMAN_TEAM_2004_ZIP:
      GetDlgItem(IDC_ZIPNAME)->EnableWindow(TRUE);
      GetDlgItem(IDC_BUTTONLOAD)->EnableWindow(TRUE);
      GetDlgItem(IDC_GOALIE)->SetWindowText("Goalie (1)");
      GetDlgItem(IDC_DEFENDER)->SetWindowText("Player (2)");
      GetDlgItem(IDC_STRIKER1)->SetWindowText("Player (3)");
      GetDlgItem(IDC_STRIKER2)->SetWindowText("Player (4)");
      GetDlgItem(IDC_TEAMNAME)->EnableWindow(TRUE);
      break;
    case GERMAN_TEAM_COPY_ERS210:
    case GERMAN_TEAM_COPY_ERS7:
      enable = true;
    case GERMAN_TEAM_2004_CONFIGURE:
      GetDlgItem(IDC_GOALIE)->SetWindowText("Goalie (1)");
      GetDlgItem(IDC_DEFENDER)->SetWindowText("Player (2)");
      GetDlgItem(IDC_STRIKER1)->SetWindowText("Player (3)");
      GetDlgItem(IDC_STRIKER2)->SetWindowText("Player (4)");
      GetDlgItem(IDC_TEAMNAME)->EnableWindow(TRUE);
      GetDlgItem(IDC_ZIPNAME)->EnableWindow(FALSE);
      GetDlgItem(IDC_BUTTONLOAD)->EnableWindow(FALSE);
      break;
    case GERMAN_TEAM_2003_CONFIGURE:
      GetDlgItem(IDC_ZIPNAME)->EnableWindow(FALSE);
      GetDlgItem(IDC_BUTTONLOAD)->EnableWindow(FALSE);
      GetDlgItem(IDC_GOALIE)->SetWindowText("Goalie (1)");
      GetDlgItem(IDC_DEFENDER)->SetWindowText("Defender (2)");
      GetDlgItem(IDC_STRIKER1)->SetWindowText("Striker 1 (3)");
      GetDlgItem(IDC_STRIKER2)->SetWindowText("Striker 2 (4)");
      GetDlgItem(IDC_TEAMNAME)->EnableWindow(FALSE);
      break;
    case CMU_ZIP:
      GetDlgItem(IDC_ZIPNAME)->EnableWindow(TRUE);
      GetDlgItem(IDC_BUTTONLOAD)->EnableWindow(TRUE);
      GetDlgItem(IDC_GOALIE)->SetWindowText("Goalie (1)");
      GetDlgItem(IDC_DEFENDER)->SetWindowText("Attacker (2)");
      GetDlgItem(IDC_STRIKER1)->SetWindowText("Attacker (3)");
      GetDlgItem(IDC_STRIKER2)->SetWindowText("Attacker (4)");
      GetDlgItem(IDC_TEAMNAME)->EnableWindow(FALSE);
      break;
    case CMU:
      GetDlgItem(IDC_ZIPNAME)->EnableWindow(FALSE);
      GetDlgItem(IDC_BUTTONLOAD)->EnableWindow(FALSE);
      GetDlgItem(IDC_GOALIE)->SetWindowText("Goalie (1)");
      GetDlgItem(IDC_DEFENDER)->SetWindowText("Attacker (2)");
      GetDlgItem(IDC_STRIKER1)->SetWindowText("Attacker (3)");
      GetDlgItem(IDC_STRIKER2)->SetWindowText("Attacker (4)");
      GetDlgItem(IDC_TEAMNAME)->EnableWindow(FALSE);
      break;
  }
  GetDlgItem(IDC_COPY)->EnableWindow(enable);
  GetDlgItem(IDC_FORCE)->EnableWindow(enable);
  GetDlgItem(IDC_UPDATE)->EnableWindow(enable);
  GetDlgItem(IDC_LOCATION)->EnableWindow(enable);
  GetDlgItem(IDC_CONSOLE)->EnableWindow(enable && m_force <= 1);
  GetDlgItem(IDC_UNSAFE)->EnableWindow(enable && m_force <= 1);
  Invalidate();
}


void CMakeStickDlg::OnLoadButton() 
{
	CString pathName = 
    AfxGetApp()->GetProfileString("MakeStick load ZIP file", "openPath", "");
  pathName.Insert(10000, "*.zip");
  
  CFileDialog fileDialog(true, ".zip",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "ZIP files (*.zip)|*.zip||", this);
  
  if (fileDialog.DoModal()==IDOK)
  {
    m_zipFileName = fileDialog.GetPathName();
    GetDlgItem(IDC_ZIPNAME)->SetWindowText(fileDialog.GetPathName());
    int endOfFileName(((CEdit*)GetDlgItem(IDC_ZIPNAME))->LineLength());
    ((CEdit*)GetDlgItem(IDC_ZIPNAME))->SetSel(endOfFileName,endOfFileName);
    InvalidateRect(NULL,FALSE);
  }
}


bool CMakeStickDlg::WriteConfigGT()
{
  char colors[2][5] = {"red", "blue"};
  char filename[100];
  sprintf(filename, "%c:\\open-r\\APP\\CONF\\player.cfg", m_stickdrive[0]);
  ofstream pcfg(filename);
  pcfg << "teamColor " << colors[m_color] << endl;
  if((m_configType == GERMAN_TEAM_2003_CONFIGURE) || (m_configType == GERMAN_TEAM_2003_ZIP))
  {
    pcfg << "playerRole " << gt2003Roles[m_role] << endl;
  }
  else //GT2004 code
  {
    pcfg << "playerNumber " << m_role + 1 << endl;
    pcfg << "teamIdentifier " << (const char*) m_teamName << endl;
  }
  pcfg.close();

  sprintf(filename, "%c:\\open-r\\SYSTEM\\CONF\\wlanconf.txt", m_stickdrive[0]);
  ofstream wcfg(filename);
  wcfg << "HOSTNAME=" << (const char*) m_name << endl;
  wcfg << "ETHER_IP=" << m_ip1 << "." << m_ip2 << "." << m_ip3 << "." << m_ip4 << endl;
  wcfg << "ETHER_NETMASK=" << (const char*) m_netmask << endl;
  wcfg << "IP_GATEWAY=" << m_ip1 << "." << m_ip2 << "." << m_ip3 << "." << m_ip4 << endl;
  wcfg << "ESSID=" << (const char*) m_essid << endl;
  wcfg << "WEPENABLE=" << (m_wepkey != "") << endl;
  wcfg << "WEPKEY=" << (const char*) m_wepkey << endl;
  wcfg << "APMODE=" << m_apmode << endl;
  wcfg << "CHANNEL=" << m_channel << endl;
  wcfg.close();

  return (pcfg.good() && wcfg.good());
}


bool CMakeStickDlg::WriteConfigCMU()
{
  char rolenames[4][13] = {"goalie", "new_attack", "new_attack", "new_attack"};
  char teamcolors[2][5] = {"red", "blue"};
  char goalcolors[2][7] = {"yellow", "cyan"};
  char filename[100];

  sprintf(filename, "%c:\\config\\behave.cfg", m_stickdrive[0]);
  ofstream pcfg(filename);
  
  pcfg << "defendGoal=\"" << goalcolors[m_color] << 
    "\"; // goal we are defending (\"yellow\",\"blue\", or \"none\")" <<endl;
  pcfg << "teamColor=\"" << teamcolors[m_color] << 
    "\"; // color of our team (\"red\", \"blue\")" << endl;
  pcfg << "fieldSide=\"pos\";  // side of field for attacker - (\"pos\" or \"neg\")" << endl;
  pcfg << "role=\"" << rolenames[m_role] << "\";" << endl;
  pcfg.close();

  if(!pcfg.good()) return false;

  sprintf(filename, "%c:\\config\\wavelan.cfg", m_stickdrive[0]);
  pcfg.open(filename);
  pcfg << "SenderID = " << m_role + 1 << ";" << endl;
  pcfg << "OutputOn = 0;" << endl;
  pcfg.close();

  sprintf(filename, "%c:\\open-r\\SYSTEM\\CONF\\wlanconf.txt", m_stickdrive[0]);
  ofstream wcfg(filename);
  wcfg << "HOSTNAME=" << (const char*) m_name << endl;
  wcfg << "ETHER_IP=" << m_ip1 << "." << m_ip2 << "." << m_ip3 << "." << m_ip4 << endl;
  wcfg << "ETHER_NETMASK=" << (const char*) m_netmask << endl;
  wcfg << "IP_GATEWAY=" << m_ip1 << "." << m_ip2 << "." << m_ip3 << "." << m_ip4 << endl;
  wcfg << "ESSID=" << (const char*) m_essid << endl;
  wcfg << "WEPENABLE=" << (m_wepkey != "") << endl;
  wcfg << "WEPKEY=" << (const char*) m_wepkey << endl;
  wcfg << "APMODE=" << m_apmode << endl;
  wcfg << "CHANNEL=" << m_channel << endl;

  wcfg.close();
  return (wcfg.good() && pcfg.good());

}


bool CMakeStickDlg::UnzipStick()
{
  ifstream zipTest;
  zipTest.open(m_zipFileName);
  if(zipTest)
  {
    zipTest.close();
  }
  else
  {
    return false;
  }
  char formatCommand[100];
  sprintf(formatCommand, "cmd /C format %c: /Q /V: /X /y", m_stickdrive[0]);
  system(formatCommand);
  char unzipCommand[300];
  sprintf(unzipCommand, "cmd /C unzip \"%s\" -d %c:",(LPCSTR)m_zipFileName,m_stickdrive[0]);
  system(unzipCommand);
  return true;
}


/*
 * Changelog:
 *
 * $Log: MakeStickDlg.cpp,v $
 * Revision 1.3  2004/06/17 09:19:53  roefer
 * MakeStick starts only a single instance
 *
 * Revision 1.2  2004/05/27 13:52:13  roefer
 * Adaptation to new location handling
 *
 * Revision 1.1.1.1  2004/05/22 17:15:15  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.11  2004/03/23 10:28:06  roefer
 * Minor bug fixed
 *
 * Revision 1.10  2004/03/22 21:55:41  roefer
 * New options
 *
 * Revision 1.9  2004/02/20 13:59:14  tim
 * Compatibility for GT2003 roles
 *
 * Revision 1.8  2004/02/19 20:08:04  tim
 * fixed problem with whitespaces in paths
 *
 * Revision 1.7  2004/02/19 17:16:14  tim
 * Added support for zipped sticks (GT as well as CMU)
 *
 * Revision 1.6  2004/02/13 13:53:45  roefer
 * Mistakes corrected
 *
 * Revision 1.5  2004/01/30 20:56:04  roefer
 * Added writing of location.cfg
 *
 * Revision 1.4  2004/01/30 14:24:46  roefer
 * teamName -> teamIdentifier
 *
 * Revision 1.3  2004/01/27 10:23:20  roefer
 * Support for team name
 *
 * Revision 1.2  2004/01/21 13:50:36  roefer
 * Strange fstream bug fixed
 *
 * Revision 1.1  2004/01/17 14:13:12  roefer
 * MakeStick inserted into project
 *
 * Revision 1.4  2003/06/23 21:55:31  roefer
 * IP4 initialized
 *
 * Revision 1.3  2003/06/10 10:57:35  timrie
 * added ConfigType (GT, GT-copyfiles, CMU)
 * added support for CMU-config files
 * added WEP-Settings
 * added AP-Mode + Channel
 *
 * Revision 1.2  2003/06/02 09:54:41  timrie
 * added letter of memorystick-drive
 *
 * Revision 1.1  2003/05/30 13:10:20  timrie
 * new file
 *
 *
 */
