#!/bin/bash

# treat unset variables as errors
#
set -u

# bash debugging output
#
#set -x


# Generate a MemoryStick Image
#
MakeStickImage() {
  local gt="$1"    # GT2004 root directory
  local sdk="$2"   # SDK MemoryStick root
  local img="$3"   # Target directory

  echo "generating stick image in $img.. "
  
  echo "  copying system files.."
    cp -r $sdk/open-r $img
    chmod -R u+w $img
    
  
    if [ $OPT_ERS7 != 0 ]; then
      echo "  JamDetectionHighThreshold"
      echo "JamDetectionHighThreshold" > $img/open-r/system/conf/VRCOMM.CFG
    fi

  echo "  copying GT2004 main files.."
    cp -r $gt/Build/MS/open-r/APP  $img/open-r

  echo "  installing tinyftpd.."
#    cp $gt/Util/TinyFTPD/bin/TINYFTPD.BIN  $img/open-r/mw/objs/tinyftpd.bin
#    echo "/ms/open-r/mw/objs/tinyftpd.bin" >> $img/open-r/mw/conf/object.cfg
    cp $gt/Util/TinyFTPD/bin/TINYFTPD.BIN $img/open-r/app/objs/tinyftpd.bin

    local pw=$(grep "^ftpPassword" $gt/Config/player.cfg | cut -f2- -d" ")

    # default to anonymous ftp
    #
    pw=${pw:-anonymous *}

    echo "    creating account for \"$pw\""
    echo "$pw /MS/" > $img/open-r/app/conf/PASSWD

  echo "  copying xabsl2 files.."
    mkdir -p $img/open-r/app/conf/xabsl2
    cp $gt/config/xabsl2/*  $img/open-r/app/conf/xabsl2/

  echo "  copying sound files.."
    mkdir -p $img/open-r/app/conf/sounds
    cp $gt/config/sounds/*.wav  $img/open-r/app/conf/sounds

  echo "  copying common config files.."
    cp $gt/config/empty_debug_request.dat  $img/open-r/app/conf/requests.dat
    cp $gt/config/spec_act.dat             $img/open-r/app/conf/spec_act.dat
    cp $gt/config/nc_pop1.dat              $img/open-r/app/conf/nc_pop1.dat
    cp $gt/config/wlanconf.txt             $img/open-r/system/conf
    cp $gt/config/*.tab                    $img/open-r/app/conf
    cp $gt/config/*.cfg                    $img/open-r/app/conf
    mv $img/open-r/app/conf/robotgw.cfg    $img/open-r/mw/conf

    # don't forget bremen's potential fields..
    #
    mkdir -p $img/open-r/app/conf/pfield/bremen
    cp $gt/config/pfield/bremen/*.pfc      $img/open-r/app/conf/pfield/bremen

  echo "  copying location dependend files.."
    # TODO: 
    #   Use more parser friendly "#" for comments!
    #
    local loc=$(grep -v "\*" $gt/config/location.cfg | grep "[A-Za-z0-9]")
    if [ $loc ]; then
      echo "    taking files from config/$loc"
      mkdir -p $img/open-r/app/conf/$loc
      cp   $gt/config/$loc/*.*   $img/open-r/app/conf/$loc
    else
      echo "    no location set in location.cfg"
    fi

  local files=$(find $img -type f | wc -l | tr -d " ")
  local dirs=$(find $img -type d | wc -l | tr -d " ")
  local bytes=$(du -hs $img | cut -f1)    
  echo  "  ${bytes}Bytes in $files files and $dirs directories."
}


UpdateStick() {
  local src=$1    # source directory
  local dst=$2    # destination URL

  local src_md5sums=$(mktemp)
  local dst_md5sums=$(mktemp)
  local src_files=$(mktemp)
  local dst_files=$(mktemp)

  echo "updating $dst.."

  # Get a listing of the current directory contents, and 
  # md5-checksums for each file if available. Note that all
  # file listings are kept in sorted order, so we can use
  # diff to compare the listings against each other easily.
  # 
  if [ $OPT_CLEAN != 0 ]; then
    echo "  removing old files.."
    lftp -c "open $dst; rm -rf open-r"
  else
    if [ $OPT_QUICK == 0 ]; then
      echo "  getting remote file listing.."
      lftp -c "open $dst; find" | grep -v "/$" | sort -f -s >$dst_files
    fi

    if [ $(lftp -c "open $dst; find MD5SUMS") ]; then
      echo "  fetching remote md5sums file.. "
      lftp -c "open $dst; cat md5sums" | sort -f -s -k2 >$dst_md5sums
    fi
  fi

  echo "  generating local md5sums.."
  pushd $src  >/dev/null
  md5sum $(find -type f | sort -f -s | tee $src_files) >$src_md5sums
  popd  >/dev/null

  echo "  calculating update list.."

  # woohoo! bash-programming at it's best ;)
  # 
  local update_list=$(
    (
      (
        # use diff to compare md5 and directory listings
        #
        diff -iw $src_md5sums $dst_md5sums | grep "^<" | cut -f2- -d*
        if [ $OPT_QUICK == 0 ]; then
          diff -iw $src_files   $dst_files   | grep "^<" | cut -f2- -d" "
        fi
      ) | (
        # pipe of exclude filters
        #
        if [ $OPT_EXCLUDE ]; then
          grep -iv "$OPT_EXCLUDE"
        else
          cat
        fi
      ) ; (
        # list of include filters
        #
        if [ $OPT_INCLUDE ]; then
          grep -i "$OPT_INCLUDE" $src_files
        fi
      )
    ) | sort -f | uniq -i
  )

  local update_num=$(echo "$update_list" | grep -c "^\.")
  echo "  $update_num file(s) modified or new.."

  if [ $update_num != 0 ]; then
    local files=$(echo "$update_list" | tr "\r\n" " ")

    pushd $src  >/dev/null
    lftp -c "open $dst; mput -d $files"

    echo "  updating remote md5sums file."

    # Make a grand unified md5sums file
    # (only the updated md5sums are replaced!)
    (
      md5sum $files
      cat    $dst_md5sums
    ) | sort -f -s -k2 | uniq -i -f 1 >$src_md5sums

    popd  >/dev/null
    lftp -c "open $dst; put $src_md5sums -o md5sums"
  fi

  if [ $OPT_BOOT != 0 ]; then
    echo "  rebooting aibo.."
    lftp -c "open $dst; quote rebt"
  fi

  # clean up temporary files
  #
  rm $src_md5sums $dst_md5sums $src_files $dst_files
}




# try to guess root directory for GT2004 (e.g. c:/GT2004)
#
GT_DIR=$(pwd | sed 's@\(.*\)/Make.*@\1@' | sed 's@/cygdrive/\(.\)\(.*\)@\1:\2@')

OPT_HELP=0
OPT_ERS7=0
OPT_CLEAN=0
OPT_QUICK=0
OPT_BOOT=0
OPT_INCLUDE=""
OPT_EXCLUDE=""

# parse command line options and parameters
#
while getopts "7bchqd:i:x:" OPTION; do
  case "$OPTION" in
    7)  OPT_ERS7=1;;
    b)  OPT_BOOT=1;;
    c)  OPT_CLEAN=1; OPT_ALL=1;;
    h)  OPT_HELP=1;;
    q)  OPT_QUICK=1;;
    d)  GT_DIR=$OPTARG;;
    i)  OPT_INCLUDE=$OPTARG;;
    x)  OPT_EXCLUDE=$OPTARG;;
    ?)  OPT_HELP=1;;
  esac
done

# swallow recognized options
#
shift $(($OPTIND - 1))

if [ $OPT_HELP -eq 1 -o $# -lt 1 ]; then
  # Important: The tab-characters in front of the
  # text must remain intact! Don't change to spaces!
  #
  cat <<-__EOM__
		Usage: copyfiles.bash [OPTION]... <URL>
		Generate a GT2004 MemoryStick and copy it to the given URL.

		  -7              Generate an ERS-7 MemoryStick
		  -b              Reboot after update (ftp mode only of course ;)
		  -c              Clean copy (rm -rf open-r)
		  -q              Quick copy (don't check for missing files)
		  -d <dir>        GT2004 root directory to copy from
		  -i "<regexp>"   Include files (even if they won't need an update)
		  -x "<regexp>"   Exclude files (even if they need an update)
		  -h              Display this help and exit

		Example target URLs:
		  file:///cygdrive/e      MemoryStick reader in drive e:
		  ftp://192.168.0.10      anynoumus ftp to 192.168.0.10
		  ftp://user:pass@aibo2   password protected ftp to aibo2

		Please report bugs and suggestions to thomas.kindler@gmx.de
	__EOM__
  exit  
fi

STICK_URL="$1"

# Set SDK Path
#

if [ $OPT_ERS7 != 0 ]; then
  if test -d /usr/local/OPEN_R_SDK_ERS7; then # 1.1.4 / 1.1.5 dual installation?
    SDKMS_DIR=/usr/local/OPEN_R_SDK_ERS7/OPEN_R/MS_ERS7/WLAN/NOMEMPROT
  else # 1.1.5 installation
    SDKMS_DIR=/usr/local/OPEN_R_SDK/OPEN_R/MS_ERS7/WLAN/NOMEMPROT
  fi
else
  if test -d /usr/local/OPEN_R_SDK_ERS7; then # 1.1.4 / 1.1.5 dual installation?
    SDKMS_DIR=/usr/local/OPEN_R_SDK_ERS210/OPEN_R/MS/WLAN/NOMEMPROT
  else
    if test -d /usr/local/OPEN_R_SDK/OPEN_R/MS_ERS200; then # 1.1.5 installation?
      SDKMS_DIR=/usr/local/OPEN_R_SDK/OPEN_R/MS_ERS200/WLAN/NOMEMPROT
    else # 1.1.4 installation
      SDKMS_DIR=/usr/local/OPEN_R_SDK/OPEN_R/MS/WLAN/NOMEMPROT
    fi
  fi
fi

# Create stick image in a temporary directory
#
IMG_DIR=$(mktemp -dt MSImage.$(date +%y%m%d_%H%M%S).XXX)

MakeStickImage $GT_DIR $SDKMS_DIR $IMG_DIR
echo
UpdateStick    $IMG_DIR $STICK_URL

echo " ... done"

# Change Log:
#
# $Log: copyfiles2.bash,v $
# Revision 1.2  2004/06/30 11:47:32  cesarz
# removed copying of *.kst files in config folder
#
# Revision 1.1.1.1  2004/05/22 17:11:43  cvsadm
# created new repository GT2004_WM
#
# Revision 1.10  2004/04/07 13:00:44  risler
# ddd checkin after go04 - second part
#
# Revision 1.2  2004/03/31 18:46:45  mkunz
# some updates (dirs)
#
# Revision 1.1.1.1  2004/03/29 08:28:41  Administrator
# initial transfer from tamara
#
# Revision 1.9  2004/03/12 12:32:56  kindler
# - removed host specific defaults
#
# Revision 1.8  2004/02/25 13:48:25  kindler
# first real working version..
#
# Revision 1.7  2004/02/18 13:16:38  kindler
# - added -b reboot after copy option
# - removed unneccessary mav.cfg/object2.cfg files.
# - test replaced by [..]
#

