#!/usr/bin/perl -w
######################################################################
#
# Usage:
#
#    emonLogParser.pl EMON_LOG LAYOUT CONFIGURATION [CONTEXTSIZE [ADDR]]
#

use strict;

######################################################################
# forward decls
#
sub run($);
sub checkCaller($);
sub readHex($$);

######################################################################
# main
#
my $sHexRe = "([0-9a-zA-Z]+)";
my $lHexRe = "0x$sHexRe";

if (@ARGV < 3 || @ARGV > 5) {
    print STDERR "usage:\n  $0 EMON_LOG LAYOUT CONFIGURATION [CONTEXTSIZE [ADDR]]\n";
    print STDERR "examples:\n  $0 EMON.LOG CMD Debug\n  $0 e:/emon.log CMD Debug 20\n  $0 /cygdrive/f/open-r/EMON.LOG CMD DebugNoWLan 500 0x43210\n";
    exit(1);
}
my ($emonLogFile, $layout, $configuration, $contextSize, $target) = @ARGV;

open(EMONLOG, $emonLogFile)
    || die "ERROR: can't open $emonLogFile: $!\n";
my $emonLog = join('', <EMONLOG>);
close(EMONLOG);

my $context = readHex($emonLog, "\n context: ");

print "context:\t$context\n";

if ($emonLog !~ /\n(\S+)\s*0x$context $lHexRe $lHexRe /) {
    die "ERROR: can't find the context: $context\n";
}
my $objName = $1;
my $nosnapFile="../Build/GT2004/$layout/$configuration/bin/$objName".".bin.unstripped.nosnap.elf";
print "object:\t\t$objName ($nosnapFile)\n\n";


my $badvaddr= readHex($emonLog, "badvaddr: \\\$8: ");
print "badvaddr(\$8):\t$badvaddr\n";

my $epc = readHex($emonLog, "epc:\\\$14: ");
print "epc(\$14):\t$epc\n";

my $ra= readHex($emonLog, "ra:r31: ");
print "ra(r31):\t$ra\n";

if (defined($target)) {
    $target =~ s/^0x//;
    print "target:\t\t$target  (command line argument)\n";
}
elsif ($epc eq $badvaddr) {
    $target = $ra;
    print "target:\t\t$target  (ra, because epc==badvaddr)\n";
}
else {
    $target = $epc;
    print "target:\t\t$target  (epc, because epc!=badvaddr)\n";
}

my $gpReg = readHex($emonLog, "gp:r28: ");
print "gpReg(r28):\t$gpReg\n";

my $gpGrep = run("/usr/local/OPEN_R_SDK/bin/mipsel-linux-readelf -s $nosnapFile |grep '_gp\$'");
if ($gpGrep !~ /^\s*\S+:\s*$sHexRe\s+.*_gp/) {
    die "INTERNAL ERROR: unexpected \$gpGrep: $gpGrep\n\n";
}
my $gpSize = $1;
print "gpSize:\t\t$1  (from readelf -s $objName".".bin.unstripped.nosnap.elf |grep '_gp\$')\n\n";

my $offset = hex($gpReg) - hex($gpSize);
printf("offset      = gpReg    - gpSize\n");
printf("%08x    = %08s - %08s\n\n",$offset,$gpReg,$gpSize);

my $callerAddr = hex($ra) - $offset;
printf("static caller = ra       - offset\n");
printf("%08x      = %08s - %08x\n\n",$callerAddr,$target,$offset);

my $staticAddr = hex($target) - $offset;
printf("static crasher = target   - offset\n");
printf("%08x       = %08s - %08x\n\n",$staticAddr,$target,$offset);

if (!defined($contextSize))
{
  $contextSize=0x80;
}

my $startAddr = $staticAddr - $contextSize;
my $stopAddr = $staticAddr + $contextSize;

my $startAddr2 = $callerAddr - $contextSize;
my $stopAddr2 = $callerAddr + $contextSize;

my $s=sprintf("echo -e \"\n\n\\\\033[33mNow we create an assembler dump of caller address (0x%x) +- 0x%x,\nthe calling line is marked \\\\033[32mgreen\\\\033[33m:\\\\033[37m\"",$callerAddr,$contextSize);
my $result=run($s);
print $result;

$s=sprintf("echo -e \"\$(/usr/local/OPEN_R_SDK/bin/mipsel-linux-objdump.exe -S -C -w -l -m mips:4300 --start-address=0x%x --stop-address=0x%x $nosnapFile |grep -v Disassembly |grep -v tradlittlemips |tr -d '\\015' |sed 's=\\(%x:.*\\)=\\\\033[32m\\1\\\\033[37m=')\"",$startAddr2,$stopAddr2,$callerAddr);
$result=run($s);
print $result;

$s=sprintf("echo -e \"\n\n\\\\033[33mNow we create an assembler dump of crash address (0x%x) +- 0x%x,\nthe crashing line is in \\\\033[31m$objName\\\\033[33m and marked \\\\033[31mred\\\\033[33m:\\\\033[37m\"",$staticAddr,$contextSize);
$result=run($s);
print $result;

$s=sprintf("echo -e \"\$(/usr/local/OPEN_R_SDK/bin/mipsel-linux-objdump.exe -S -C -w -l -m mips:4300 --start-address=0x%x --stop-address=0x%x $nosnapFile |grep -v Disassembly |grep -v tradlittlemips |tr -d '\\015' |sed 's=\\(%x:.*\\)=\\\\033[31m\\1\\\\033[37m=')\"",$startAddr,$stopAddr,$staticAddr);
$result=run($s);
print $result;

#create a list of all methods in this binary with its addresses
$s=sprintf("echo -e \"\n\n\\\\033[33mNow we create a list of all methods in this binary for caller stack analysis.\nThis will take some seconds.\n\\\\033[37m\"");
$result=run($s);
print $result;
my @methods=split(/\n/,run("/usr/local/OPEN_R_SDK/bin/mipsel-linux-objdump.exe -D -C -w -m mips:4300 $nosnapFile |grep \"^[0-9a-f]* <\" |sed 's=:\$=='"));

# check the [stack dump] section of emon.log
print "\n\nAnalysis of stack dump. The following numbers were found on the stack\nand some could be return adresses (calling method is written).\nExcept some false friends!:\n\n";
my @lines = split(/\n/,$emonLog);
foreach(@lines)
{
  if (my @StackData = ($_ =~ /$sHexRe:\s+$sHexRe\s+$sHexRe\s+$sHexRe\s+$sHexRe\s+(\S+)/))
  {
    checkCaller($StackData[1]);
    checkCaller($StackData[2]);
    checkCaller($StackData[3]);
    checkCaller($StackData[4]);
  }
}

######################################################################
# subs
#

sub readHex($$) {
    my ($str, $re) = @_;
    if ($emonLog !~ /$re$lHexRe/) {
        die "INTERNAL ERROR: readHex failed: $re\n";
    }
    $1;
}

# executes the command given as parameter
sub run($) {
    my ($command) = @_;
    my $ans = `$command`;
    if ($?) {
        die "ERROR: command failed: $command\n";
    }
    $ans;
}

# checks whether the parameter (a number found on the stack)
# can be a valid (jump back) address and outputs the
# method this (assumed) call was executed in
sub checkCaller($) {
  my ($number) = @_;
  print $number;
  if ((hex($gpReg)-hex($number)) eq 0)
  {
    print "  = gpReg";
  }
  if ((hex($number)-$offset) ge 0)
  {
    if ((hex($gpReg)-hex($number)) gt 0)
    {
      for(my $i=0;$i<(@methods.length)-1;$i++)
      {
        $methods[$i] =~ /$sHexRe\s+(.*)/;
        my $akt = $1;
        if ($2 eq "<.fini>")
        {
          $i=@methods.length;
        }
        else
        {
          $methods[$i+1] =~ /$sHexRe\s+(.*)/;
          my $nex = $1;
          if (hex($number)-$offset-hex($akt) ge 0)
          {
            if (hex($number)-$offset-hex($nex) lt 0)
            {
              my $t = sprintf("%x",hex($number)-$offset);
              print "  = $t   in ";
              print $methods[$i];
              $i=@methods.length;
            }
          }
        }
      }
    }
  }
  print "\n";
}

#
# Change log:
#
# $Log: emonLogParser.pl,v $
# Revision 1.1.1.1  2004/05/22 17:10:24  cvsadm
# created new repository GT2004_WM
#
# Revision 1.3  2003/12/27 17:57:32  roefer
# Simpler emon.log handling
#
# Revision 1.2  2003/10/07 12:21:50  dueffert
# no message
#
# Revision 1.1  2003/10/07 10:16:56  cvsadm
# Created GT2004 (M.J.)
#
# Revision 1.2  2003/09/05 15:10:40  dueffert
# beautified; stack dump analysis added (idea from team Griffith)
#
# Revision 1.1.1.1  2003/07/02 09:40:21  cvsadm
# created new repository for the competitions in Padova from the 
# tamara CVS (Tuesday 2:00 pm)
#
# removed unused solutions
#
# Revision 1.5  2003/04/02 17:17:02  dueffert
# caller dump added
#
# Revision 1.4  2003/04/02 09:23:59  dueffert
# beautified
#
# Revision 1.3  2003/04/02 09:21:51  dueffert
# colored output instead of dump file
#
# Revision 1.2  2003/03/29 15:27:21  loetzsch
# the configuration has to be given as parameter 2
#
# Revision 1.1  2003/03/14 13:48:13  dueffert
# emonLogParse from sample/Crash adapted to our needs: needs only EMON.log as parameter
#
#
#