/**
* @file Xabsl2Symbols.h
*
* Definition of class Xabsl2Symbols and helper classes
*
* @author Martin Ltzsch
*/

#ifndef __Xabsl2Symbols_h_
#define __Xabsl2Symbols_h_

#include "Xabsl2Tools.h"

/**
* @class Xabsl2FunctionProvider
*
* Base class for all those classes that want to register functions for symbols
* at a Xabsl2Engine
*
* @author Martin Ltzsch
*/
class Xabsl2FunctionProvider {
  //virtual void dummy(){}
};


/** 
* A Template for the input symbol classes
* @author Martin Ltzsch
*/
template<class T> class Xabsl2InputSymbol : public Xabsl2NamedItem
{
public:
/** 
* Constructor 
* @param name The name of the symbol, for debugging purposes
* @param pVariable A pointer to the variable that the symbol stands for
  */
  Xabsl2InputSymbol(const char* name, const T* pVariable)
    : Xabsl2NamedItem(name), pV(pVariable), pI(0), pF(0)
  {};
  
  
  /** Constructor 
  * @param name The name of the symbol, for debugging purposes
  * @param pFunction A pointer to a boolean returning function in the software environment 
  * @param pInstance A pointer to the instance that contains the function
  */
  Xabsl2InputSymbol(const char* name,
    Xabsl2FunctionProvider* pInstance,
    T (Xabsl2FunctionProvider::*pFunction)())
    : Xabsl2NamedItem(name), pV(0), pI(pInstance), pF(pFunction) {};
  
  /** returns the value of the symbol */
  T getValue() const
  { if (pF!=0) return (pI->*pF)(); else return *pV; }
  
private:
  /** A pointer to a variable in the software environment */
  const T* pV; 
  
  /** A pointer to the instance object that contains the function */
  Xabsl2FunctionProvider* pI;
  
  /** A pointer to a T returning function in the software environment */
  T (Xabsl2FunctionProvider::*pF)(); 
};

/** 
* @class Xabsl2DecimalInputSymbol
*
* Represents a decimal input symbol of the Xabsl2Engine 
*
* @author Martin Ltzsch
*/
class Xabsl2DecimalInputSymbol : public Xabsl2InputSymbol<double>
{
public:
/** 
* Constructor 
* @param name The name of the symbol, for debugging purposes
* @param pVariable A pointer to the variable that the symbol stands for
  */
  Xabsl2DecimalInputSymbol(const char* name, const double* pVariable)
    : Xabsl2InputSymbol<double>(name, pVariable) 
  {};
  
  
    /** Constructor 
    * @param name The name of the symbol, for debugging purposes
    * @param pFunction A pointer to a double returning function in the software environment 
    * @param pInstance A pointer to the instance that contains the function
  */
  Xabsl2DecimalInputSymbol(const char* name,
    Xabsl2FunctionProvider* pInstance,
    double (Xabsl2FunctionProvider::*pFunction)())
    : Xabsl2InputSymbol<double>(name, pInstance, pFunction) {};
};

/** 
* @class Xabsl2DecimalInputFunction
*
* Represents a parameterized decimal input function of the Xabsl2Engine 
*
* @author Martin Ltzsch
*/
class Xabsl2DecimalInputFunction : public Xabsl2NamedItem
{
public:
 /** Constructor 
 * @param name The name of the function, for debugging purposes
 * @param pFunction A pointer to a double returning function in the software environment 
 * @param pInstance A pointer to the instance that contains the function
  */
  Xabsl2DecimalInputFunction(const char* name,
    Xabsl2FunctionProvider* pInstance,
    double (Xabsl2FunctionProvider::*pFunction)())
    : Xabsl2NamedItem(name), pF(pFunction), pI(pInstance)
  {};

  /** 
  * Calculates the value of the function. 
  * Note that the parameters for the function have to be set before that function is called.
  */ 
  double getValue() { return (pI->*pF)(); }

  /** References to the function parameters */
  Xabsl2Array<double&> parameters;

private:
  /** A pointer to a double returning function in the software environment */
  double (Xabsl2FunctionProvider::*pF)(); 
  
  /** A pointer to the instance object that contains the function */
  Xabsl2FunctionProvider* pI;
};

/** 
* @class Xabsl2BooleanInputSymbol
*
* Represents a boolean input symbol of the Xabsl2Engine 
*
* @author Martin Ltzsch
*/
class Xabsl2BooleanInputSymbol : public Xabsl2InputSymbol<bool>
{
public:
/** 
* Constructor 
* @param name The name of the symbol, for debugging purposes
* @param pVariable A pointer to the variable that the symbol stands for
  */
  Xabsl2BooleanInputSymbol(const char* name, const bool* pVariable)
    : Xabsl2InputSymbol<bool>(name, pVariable) 
  {};
  
  
  /** Constructor 
  * @param name The name of the symbol, for debugging purposes
  * @param pFunction A pointer to a boolean returning function in the software environment 
  * @param pInstance A pointer to the instance that contains the function
  */
  Xabsl2BooleanInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    bool (Xabsl2FunctionProvider::*pFunction)())
    : Xabsl2InputSymbol<bool>(name, pInstance, pFunction) {};
};

/**
* @class Xabsl2EnumElement
* Represents an enum element that is part of an enumerated input or output symbol.
* @author Martin Ltzsch
*/
class Xabsl2EnumElement : public Xabsl2NamedItem
{
public:
/** 
* Constructor 
* @param name The name of the enum element as specified in the XML formalization
* @param value The value for the element from the software environment
  */
  Xabsl2EnumElement(const char* name, int value)
    : Xabsl2NamedItem(name), v(value) {};
  
  /** The enum value from a function or variable in the software environment */
  int v;
};

/** 
* @class Xabsl2EnumeratedInputSymbol
*
* Represents a enumerated input symbol of the Xabsl2Engine 
*
* @author Martin Ltzsch
*/
class Xabsl2EnumeratedInputSymbol : public Xabsl2InputSymbol<int>
{
public:
/** 
* Constructor 
* @param name The name of the symbol, for debugging purposes
* @param pVariable A pointer to the variable that the symbol stands for
  */
  Xabsl2EnumeratedInputSymbol(const char* name, const int* pVariable)
    : Xabsl2InputSymbol<int>(name, pVariable) 
  {};
  
  
    /** Constructor 
    * @param name The name of the symbol, for debugging purposes
    * @param pFunction A pointer to an int returning function in the software environment 
    * @param pInstance A pointer to the instance that contains the function
  */
  Xabsl2EnumeratedInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    int (Xabsl2FunctionProvider::*pFunction)())
    : Xabsl2InputSymbol<int>(name, pInstance, pFunction) {};
  
  /** Destructor. Deletes the enum elements */
  ~Xabsl2EnumeratedInputSymbol();
  
  /** 
  * Assignes a enum value from a function or variable in the software environment 
  * to the enum-element string in the XML formalization.
  */
  Xabsl2Array<Xabsl2EnumElement*> enumElements;
};

/** 
* @class Xabsl2EnumeratedOutputSymbol
*
* Represents a enumerated output symbol of the Xabsl2Engine 
*
* @author Martin Ltzsch
*/
class Xabsl2EnumeratedOutputSymbol : public Xabsl2NamedItem
{
public:
/** 
* Constructor 
* @param name The name of the symbol, for debugging purposes
* @param pVariable A pointer to the variable that the symbol stands for
  */
  Xabsl2EnumeratedOutputSymbol(const char* name, int* pVariable)
    : Xabsl2NamedItem(name), activeValue(0), pV(pVariable), pF(0), pI(0)
  {};
  
  
  /** Constructor 
  * @param name The name of the symbol, for debugging purposes
  * @param pFunction A pointer to a function in the software environment that sets the symbol
  * @param pInstance A pointer to the instance that contains the function
  */
  Xabsl2EnumeratedOutputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    void (Xabsl2FunctionProvider::*pFunction)(int))
    : Xabsl2NamedItem(name), activeValue(0), activeValueWasSet(false),
    pV(0), pF(pFunction), pI(pInstance) {};
  
  /** Destructor. Deletes the enum elements */
  ~Xabsl2EnumeratedOutputSymbol();
  
  /** 
  * Assignes a enum value from a function or variable in the software environment 
  * to the enum-element string in the XML formalization.
  */
  Xabsl2Array<Xabsl2EnumElement*> enumElements;

  /** The value that was set during the last execution of the option graph. */
  int activeValue;

  /** If true, the value was set during the last execution of the option graph. */
  bool activeValueWasSet;

  /** Sets the current set value to the software environment */
  void setActiveValue();

private:
  
  /** A pointer to a variable in the software environment */
  int* pV; 
  
  /** A pointer to a function that sets the symbol in the software environment */
  void (Xabsl2FunctionProvider::*pF)(int); 
  
  /** A pointer to the instance object that contains the function */
  Xabsl2FunctionProvider* pI;
};

/**
* @class Xabsl2Symbols
*
* Handles the symbols of the Xabsl2Engine.
*
* @author Martin Ltzsch
*/
class Xabsl2Symbols
{
public:
/** 
* Constructor.
* @param errorHandler Is invoked when errors occur
  */
  Xabsl2Symbols(Xabsl2ErrorHandler& errorHandler)
    : errorHandler(errorHandler) {};
  
  /** Destructor */
  ~Xabsl2Symbols();
  
  /** 
  * Registers the address of a variable for a decimal input symbol.
  * @param name The name of the symbol
  * @param pVariable A pointer to a variable in the software environment 
  */
  void registerDecimalInputSymbol(const char* name, const double* pVariable);
  
  /** 
  * Registers the address of a function for a decimal input symbol.
  * @param name The name of the symbol
  * @param pFunction A pointer to a function that calculates a value for the symbol
  * @param pInstance A pointer to an object that provides the function
  */
  void registerDecimalInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    double (Xabsl2FunctionProvider::*pFunction)());
  
    /** 
    * Returns a decimal input symbol for a given name
    * Note that the function crashes if the symbol does not exist.
    * @param name The name of the symbol
    * @return A pointer to the symbol
  */
  Xabsl2DecimalInputSymbol* getDecimalInputSymbol(const char* name);
  
  /** Returns whether a decimal input symbol exists */
  bool existsDecimalInputSymbol(const char* name);
  
  
  /** 
  * Registers the address of a parameterized decimal input function.
  * @param name The name of the function
  * @param pFunction A pointer to the function.
  * @param pInstance A pointer to an object that provides the function.
  */
  void registerDecimalInputFunction(const char* name, Xabsl2FunctionProvider* pInstance,
    double (Xabsl2FunctionProvider::*pFunction)());
  
  /** 
  * Registers a parameter of a parameterized decimal input function.
  * @param functionName The name of the function
  * @param name The name of the parameter
  * @param param A reference to the parameter
  */
  void registerDecimalInputFunctionParameter(const char* functionName, 
    const char* name, double& param);
  
  /** 
  * Returns a decimal input function for a given name
  * Note that the function crashes if the function does not exist.
    * @param name The name of the function
    * @return A pointer to the function
  */
  Xabsl2DecimalInputFunction* getDecimalInputFunction(const char* name);
  
  /** Returns whether a decimal input function exists */
  bool existsDecimalInputFunction(const char* name);

  
  /** 
  * Registers the address of a variable for a boolean input symbol.
  * @param name The name of the symbol
  * @param pVariable A pointer to a variable in the software environment 
  */
  void registerBooleanInputSymbol(const char* name, const bool* pVariable);
  
  /** 
  * Registers the address of a function for a boolean input symbol.
  * @param name The name of the symbol
  * @param pFunction A pointer to a function that calculates a value for the symbol
  * @param pInstance A pointer to an object that provides the function
  */
  void registerBooleanInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    bool (Xabsl2FunctionProvider::*pFunction)());
  
    /** 
    * Returns a boolean input symbol for a given name
    * Note that the function crashes if the symbol does not exist.
    * @param name The name of the symbol
    * @return A pointer to the symbol
  */
  Xabsl2BooleanInputSymbol* getBooleanInputSymbol(const char* name);
  
  /** Returns whether a boolean input symbol exists */
  bool existsBooleanInputSymbol(const char* name);
  
  /** 
  * Registers the address of a variable for a enumerated input symbol.
  * @param name The name of the symbol
  * @param pVariable A pointer to a variable in the software environment 
  */
  void registerEnumeratedInputSymbol(const char* name, const int* pVariable);
  
  /** 
  * Registers the address of a function for a enumerated input symbol.
  * @param name The name of the symbol
  * @param pFunction A pointer to a function that calculates a value for the symbol
  * @param pInstance A pointer to an object that provides the function
  */
  void registerEnumeratedInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    int (Xabsl2FunctionProvider::*pFunction)());
  
    /** 
    * Returns a enumerated input symbol for a given name
    * Note that the function crashes if the symbol does not exist.
    * @param name The name of the symbol
    * @return A pointer to the symbol
  */
  Xabsl2EnumeratedInputSymbol* getEnumeratedInputSymbol(const char* name);
  
  /**
  * Registers an enum element for an enumerated input symbol.
  * @param symbolName The name of the symbol
  * @param name The name of the enum element
  * @param value The value of the element
  */
  void registerEnumeratedInputSymbolEnumElement(const char* symbolName, 
    const char* name, int value);
  
  /** Returns whether a boolean input symbol exists */
  bool existsEnumeratedInputSymbol(const char* name);
  
  
  /** 
  * Registers the address of a variable for a enumerated output symbol.
  * @param name The name of the symbol
  * @param pVariable A pointer to a variable in the software environment 
  */
  void registerEnumeratedOutputSymbol(const char* name, int* pVariable);
  
  /** 
  * Registers the address of a function for a enumerated output symbol.
  * @param name The name of the symbol
  * @param pFunction A pointer to a function that sets a value for the symbol
  * @param pInstance A pointer to an object that provides the function
  */
  void registerEnumeratedOutputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    void (Xabsl2FunctionProvider::*pFunction)(int));
  
    /** 
    * Returns a enumerated output symbol for a given name
    * Note that the function crashes if the symbol does not exist.
    * @param name The name of the symbol
    * @return A pointer to the symbol
  */
  Xabsl2EnumeratedOutputSymbol* getEnumeratedOutputSymbol(const char* name);
  
  /**
  * Registers an enum element for an enumerated output symbol.
  * @param symbolName The name of the symbol
  * @param name The name of the enum element
  * @param value The value of the element
  */
  void registerEnumeratedOutputSymbolEnumElement(const char* symbolName, 
    const char* name, int value);
  
  /** Returns whether a boolean output symbol exists */
  bool existsEnumeratedOutputSymbol(const char* name);
  
  /** Sets the output symbols to the software environment */
  void setOutputSymbols();
  
private:
  /** The decimal input symbols */
  Xabsl2Array<Xabsl2DecimalInputSymbol*> decimalInputSymbols;
  
  /** The decimal input functions */
  Xabsl2Array<Xabsl2DecimalInputFunction*> decimalInputFunctions;
  
  /** The decimal input symbols */
  Xabsl2Array<Xabsl2BooleanInputSymbol*> booleanInputSymbols;
  
  /** The enumerated input symbols */
  Xabsl2Array<Xabsl2EnumeratedInputSymbol*> enumeratedInputSymbols;
  
  /** The enumerated output symbols */
  Xabsl2Array<Xabsl2EnumeratedOutputSymbol*> enumeratedOutputSymbols;
  
  /** Is invoked when errors occur */
  Xabsl2ErrorHandler& errorHandler;
};



#endif //__Xabsl2Symbols_h_

/*
* Change Log:
*
* $Log: Xabsl2Symbols.h,v $
* Revision 1.2  2004/03/19 15:34:01  tim
* added some const qualifiers
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.3  2003/09/16 13:27:21  loetzsch
* changed all occurrences of "option tree" to "option graph"
*
* Revision 1.2  2003/08/09 14:53:10  dueffert
* files and docu beautified
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/05/25 22:36:32  loetzsch
* only when an output symbol was set during the execution of the graph,
* coresponding function is invoked or the coresponding variable is changed.
*
* Revision 1.11  2003/03/06 11:44:33  dueffert
* re-order warnings removed
*
* Revision 1.10  2003/01/28 18:07:47  loetzsch
* no message
*
* Revision 1.9  2003/01/13 13:18:44  loetzsch
* removed "#include <string.h>"
*
* Revision 1.8  2003/01/09 17:27:59  loetzsch
* changed comments
*
* Revision 1.7  2003/01/09 10:03:00  dueffert
* added missing chars
*
* Revision 1.6  2003/01/09 10:01:20  loetzsch
* added interfaces to the Xabsl2 Dialog in the RobotControl application
*
* Revision 1.5  2003/01/08 11:07:12  loetzsch
* all kinds of symbols now can be registered at the Xabsl2Engine
*
* Revision 1.4  2002/12/18 16:22:56  dueffert
* doxygen docu corrected
*
* Revision 1.3  2002/12/16 14:19:58  loetzsch
* added boolean input symbols
*
* Revision 1.2  2002/12/07 12:34:13  loetzsch
* added debug interfaces
*
* Revision 1.1  2002/12/06 21:13:37  loetzsch
* Decimal input symbols can now be registered at the engine
*
*/
