/**
* @file Xabsl2Symbols.cpp
*
* Implementation of class Xabsl2Symbols and helper classes
*
* @author Martin Ltzsch
*/

#include "Xabsl2Symbols.h"

Xabsl2EnumeratedInputSymbol::~Xabsl2EnumeratedInputSymbol()
{
  int i;
  for (i=0; i< enumElements.getSize(); i++)
  {
    delete enumElements[i];
  }
}

Xabsl2EnumeratedOutputSymbol::~Xabsl2EnumeratedOutputSymbol()
{
  int i;
  for (i=0; i< enumElements.getSize(); i++)
  {
    delete enumElements[i];
  }
}

void Xabsl2EnumeratedOutputSymbol::setActiveValue()
{
  if (activeValueWasSet)
  {
    if (pF!=0) 
    {
      (pI->*pF)(activeValue);
    }
    else 
    {
      *pV=activeValue;
    }
    activeValueWasSet = false;
  }
}

Xabsl2Symbols::~Xabsl2Symbols()
{
  int i;
  for (i=0; i< decimalInputSymbols.getSize(); i++) delete decimalInputSymbols[i];
  for (i=0; i< decimalInputFunctions.getSize(); i++) delete decimalInputFunctions[i];
  for (i=0; i< booleanInputSymbols.getSize(); i++) delete booleanInputSymbols[i];
  for (i=0; i< enumeratedInputSymbols.getSize(); i++) delete enumeratedInputSymbols[i];
  for (i=0; i< enumeratedOutputSymbols.getSize(); i++) delete enumeratedOutputSymbols[i];
}

void Xabsl2Symbols::registerDecimalInputSymbol(const char* name, const double* pVariable)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering decimal input symbol \"%s\"",name));
  
  if (decimalInputSymbols.exists(name))
  {
    errorHandler.error("registerDecimalInputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  decimalInputSymbols.append(name,new Xabsl2DecimalInputSymbol(name, pVariable));
}

void Xabsl2Symbols::registerDecimalInputSymbol
(const char* name, Xabsl2FunctionProvider* pInstance,
 double (Xabsl2FunctionProvider::*pFunction)())
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering decimal input symbol \"%s\"",name));
  
  if (decimalInputSymbols.exists(name))
  {
    errorHandler.error("registerDecimalInputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  decimalInputSymbols.append(name,new Xabsl2DecimalInputSymbol(name, pInstance,pFunction));
}

Xabsl2DecimalInputSymbol* Xabsl2Symbols::getDecimalInputSymbol(const char* name)
{
  return decimalInputSymbols.getElement(name);
}

bool Xabsl2Symbols::existsDecimalInputSymbol(const char* name)
{
  return decimalInputSymbols.exists(name);
} 


void Xabsl2Symbols::registerDecimalInputFunction(const char* name, Xabsl2FunctionProvider* pInstance,
                                                 double (Xabsl2FunctionProvider::*pFunction)())
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering decimal input function\"%s\"",name));

  if (decimalInputFunctions.exists(name))
  {
    errorHandler.error("registerDecimalInputFunction(): function \"%s\" was already registered",name);
    return;
  }

  decimalInputFunctions.append(name, new Xabsl2DecimalInputFunction(name,pInstance,pFunction));
}
  
void Xabsl2Symbols::registerDecimalInputFunctionParameter(const char* functionName, 
    const char* name, double& param)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering parameter \"%s\" for decimal input function\"%s\"",name, functionName));

  if (!decimalInputFunctions.exists(functionName))
  {
    errorHandler.error("registerDecimalInputFunctionParameter(): function \"%s\" was not registered",functionName);
    return;
  }
  if (decimalInputFunctions.getElement(functionName)->parameters.exists(name))
  {
    errorHandler.error("registerDecimalInputFunctionParameter(): parameter \"%s\" was already registered",name);
    return;
  }

  decimalInputFunctions.getElement(functionName)->parameters.append(name,param);
}
  
Xabsl2DecimalInputFunction* Xabsl2Symbols::getDecimalInputFunction(const char* name)
{
  return decimalInputFunctions.getElement(name);
}
  
bool Xabsl2Symbols::existsDecimalInputFunction(const char* name)
{
  return decimalInputFunctions.exists(name);
}


void Xabsl2Symbols::registerBooleanInputSymbol(const char* name, const bool* pVariable)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering boolean input symbol \"%s\"",name));
  
  if (booleanInputSymbols.exists(name))
  {
    errorHandler.error("registerBooleanInputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  booleanInputSymbols.append(name,new Xabsl2BooleanInputSymbol(name, pVariable));
}


void Xabsl2Symbols::registerBooleanInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
                                               bool (Xabsl2FunctionProvider::*pFunction)())
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering boolean input symbol \"%s\"",name));
  
  if (booleanInputSymbols.exists(name))
  {
    errorHandler.error("registerBooleanInputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  booleanInputSymbols.append(name,new Xabsl2BooleanInputSymbol(name, pInstance,pFunction));
}

Xabsl2BooleanInputSymbol* Xabsl2Symbols::getBooleanInputSymbol(const char* name)
{
  return booleanInputSymbols.getElement(name);
}

bool Xabsl2Symbols::existsBooleanInputSymbol(const char* name)
{
  return booleanInputSymbols.exists(name);
}

void Xabsl2Symbols::registerEnumeratedInputSymbol(const char* name, const int* pVariable)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering enumerated input symbol \"%s\"",name));
  
  if (enumeratedInputSymbols.exists(name))
  {
    errorHandler.error("registerEnumeratedInputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  enumeratedInputSymbols.append(name,new Xabsl2EnumeratedInputSymbol(name, pVariable));
}

void Xabsl2Symbols::registerEnumeratedInputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    int (Xabsl2FunctionProvider::*pFunction)())
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering enumerated input symbol \"%s\"",name));
  
  if (enumeratedInputSymbols.exists(name))
  {
    errorHandler.error("registerEnumeratedInputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  enumeratedInputSymbols.append(name,new Xabsl2EnumeratedInputSymbol(name, pInstance,pFunction));
}

Xabsl2EnumeratedInputSymbol* Xabsl2Symbols::getEnumeratedInputSymbol(const char* name)
{
  return enumeratedInputSymbols.getElement(name);
}

void Xabsl2Symbols::registerEnumeratedInputSymbolEnumElement(const char* symbolName, 
    const char* name, int value)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering enum element \"%s\" for enumerated input symbol \"%s\"",name, symbolName));

  if (!enumeratedInputSymbols.exists(symbolName))
  {
    errorHandler.error("registerEnumeratedInputSymbolEnumElement(): symbol \"%s\" was not registered.", symbolName);
    return;
  }
  if (enumeratedInputSymbols.getElement(symbolName)->enumElements.exists(name))
  {
    errorHandler.error("registerEnumeratedInputSymbolEnumElement(): enum element \"%s\" for symbol \"%s\" was already registered.", name, symbolName);
    return;
  }
  enumeratedInputSymbols.getElement(symbolName)->enumElements.append(name, new Xabsl2EnumElement(name,value));
}

bool Xabsl2Symbols::existsEnumeratedInputSymbol(const char* name)
{
  return enumeratedInputSymbols.exists(name);
}


void Xabsl2Symbols::registerEnumeratedOutputSymbol(const char* name, int* pVariable)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering enumerated output symbol \"%s\"",name));
  
  if (enumeratedOutputSymbols.exists(name))
  {
    errorHandler.error("registerEnumeratedOutputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  enumeratedOutputSymbols.append(name,new Xabsl2EnumeratedOutputSymbol(name, pVariable));
}

void Xabsl2Symbols::registerEnumeratedOutputSymbol(const char* name, Xabsl2FunctionProvider* pInstance,
    void (Xabsl2FunctionProvider::*pFunction)(int))
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering enumerated output symbol \"%s\"",name));
  
  if (enumeratedOutputSymbols.exists(name))
  {
    errorHandler.error("registerEnumeratedOutputSymbol(): symbol \"%s\" was already registered",name);
    return;
  }
  enumeratedOutputSymbols.append(name,new Xabsl2EnumeratedOutputSymbol(name, pInstance,pFunction));
}

Xabsl2EnumeratedOutputSymbol* Xabsl2Symbols::getEnumeratedOutputSymbol(const char* name)
{
  return enumeratedOutputSymbols.getElement(name);
}

void Xabsl2Symbols::registerEnumeratedOutputSymbolEnumElement(const char* symbolName, 
    const char* name, int value)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering enum element \"%s\" for enumerated output symbol \"%s\"",name, symbolName));

  if (!enumeratedOutputSymbols.exists(symbolName))
  {
    errorHandler.error("registerEnumeratedOutputSymbolEnumElement(): symbol \"%s\" was not registered.", symbolName);
    return;
  }
  if (enumeratedOutputSymbols.getElement(symbolName)->enumElements.exists(name))
  {
    errorHandler.error("registerEnumeratedOutputSymbolEnumElement(): enum element \"%s\" for symbol \"%s\" was already registered.", name, symbolName);
    return;
  }
  enumeratedOutputSymbols.getElement(symbolName)->enumElements.append(name, new Xabsl2EnumElement(name,value));
}

bool Xabsl2Symbols::existsEnumeratedOutputSymbol(const char* name)
{
  return enumeratedOutputSymbols.exists(name);
}


void Xabsl2Symbols::setOutputSymbols()
{
  for (int i=0;i<enumeratedOutputSymbols.getSize();i++)
    enumeratedOutputSymbols[i]->setActiveValue();
}


/*
* Change Log:
*
* $Log: Xabsl2Symbols.cpp,v $
* Revision 1.2  2004/03/19 15:34:01  tim
* added some const qualifiers
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/25 22:36:32  loetzsch
* only when an output symbol was set during the execution of the graph,
* coresponding function is invoked or the coresponding variable is changed.
*
* Revision 1.5  2003/05/19 10:30:02  risler
* wrong error messages corrected
*
* Revision 1.4  2003/01/08 11:07:12  loetzsch
* all kinds of symbols now can be registered at the Xabsl2Engine
*
* Revision 1.3  2002/12/16 14:19:58  loetzsch
* added boolean input symbols
*
* Revision 1.2  2002/12/07 12:34:13  loetzsch
* added debug interfaces
*
* Revision 1.1  2002/12/06 21:13:37  loetzsch
* Decimal input symbols can now be registered at the engine
*
*/
