/** 
* @file Xabsl2Engine.cpp
*
* Implementation of class Xabsl2Engine
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#include "Xabsl2Engine.h"

Xabsl2Engine::Xabsl2Engine(Xabsl2ErrorHandler& e,unsigned long (*pTimeFunction)())
: Xabsl2Symbols(e), selectedAgent(0), rootOption(0),
selectedBasicBehavior(0), errorHandler(e), initialized(false), 
pTimeFunction(pTimeFunction)
{
}

Xabsl2Engine::~Xabsl2Engine()
{
  int i;
  for (i=0; i< options.getSize(); i++)
    if (options[i]!=0) 
      delete options[i];
  for (i=0; i< agents.getSize(); i++)
    if (agents[i]!=0) 
      delete agents[i];
}

void Xabsl2Engine::execute()
{
  Xabsl2Option* currentOption = rootOption;

  int i;

  for (i=0;i<basicBehaviors.getSize();i++)
  {
    if (&basicBehaviors[i] == selectedBasicBehavior) 
      basicBehaviors[i].basicBehaviorWasActiveDuringLastExecutionOfEngine = true;
    else
      basicBehaviors[i].basicBehaviorWasActiveDuringLastExecutionOfEngine = false;
  }
  
  while (currentOption!=0)
  {
    currentOption->execute();
    
    Xabsl2Option* nextOption = currentOption->activeState->subsequentOption;
    if (nextOption == 0)
    {
      selectedBasicBehavior = currentOption->activeState->subsequentBasicBehavior;
    }
    currentOption = nextOption;
  }

  for (i=0; i< options.getSize(); i++)
  {
    Xabsl2Option* option = options[i];
    option->optionWasActive = option->optionIsActive;
    option->optionIsActive = false;
  }

  setOutputSymbols();

  executeSelectedBasicBehavior();
}

void Xabsl2Engine::executeSelectedBasicBehavior()
{
  selectedBasicBehavior->execute();

  selectedBasicBehavior->basicBehaviorWasActiveDuringLastExecutionOfEngine = true;
}

void Xabsl2Engine::createOptionGraph(Xabsl2InputSource& input)
{
  int i;
  char buf1[100],buf2[100];

  if (initialized)
  {
    errorHandler.error("createOptionGraph(): Don't call this function twice");
    return;
  }
  if (!input.open()) 
  {
    errorHandler.error("createOptionGraph(): Can't open input source");
    return;
  }

  // the total number of options in the intermediate code
  int numberOfOptions = (int)input.readValue(); 
  
  // create empty options
  for (i=0; i< numberOfOptions; i++)
  {
    input.readString(buf1,99);
    options.append(buf1,new Xabsl2Option(buf1,input,errorHandler,pTimeFunction));
  }
  XABSL2_DEBUG_INIT(errorHandler.message("registered %i options",numberOfOptions));

  // create the options and its states
  for (i=0; i< numberOfOptions; i++)
  {
    XABSL2_DEBUG_INIT(errorHandler.message("creating option \"%s\"",options[i]->n));
    options[i]->create(input,options,basicBehaviors,*this);
    if (errorHandler.errorsOccurred)
    {
      errorHandler.error("Xabsl2Engine::createOptionGraph(): could not create option \"%s\"",options[i]->n);
      return;
    }
  }

  // create the agents
  int numberOfAgents = (int)input.readValue();
  for (i=0; i< numberOfAgents; i++)
  {
    input.readString(buf1,99);
    input.readString(buf2,99);
    agents.append(buf1,new Xabsl2Agent(buf1,options.getElement(buf2),errorHandler));
  }

  // check for loops in the option graph
  for (i=0;i<agents.getSize();i++)
  {
    Xabsl2Option* currentOptionPath[1000];
    
    currentOptionPath[0] = agents[i]->getRootOption();

    // call recursively the checkForLoops function
    if (checkForLoops(currentOptionPath,0)) 
    {
      errorHandler.error("createOptionGraph(): The created option graph contains loops");
    };
  }


  selectedAgent = agents[0];
  selectedBasicBehavior = &basicBehaviors[0];
  rootOption = selectedAgent->getRootOption();

  XABSL2_DEBUG_INIT(errorHandler.message("selected agent: \"%s\"",selectedAgent->n));
  input.close();
  initialized = true;
}


bool Xabsl2Engine::checkForLoops(Xabsl2Option* currentOptionPath[],int currentDepth)
{
  int i,j;
  Xabsl2Option* currentOption = currentOptionPath[currentDepth];

  for (i=0; i<currentOption->states.getSize(); i++)
  {
    if (currentOption->states[i]->subsequentOption != 0)
    {
      for(j=0; j<=currentDepth; j++)
      {
        // check for the subsequent option of each state whether the referenced 
        // option is contained in the current option path
        if (currentOption->states[i]->subsequentOption == currentOptionPath[j])
        {
          errorHandler.error("checkForLoops() : state \"%s\" in option \"%s\" references subsequent option \"%s\". But option \"%s\" is also directly or indirectly referenced by option \"%s\".",
            currentOption->states[i]->n, 
            currentOption->n,
            currentOption->states[i]->subsequentOption->n,
            currentOption->n,
            currentOption->states[i]->subsequentOption->n);

          return true;
        }
      }
      
      // recursion
      currentOptionPath[currentDepth+1] = currentOption->states[i]->subsequentOption;
      if (checkForLoops(currentOptionPath,currentDepth+1))
      {
        return true;
      }
    }
  }

  return false;
}


void Xabsl2Engine::registerBasicBehavior(Xabsl2BasicBehavior& basicBehavior)
{
  XABSL2_DEBUG_INIT(errorHandler.message("registering basic behavior \"%s\"",basicBehavior.n));

  if (basicBehaviors.exists(basicBehavior.n))
  {
    errorHandler.error("registerBasicBehavior(): basic behavior \"%s\" was already registered",basicBehavior.n);
    return;
  }
  basicBehaviors.append(basicBehavior.n,basicBehavior);
}

bool Xabsl2Engine::setSelectedBasicBehavior(const char* name)
{
  if (!basicBehaviors.exists(name))
    return false;
  else
  {
    selectedBasicBehavior = &basicBehaviors.getElement(name);
    return true;
  }
}

bool Xabsl2Engine::setRootOption(const char* name)
{
  // check if the option exists
  if (!options.exists(name)) return false;

  // set the curren root option to the requested option
  rootOption = options.getElement(name);

  return true;
}

void Xabsl2Engine::setRootOption()
{
  rootOption = selectedAgent->getRootOption();
}

const Xabsl2Option* Xabsl2Engine::getRootOption() const
{
  return rootOption;
}

bool Xabsl2Engine::setBasicBehaviorParameter(const char* name, const char* param, double value)
{
  if (!basicBehaviors.exists(name)) 
  {
    return false;
  }
  else if (!basicBehaviors.getElement(name).parameters.exists(param))
  {
    return false;
  }
  else 
  { 
    basicBehaviors.getElement(name).parameters.setElement(param,value);
    return false;
  }
}

bool Xabsl2Engine::setOptionParameter(const char* name, const char* param, double value)
{
  if (!options.exists(name)) 
  {
    return false;
  }
  else if (!options.getElement(name)->parameters.exists(param))
  {
    return false;
  }
  else 
  { 
    options.getElement(name)->parameters.setElement(param,value);
    return false;
  }
}

const char* Xabsl2Engine::getSelectedAgentName()
{
  return selectedAgent->n;
}

const Xabsl2BasicBehavior* Xabsl2Engine::getSelectedBasicBehavior()
{
  return selectedBasicBehavior;
}

bool Xabsl2Engine::setSelectedAgent(const char* name)
{
  if (!agents.exists(name)) 
  {
    return false;
  }

  Xabsl2Agent* newAgent = agents.getElement(name);

  if (selectedAgent != newAgent)
  {
    selectedAgent = newAgent;
    rootOption = selectedAgent->getRootOption();
  }

  return true;
}

/*
* Change Log
* 
* $Log: Xabsl2Engine.cpp,v $
* Revision 1.4  2004/04/09 14:31:47  tim
* basic behaviors are now executed after the computation of the output symbols
*
* Revision 1.3  2003/12/16 18:53:22  loetzsch
* The XabslEngine now checks for loops
*
* Revision 1.2  2003/10/08 11:50:09  loetzsch
* made the Xabsl2Engine really platform independent
* (removed inclusion of Platform/SystemCall.h)
* A time function is given to the engine by parameter.
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.4  2003/09/20 16:34:16  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.3  2003/09/16 13:27:21  loetzsch
* changed all occurrences of "option tree" to "option graph"
*
* Revision 1.2  2003/08/04 16:02:50  loetzsch
* ::setSelectedAgent doesn't throw error messages when anymore
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.17  2003/04/14 16:18:51  loetzsch
* ATH after GermanOpen CVS merge
* added basicBehaviorWasActiveDuringLastExecutionOfEngine
*
* Revision 1.2  2003/04/09 17:01:12  loetzsch
* added the boolean basicBehaviorWasActiveDuringLastExecutionOfEngine
*
* Revision 1.1.1.1  2003/04/09 14:23:22  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.16  2003/03/06 11:45:04  dueffert
* re-order warning removed
*
* Revision 1.15  2003/01/28 18:07:47  loetzsch
* no message
*
* Revision 1.14  2003/01/28 17:51:35  loetzsch
* added function setOptionParameter()
*
* Revision 1.13  2003/01/19 13:04:52  loetzsch
* xabsl2 agents now can be changed by using the Module and SolutionRequest
* mechanism
*
* Revision 1.12  2003/01/14 16:28:32  loetzsch
* creation of references to output symbols added
* setting of output symbols added
*
* Revision 1.11  2003/01/13 22:39:38  loetzsch
* implemented the execution of the engine.
*
* Revision 1.10  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
* Revision 1.9  2003/01/12 14:54:04  loetzsch
* continued creation of option tree: Xabsl2Statement and derivates added
*
* Revision 1.8  2003/01/11 14:41:42  loetzsch
* continued creation of the option tree
*
* Revision 1.7  2003/01/09 17:28:33  loetzsch
* introduced Xabsl2Agent, continued Xabsl2Option
*
* Revision 1.6  2003/01/08 18:47:17  loetzsch
* first version of state implementation
*
* Revision 1.5  2003/01/08 15:22:33  loetzsch
* - started implementation of the option tree
* - started the reading of the intermediate code
*
* Revision 1.4  2002/12/11 12:23:31  loetzsch
* basic behaviors register their parameters in their constructor
* the parameters array contains only references to doubles in the basic behavior
*
* Revision 1.3  2002/12/06 21:13:37  loetzsch
* Decimal input symbols can now be registered at the engine
*
* Revision 1.2  2002/12/02 19:56:32  loetzsch
* - Xabsl2Array now seems to work for more than 1 element
* - Basic behaviors now can be registered at the engine
* - Basic behaviors have access to their parameters
*
* Revision 1.1  2002/12/01 17:54:30  loetzsch
* continued Xabsl2Engine: Xabsl2Array seems to work now
*
*/

