/** 
* @file Xabsl2Array.h
*
* Declaration and implementation of template class Xabsl2Array
*
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Thomas Rfer
*/


#ifndef __Xabsl2Array_h_
#define __Xabsl2Array_h_

#include <stdlib.h>
#include <string.h>

/**
* @class Xabsl2NamedItem
* A class that has a text label
* @author Martin Ltzsch
*/
class Xabsl2NamedItem
{
public:
  /** 
  * Constructor
  * @param name The name of the item
  */
  Xabsl2NamedItem(const char* name)
  {
      n = new char[strlen(name)+1];
      strcpy(n,name);
  }

  /** Destructor. Deletes the name */
  ~Xabsl2NamedItem() { delete[] n; }

  /** The text label */
  char* n;
};

/**
* An element of an Xabsl2Array.
*
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Thomas Rfer
*/
template<class T> class Xabsl2ArrayElement : public Xabsl2NamedItem
{
  public:
    /**
    * Constructor.
    * @param name A string label for the element.
    * @param element The new element.
    */
    Xabsl2ArrayElement(const char* name, T element)
      : Xabsl2NamedItem(name), e(element) {}
    
    /** 
    * Destructor. If the element is a pointer, it has to be 
    * deleted externally 
    */
    ~Xabsl2ArrayElement() {}
    
    /** The element */
    T e;
};

/**
* The class implements a dynamic array. Each array element can have a text label.
*
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Thomas Rfer
*/
template <class T> class Xabsl2Array
{
  public:
    /** Constructor */
    Xabsl2Array() 
    {
      usedSize = 0;
      allocatedSize = 2;
      data = new Xabsl2ArrayElement<T>*[allocatedSize];
    }
    
    /** Destructor */
    ~Xabsl2Array() 
    { 
      for(int i = 0; i < getSize(); ++i)
        delete data[i];
      delete[] data;
    }

    /** Clears the array */
    void clear()
    {
      for(int i = 0; i < getSize(); ++i)
        delete data[i];
      delete[] data;
      usedSize = 0;
      allocatedSize = 2;
      data = new Xabsl2ArrayElement<T>*[allocatedSize];
    }
    
    /** 
    * Returns the value for a given name. 
    * If no element exists for the name, the default value is returned.
    * @param name The name element
    * @param defaultValue The value that is returned if no element exists for the name.
    * @return Either the element found or the default value. 
    */
    T getElement(const char* name, T defaultValue) const
    {
      int pos = find(name);
      if(pos < 0) 
        return defaultValue;
      else
        return getElement(pos);
    }
    
    /** 
    * Returns the value for a given name. 
    * Note that the function crashes if the element does not exist.
    * @param name The name of the element
    */
    T getElement(const char* name) const
    {
      return getElement(find(name));
    }

    /** 
    * Returns the value for a given array position.
    * Note that the function crashes if the required position is bigger than the 
    * size of the array.
    */
    T getElement(int pos) const
    {
      return data[pos]->e;
    }
    
    /** 
    * Returns a pointer to the array element for a given name.
    * Note that the function crashes if the element does not exist
    * @param name the name of the element 
    */
    Xabsl2ArrayElement<T>* getPElement(const char* name)
    {
      return data[find(name)];
    }

    /** Returns the name of an element */
    const char* getName(int pos) const
    { 
      return data[pos]->n;
    }

    /**
    * The function appends a new element to the array.
    * @param name A string label for the element.
    * @param element The new element.
    */
    void append(const char* name, T element)
    {
      if(usedSize == allocatedSize)
      {
        allocatedSize += allocatedSize / 2; // note that allocatedSize must be at least 2
        Xabsl2ArrayElement<T>** temp = new Xabsl2ArrayElement<T>*[allocatedSize];
        for(int i = 0; i < getSize(); ++i)
          temp[i] = data[i];
        delete[] data;
        data = temp;
      }
      data[usedSize++] = new Xabsl2ArrayElement<T>(name,element);
    }
    
    /**
    * The function sets the value of an element in the array.
    * Note that the function crashes if the element does not exist.
    * @param name A string label for the element.
    * @param value The new element.
    */
    void setElement(const char* name, T value)
    {
      setElement(find(name),value);
    }

    /**
    * The function sets the value of an element in the array.
    * Note that the function crashes if the element does not exist.
    * @param pos The position of the element in the array.
    * @param value The new element.
    */
    void setElement(int pos, T value)
    {
      data[pos]->e = value;
    }

    /**
    * The function returns the number of elements in the array.
    * @return The length of the list.
    */
    int getSize() const {return usedSize;}
    
    /** 
    * Returns the value for a given array position.
    * Note that the function crashes if the required position is bigger than the 
    * size of the array.
    */
    T operator[](int pos) const
    {
      return getElement(pos);
    }
    
    /** Returns whether an element for the given name exists */
    bool exists(const char* name) const
    {
      return find(name) >= 0;
    }

  protected:
    /** 
    * Returns the index of an element with the given name.
    * @param name The name that is searched for.
    * @return The index of the element of -1 if the name does not exist.
    */
    int find(const char* name) const
    {
      for(int i = 0; i < getSize(); ++i)
        if(!strcmp(getName(i),name)) 
          return i;
      return -1;
    }

    /** The array */
    Xabsl2ArrayElement<T>** data;

    /** The number of elements in the array */
    int usedSize, allocatedSize;
};




#endif // __Xabsl2Array_h_

/*
* Change Log:
*
* $Log: Xabsl2Array.h,v $
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/03/06 11:45:55  dueffert
* re-order warning removed
*
* Revision 1.9  2003/02/05 13:31:09  loetzsch
* removed crash comment. (The bug was somewhere else)
*
* Revision 1.8  2003/02/05 12:10:03  dueffert
* crash comment added
*
* Revision 1.7  2003/01/13 13:19:14  loetzsch
* added Xabsl2Array::getPElement(const char* name) for pointer access to elements
*
* Revision 1.6  2003/01/09 10:00:21  loetzsch
* added clear() method
*
* Revision 1.5  2003/01/08 11:06:46  loetzsch
* introduced Xabsl2NamedItem
*
* Revision 1.4  2002/12/05 16:46:52  loetzsch
* Thomas Rfer replaced the linked list by an array
*
* Revision 1.3  2002/12/02 19:56:31  loetzsch
* - Xabsl2Array now seems to work for more than 1 element
* - Basic behaviors now can be registered at the engine
* - Basic behaviors have access to their parameters
*
* Revision 1.2  2002/12/01 17:54:30  loetzsch
* continued Xabsl2Engine: Xabsl2Array seems to work now
*
* Revision 1.1  2002/12/01 13:45:58  loetzsch
* first version of Xabsl2Engine
*
*/






