/**
* @file OutStreams.h
*
* Declaration of out stream classes for different media and formats.
*
* @author Thomas Rfer
* @author Martin Ltzsch
*/

#ifndef __OutStreams_h_
#define __OutStreams_h_

#include "InOut.h"
#include "Platform/File.h"
#include <string.h>

/** 
* @class PhysicalOutStream
*
* The base class for physical out streams. Derivates of PhysicalOutStream only handle the 
* writing of data to a medium, not of formating data.
*/
class PhysicalOutStream
{
public:
/**
* The function writes a number of bytes into a physical stream.
* @param p The address the data is located at.
* @param size The number of bytes to be written.
  */
  virtual void writeToStream(const void* p,int size) = 0;

};

/**
* @class StreamWriter
*
* Generic class for formating data to be used in streams. 
* The physical writing is then done by OutStream derivates.
*/
class StreamWriter
{
protected:
/**
* Writes a character to a stream.
* @param d the data to write.
* @param stream the stream to write on.
  */
  virtual void writeChar(char d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a unsigned character to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUChar(unsigned char d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeShort(short d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a unsigned short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUShort(unsigned short d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeInt(int d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a unsigned int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUInt(unsigned int d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeLong(long d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a unsigned long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeULong(unsigned long d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a float to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeFloat(float d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a double to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeDouble(double d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a string to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeString(const char *d, PhysicalOutStream& stream) = 0;
  
  /**
  * Writes a 'end of line' to a stream.
  * @param stream the stream to write on.
  */
  virtual void writeEndL(PhysicalOutStream& stream) = 0;
  
  /**
  * The function writes a number of bytes into the stream.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  * @param stream the stream to write on.
  */
  virtual void writeData(const void* p,int size, PhysicalOutStream& stream) = 0;
};


/**
* @class OutStream
*
* Generic class for classes that do both formating and physical writing of data to streams.
*/
template <class S, class W> class OutStream : public S, public W, public Out
{
public:
  /** Standard constructor */
  OutStream() {};
  
  /**
  * The function writes a number of bytes into a stream.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  */
  virtual void write(const void* p,int size)
  { writeData(p, size, *this); }
  
protected:
/**
* Virtual redirection for operator<<(const char& value).
  */
  virtual void outChar(char d)
  { writeChar(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const unsigned char& value).
  */
  virtual void outUChar(unsigned char d)
  { writeUChar(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const short& value).
  */
  virtual void outShort(short d)
  { writeShort(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const unsigned short& value).
  */
  virtual void outUShort(unsigned short d)
  { writeUShort(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const int& value).
  */
  virtual void outInt(int d)
  { writeInt(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const unsigned& value).
  */
  virtual void outUInt(unsigned int d)
  { writeUInt(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const long& value).
  */
  virtual void outLong(long d)
  { writeLong(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const unsigned long& value).
  */
  virtual void outULong(unsigned long d)
  { writeULong(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const float& value).
  */
  virtual void outFloat(float d)
  { writeFloat(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const double& value).
  */
  virtual void outDouble(double d)
  { writeDouble(d,*this); }
  
  /**
  * Virtual redirection for operator<<(const char* value).
  */
  virtual void outString(const char *d)
  { writeString(d,*this); }
  
  
  /**
  * Virtual redirection for operator<<(Out& (*f)(Out&)) that writes
  * the symbol "endl";
  */
  virtual void outEndL()
  { writeEndL(*this); }
};


/**
* @class OutBinary
*
* Formats data binary to be used in streams. 
* The physical writing is then done by OutStream derivates.
*/
class OutBinary : public StreamWriter
{
protected:
/**
* Writes a character to a stream.
* @param d the data to write.
* @param stream the stream to write on.
  */
  virtual void writeChar(char d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a unsigned character to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUChar(unsigned char d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeShort(short d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a unsigned short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUShort(unsigned short d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeInt(int d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a unsigned int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUInt(unsigned int d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeLong(long d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a unsigned long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeULong(unsigned long d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a float to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeFloat(float d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a double to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeDouble(double d, PhysicalOutStream& stream)
  { stream.writeToStream(&d,sizeof(d)); }
  
  /**
  * Writes a string to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeString(const char *d, PhysicalOutStream& stream) 
  { int size = (int)strlen(d); stream.writeToStream(&size,sizeof(size)); stream.writeToStream(d,size);}
  
  /**
  * Writes a 'end of line' to a stream.
  * @param stream the stream to write on.
  */
  virtual void writeEndL(PhysicalOutStream& stream) {};
  
  /**
  * The function writes a number of bytes into the stream.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  * @param stream the stream to write on.
  */
  virtual void writeData(const void* p,int size, PhysicalOutStream& stream) 
  { stream.writeToStream(p,size); }
  
};

/**
* @class OutText
*
* Formats data as text to be used in streams. 
* The physical writing is then done by PhysicalOutStream derivates.
*/
class OutText : public StreamWriter
{
private:
  /** A buffer for formatting the numeric data to a text format. */
  char buf[50]; 
protected:
/**
* Writes a character to a stream.
* @param d the data to write.
* @param stream the stream to write on.
  */
  virtual void writeChar(char d, PhysicalOutStream& stream)
  { sprintf(buf," %d",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned character to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUChar(unsigned char d, PhysicalOutStream& stream)
  { sprintf(buf," %u",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeShort(short d, PhysicalOutStream& stream)
  { sprintf(buf," %d",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUShort(unsigned short d, PhysicalOutStream& stream)
  { sprintf(buf," %u",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeInt(int d, PhysicalOutStream& stream)
  { sprintf(buf," %d",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUInt(unsigned int d, PhysicalOutStream& stream)
  { sprintf(buf," %u",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeLong(long d, PhysicalOutStream& stream)
  { sprintf(buf," %ld",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeULong(unsigned long d, PhysicalOutStream& stream)
  { sprintf(buf," %lu",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a float to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeFloat(float d, PhysicalOutStream& stream)
  { sprintf(buf," %g",double(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a double to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeDouble(double d, PhysicalOutStream& stream)
  { sprintf(buf," %g",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a string to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeString(const char *d, PhysicalOutStream& stream); 
  
  /**
  * Writes a 'end of line' to a stream.
  * @param stream the stream to write on.
  */
  virtual void writeEndL(PhysicalOutStream& stream) 
  { sprintf(buf,"\r\n"); stream.writeToStream(buf,(int)strlen(buf)); } 
  
  /**
  * The function writes a number of bytes into the stream.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  * @param stream the stream to write on.
  */
  virtual void writeData(const void* p,int size, PhysicalOutStream& stream);
};

/**
* @class OutTextRaw
*
* Formats data as raw text to be used in streams. 
* The physical writing is then done by PhysicalOutStream derivates.
* Different from OutText, OutTextRaw does not escape spaces
* and other special characters and no spaces are inserted before numbers.
* (The result of the OutTextRaw StreamWriter is the same as the result of "std::cout")
*/
class OutTextRaw : public StreamWriter
{
private:
  /** A buffer for formatting the numeric data to a text format. */
  char buf[50]; 
protected:
/**
* Writes a character to a stream.
* @param d the data to write.
* @param stream the stream to write on.
  */
  virtual void writeChar(char d, PhysicalOutStream& stream)
  { sprintf(buf,"%d",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned character to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUChar(unsigned char d, PhysicalOutStream& stream)
  { sprintf(buf,"%u",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeShort(short d, PhysicalOutStream& stream)
  { sprintf(buf,"%d",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned short to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUShort(unsigned short d, PhysicalOutStream& stream)
  { sprintf(buf,"%u",int(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeInt(int d, PhysicalOutStream& stream)
  { sprintf(buf,"%d",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned int to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeUInt(unsigned int d, PhysicalOutStream& stream)
  { sprintf(buf,"%u",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeLong(long d, PhysicalOutStream& stream)
  { sprintf(buf,"%ld",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a unsigned long to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeULong(unsigned long d, PhysicalOutStream& stream)
  { sprintf(buf,"%lu",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a float to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeFloat(float d, PhysicalOutStream& stream)
  { sprintf(buf,"%g",double(d)); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a double to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeDouble(double d, PhysicalOutStream& stream)
  { sprintf(buf,"%g",d); stream.writeToStream(buf,(int)strlen(buf)); }
  
  /**
  * Writes a string to a stream.
  * @param d the data to write.
  * @param stream the stream to write on.
  */
  virtual void writeString(const char *d, PhysicalOutStream& stream); 
  
  /**
  * Writes a 'end of line' to a stream.
  * @param stream the stream to write on.
  */
  virtual void writeEndL(PhysicalOutStream& stream) 
  { sprintf(buf,"\r\n"); stream.writeToStream(buf,(int)strlen(buf)); } 
  
  /**
  * The function writes a number of bytes into the stream.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  * @param stream the stream to write on.
  */
  virtual void writeData(const void* p,int size, PhysicalOutStream& stream);
};



/**
* @class OutFile.
*
* An PhysicalOutStream that writes the data to a file.
*/
class OutFile : public PhysicalOutStream
{
private:
  File* stream; /**< Object representing the file. */
  
public:
  /** Default constructor */
  OutFile() : stream(0) {};
  
  /** Destructor */
  ~OutFile() { if (stream != 0) delete stream; }
  
  /**
  * The function states whether the file actually exists.
  * @return Does the file exist?
  */
  virtual bool exists() const {return (stream != 0 ? stream->exists() : false);}
  
protected:
  /**
  * Opens the stream.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be discared.
  */
  void open(const char* name) 
  { stream=new File(name,"wb"); }
  
  /**
  * Opens the stream.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be preserved, 
  *             if append = true.
  * @param append Determines, if the file content is preserved or discared.
  */
  void open(const char* name, bool append) 
  { stream = append ? new File(name,"ab") : new File(name,"wb");}

  /**
  * The function writes a number of bytes into the file.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  */
  virtual void writeToStream(const void *p,int size) 
  { if (stream!=0) stream->write(p,size); }
};

/**
* @class OutMemory.
*
* A  PhysicalOutStream that writes the data to a memory block.
*/
class OutMemory : public PhysicalOutStream
{
private:
  char* memory; /**< Points to the next byte to write at. */
  int length; /**< The number of stored bytes */
  void* start; /**< Points to the first byte */
  
public:
  /** Default constructor */
  OutMemory() : memory(0), length(0), start(0) {}
  
  /**
  * Returns the number of written bytes 
  */
  int getLength() { return length; }
  
  /** 
  * Returns the address of the first byte 
  */
  void* getMemory() { return start; }
  
protected:
/**
* opens the stream.
* @param mem The address of the memory block into which is written.
  */
  void open(void* mem) 
  { memory = (char*)mem; start = mem; length = 0;}
  
  /**
  * The function writes a number of bytes into memory.
  * @param p The address the data is located at.
  * @param size The number of bytes to be written.
  */
  virtual void writeToStream(const void *p, int size) 
  { if(memory != 0) { memcpy(memory,p,size); memory += size; length += size; } }
};

/**
* @class OutSize.
*
* A PhysicalOutStream that doesn't write any data. Instead it works as a
* counter that determines how many bytes would be streamed into memory.
* It can be used to determine the size of the memory block that is
* required as the parameter for an instance of class OutMemoryStream.
*/
class OutSize : public PhysicalOutStream
{
private:
  unsigned size; /**< Accumulator of the required number of bytes. */
  
public:
/**
* The function resets the counter to zero.
  */
  void reset() {size = 0;}
  
  /**
  * Constructor.
  */
  OutSize() {reset();}
  
  /**
  * The function returns the number of bytes required to store the
  * data written so far.
  * @return The size of the memory block required for the data written.
  */
  unsigned getSize() const {return size;}
  
protected:
/**
* The function counts the number of bytes that should be written.
* @param p The address the data is located at.
* @param s The number of bytes to be written.
  */
  virtual void writeToStream(const void* p,int s) {size += s;}
  
};

/**
* @class OutBinaryFile
* 
* A binary stream into a file.
*/
class OutBinaryFile : public OutStream<OutFile,OutBinary>
{
public:
  /**
  * Constructor.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be discared.
  */
  OutBinaryFile(const char* name)
  { open(name); }
  
  /**
  * Constructor.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be preserved,
  *             if append = true.
  * @param append Determines, if the file content is preserved or discared. 
  */
  OutBinaryFile(const char* name, bool append)
  { open(name, append); }
};

/**
* @class OutBinaryMemory
* 
* A binary stream into a memory region.
*/
class OutBinaryMemory : public OutStream<OutMemory,OutBinary>
{
public:  
/**
* Constructor.
* @param mem The address of the memory block into which is written.
  */
  OutBinaryMemory(void* mem) 
  { open(mem); }
};

/** 
* @class OutBinarySize
*
* A binary stream size counter
*/
class OutBinarySize : public OutStream<OutSize,OutBinary>
{
public:
/**
* Constructor.
  */
  OutBinarySize() {}
};

/**
* @class OutTextFile
* 
* A text stream into a file.
*/
class OutTextFile : public OutStream<OutFile,OutText>
{
public:
  /**
  * Constructor.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be discared.
  */
  OutTextFile(const char* name) 
  { open(name); }

  /**
  * Constructor.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be preserved,
  *             if append = true.
  * @param append Determines, if the file content is preserved or discared. 
  */
  OutTextFile(const char* name, bool append) 
  { open(name, append); }
};

/**
* @class OutTextRawFile
* 
* A text stream into a file.
*/
class OutTextRawFile : public OutStream<OutFile,OutTextRaw>
{
public:
  /**
  * Constructor.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be discared.
  */
  OutTextRawFile(const char* name) 
  { open(name); }

  /**
  * Constructor.
  * @param name The name of the file to open. It will be interpreted
  *             as relative to the configuration directory. If the file 
  *             does not exist, it will be created. If it already 
  *             exists, its previous contents will be preserved,
  *             if append = true.
  * @param append Determines, if the file content is preserved or discared. 
  */
  OutTextRawFile(const char* name, bool append) 
  { open(name, append); }
};

/**
* @class OutTextMemory
* 
* A text stream into a memory region.
*/
class OutTextMemory : public OutStream<OutMemory,OutText>
{
public:
/**
* Constructor.
* @param mem The address of the memory block into which is written.
  */
  OutTextMemory(void* mem) 
  { open(mem); }
};

/**
* @class OutTextRawMemory
* 
* A text stream into a memory region.
*/
class OutTextRawMemory : public OutStream<OutMemory,OutTextRaw>
{
public:
/**
* Constructor.
* @param mem The address of the memory block into which is written.
  */
  OutTextRawMemory(void* mem) 
  { open(mem); }
};

/** 
* @class OutTextSize
*
* A Text stream size counter
*/
class OutTextSize : public OutStream<OutSize,OutText>
{
public:
/**
* Constructor.
  */
  OutTextSize() {}
};

/** 
* @class OutTextRawSize
*
* A Text stream size counter
*/
class OutTextRawSize : public OutStream<OutSize,OutTextRaw>
{
public:
/**
* Constructor.
  */
  OutTextRawSize() {}
};



#endif //__OutStreams_h_

/*
* Change Log:
*
* $Log: OutStreams.h,v $
* Revision 1.6  2004/01/28 20:52:33  loetzsch
* Added the OutTextRaw stream writer.
*
* Revision 1.5  2004/01/22 12:38:12  dueffert
* flush removed because it doesnt work properly on robots
*
* Revision 1.4  2004/01/16 15:46:47  dueffert
* flush added to File and OutFile
*
* Revision 1.3  2003/11/30 01:53:21  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.2  2003/11/29 07:53:53  roefer
* sprintf using %f resulted in buffer overflows. Now using %g.
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/03/06 11:55:00  dueffert
* parameter type warnings removed
*
* Revision 1.4  2003/01/23 22:17:44  cesarz
* added append mode also for OutBinaryFile.
*
* Revision 1.3  2003/01/23 14:22:26  cesarz
* added append support for OutTextFile
*
* Revision 1.2  2002/10/14 13:14:24  dueffert
* doxygen comments corrected
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/07/23 13:46:43  loetzsch
* - new streaming classes
*
*/
