/**
* @file PfieldGeometry.h
* 
* Definition of several geometric functions and objects used by potential fields
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef PFIELDGEOMETRY_H_
#define PFIELDGEOMETRY_H_


#include "PfieldDatatypes.h"
#include "PotentialFunctions.h"

class Polygon;
class PfieldGeometricObject;
class Line;
class Circle;


/** Definition of the different types of geometric objects*/
enum GeometryType {POLYGON, LINE, CIRCLE, NONE};


/** Determines if a point is perpendicular to a line
* @param l1 One end of the line
* @param l2 The other end of the line
* @param p The point
* @param distance The distance to the line, if perpendicular
* @param perpendicularPoint The point on the line, if perpendicular
* @return true, if point is perpendicular.
*/
bool pointPerpendicularToLine(const PfVec& l1, const PfVec& l2, const PfVec& p,
                              double& distance, PfVec& perpendicularPoint);


/** Reduces the points of a polygon to its convex hull
* @param p The polygon
*/
void reduceToConvexHullByWrapping(Polygon& p);


/** Intersects two geometric objects and returns the intersection points as
* well as significant points of the second object inside the intersection
* @param g1 The first geometric object
* @param g2 The second geometric object
* @param intersections The intersections
*/
void intersectGeometricObjects(PfieldGeometricObject* g1, PfieldGeometricObject* g2, 
                               std::vector<PfVec>& intersections);


/** Intersects a line with another line 
*   (basic intersection checking has to be done before)
* @param l1 A line
* @param l2 Another line
* @param intersections The intersections found
*/
void intersectLineAndLine(const Line& l1, const Line& l2, 
                          std::vector<PfVec>& intersections);

/** Intersects a line with a circle
*   (basic intersection checking has to be done before)
* @param line A line
* @param circle A circle
* @param intersections The intersections found
*/
void intersectLineAndCircle(const Line& line, const Circle& circle, 
                            std::vector<PfVec>& intersections);

/** Intersects a polygon with a circle
*   (basic intersection checking has to be done before)
* @param polygon A polygon
* @param circle A circle
* @param intersections The intersections found
*/
void intersectPolygonAndCircle(const Polygon& polygon, const Circle& circle, 
                              std::vector<PfVec>& intersections);

/** Intersects a circle with a circle
*   (basic intersection checking has to be done before)
* @param circle1 A circle
* @param circle2 Another circle
* @param intersections The intersections found
*/
void intersectCircleAndCircle(const Circle& circle1, const Circle& circle2, 
                              std::vector<PfVec>& intersections);


/**
* @class PfieldGeometricObject
*
* Abstract base class for geometric objects used by potentialfields
*/
class PfieldGeometricObject
{
public:
  /** The radius of a circle around the whole object*/
  double radiusOfCollisionCircle;
  /** Flag, true if object is intersectable by other objects*/
  bool intersectable;
  /** The position of the object, only used with absolute positions*/
  PfVec position;

  /** Virtual destructor */
  virtual ~PfieldGeometricObject() {}
  
  /** Returns the type of a geometric object
  * @return The type
  */
  virtual GeometryType getType() const = 0;

  /** Clones a geometric object
  * @return A pointer to a copy of the object
  */
  virtual PfieldGeometricObject* clone() const = 0;

  /** Computes the distance from the border of the object to a given point
  * @param base The pose of the object to which the geometric object is assigned
  * @param pos The tested point
  * @param contact Returns the next position of the geometric object to the point
  * @return The distance
  */
  virtual double distanceTo(const PfPose& base, const PfVec& pos, PfVec& contact) const = 0;

  /** Computes value of the radiusOfCollisionCircle variable */
  virtual void initRadiusOfCollisionCircle() = 0;

  /** Returns the value of radiusOfCollisionCircle
  * @return The radius
  */
  double getRadiusOfCollisionCircle() const
  { return radiusOfCollisionCircle;}

  /** Returns the position
  * @return The position
  */
  PfVec getPosition() const
  { return position;}

  /** Returns a geometric object with absolute coordinates
  * @param base The pose of the object
  * @return A pointer to a new geometric object
  */
  virtual PfieldGeometricObject* getAbs(const PfPose& base) const = 0;

  /** Returns all points of the object
  * @param points The list of points to which the object's points will be attached
  */
  virtual void getPoints(std::vector<PfVec>& points){}

  /** Computes absolute coordinates from a base pose and another, relative
  * polygon. Use careful! The other object has to be the same type and
  * the size! Only positions are changed, other members will be ignored!
  * @param base The new absolute position
  * @param other A relative geometric object 
  */
  virtual void setAbsoluteFromOther(const PfPose& base, PfieldGeometricObject* other) = 0;
};


/**
* @class Polygon
*
* A class representing a convex polygon
*/
class Polygon: public PfieldGeometricObject
{
public:
  /** A list of points.*/
  std::vector<PfVec> pts;

  /** Adds a point to the polygon
  * @param p The point
  */
  void addPoint(const PfVec& p)
  {pts.push_back(p);}

  /** Returns the type of a geometric object
  * @return The type
  */
  GeometryType getType() const
  { return POLYGON;}

  /** Clones a polygon
  * @return A pointer to a copy of the object
  */
  PfieldGeometricObject* clone() const;

  /** Computes the distance from the border of the object to a given point
  * @param base The pose of the object to which the geometric object is assigned
  * @param pos The tested point
  * @param contact Returns the next position of the geometric object to the point
  * @return The distance
  */
  double distanceTo(const PfPose& base, const PfVec& pos, PfVec& contact) const;

  /** Computes value of the radiusOfCollisionCircle variable */
  virtual void initRadiusOfCollisionCircle();

  /** Returns a geometric object with absolute coordinates
  * @param base The pose of the object
  * @return A pointer to a new geometric object
  */
  PfieldGeometricObject* getAbs(const PfPose& base) const;

  /** Returns all points of the object
  * @param points The list of points to which the object's points will be attached
  */
  virtual void getPoints(std::vector<PfVec>& points);

  /** Tests if a point is inside the polygon
  * @param point The point
  * @return true, if the point is inside
  */
  bool pointInside(const PfVec& point) const;

  /** Computes absolute coordinates from a base pose and another, relative
  * polygon. Use careful! The other object has to be the same type and
  * the size! Only positions are changed, other members will be ignored!
  * @param base The new absolute position
  * @param other A relative geometric object 
  */
  void setAbsoluteFromOther(const PfPose& base, PfieldGeometricObject* other);
};


/**
* @class Line
*
* A class representing a line
*/
class Line: public PfieldGeometricObject
{
public:
  /** One end point of the line */
  PfVec p1;
  /** The other end point of the line */
  PfVec p2;

  /** Returns the type of a geometric object
  * @return The type
  */
  GeometryType getType() const
  { return LINE;}

  /** Clones a line
  * @return A pointer to a copy of the object
  */
  PfieldGeometricObject* clone() const;

  /** Computes the distance from the border of the object to a given point
  * @param base The pose of the object to which the geometric object is assigned
  * @param pos The tested point
  * @param contact Returns the next position of the geometric object to the point
  * @return The distance
  */
  double distanceTo(const PfPose& base, const PfVec& pos, PfVec& contact) const;

  /** Computes value of the radiusOfCollisionCircle variable */
  virtual void initRadiusOfCollisionCircle();

  /** Returns a geometric object with absolute coordinates
  * @param base The pose of the object
  * @return A pointer to a new geometric object
  */
  PfieldGeometricObject* getAbs(const PfPose& base) const;

  /** Computes absolute coordinates from a base pose and another, relative
  * polygon. Use careful! The other object has to be the same type and
  * the size! Only positions are changed, other members will be ignored!
  * @param base The new absolute position
  * @param other A relative geometric object 
  */
  void setAbsoluteFromOther(const PfPose& base, PfieldGeometricObject* other);

  /** Returns all points of the object
  * @param points The list of points to which the object's points will be attached
  */
  virtual void getPoints(std::vector<PfVec>& points);
};


/**
* @class Circle
*
* A class representing a circle
*/
class Circle: public PfieldGeometricObject
{
public:
  /** The radius of the circle */
  double radius;

  /** Returns the type of a geometric object
  * @return The type
  */
  GeometryType getType() const
  { return CIRCLE;}

  /** Clones a circle
  * @return A pointer to a copy of the object
  */
  PfieldGeometricObject* clone() const;

  /** Computes the distance from the border of the object to a given point
  * @param base The pose of the object to which the geometric object is assigned
  * @param pos The tested point
  * @param contact Returns the next position of the geometric object to the point
  * @return The distance
  */
  double distanceTo(const PfPose& base, const PfVec& pos, PfVec& contact) const;

  /** Computes value of the radiusOfCollisionCircle variable */
  virtual void initRadiusOfCollisionCircle();

  /** Returns a geometric object with absolute coordinates
  * @param base The pose of the object
  * @return A pointer to a new geometric object
  */
  PfieldGeometricObject* getAbs(const PfPose& base) const;

  /** Computes absolute coordinates from a base pose and another, relative
  * polygon. Use careful! The other object has to be the same type and
  * the size! Only positions are changed, other members will be ignored!
  * @param base The new absolute position
  * @param other A relative geometric object 
  */
  void setAbsoluteFromOther(const PfPose& base, PfieldGeometricObject* other);

  /** Returns all points of the object
  * @param points The list of points to which the object's points will be attached
  */
  virtual void getPoints(std::vector<PfVec>& points);
};


/**
* @class NoGeometry
*
* A class representing an empty geometric object
*/
class NoGeometry: public PfieldGeometricObject
{
public:
  /** Returns the type of a geometric object
  * @return The type
  */
  GeometryType getType() const
  { return NONE;}

  /** Clones an empty geometric object
  * @return A pointer to a copy of the object
  */
  PfieldGeometricObject* clone() const;

  /** Computes the distance from the border of the object to a given point
  * @param base The pose of the object to which the geometric object is assigned
  * @param pos The tested point
  * @param contact Returns the next position of the geometric object to the point
  * @return The distance
  */
  double distanceTo(const PfPose& base, const PfVec& pos, PfVec& contact) const;

  /** Computes value of the radiusOfCollisionCircle variable */
  virtual void initRadiusOfCollisionCircle()
  { radiusOfCollisionCircle = 0.0;}

  /** Returns a geometric object with absolute coordinates
  * @param base The pose of the object
  * @return A pointer to a new geometric object
  */
  PfieldGeometricObject* getAbs(const PfPose& base) const;

  /** Computes absolute coordinates from a base pose and another, relative
  * polygon. Use careful! The other object has to be the same type and
  * the size! Only positions are changed, other members will be ignored!
  * @param base The new absolute position
  * @param other A relative geometric object 
  */
  void setAbsoluteFromOther(const PfPose& base, PfieldGeometricObject* other);
};


/**
* @class Sector
*
* A class representing a sector
*/
class Sector
{
public:
  /** The opening angle of the sector*/
  double openingAngle;
  /** The function used to compute the gradient to the center of the sector*/
  PotentialfieldFunction* crossFunction;

  /** Constructor */
  Sector()
  { crossFunction = 0;}

  /** Destructor*/
  ~Sector()
  { 
    if(crossFunction != 0) 
      delete crossFunction;
  }

  /** Tests if a point is in the sector
  * @param base The pose of the object to which the sector is assigned
  * @param pos The point to be tested
  * @return true, if the point is inside
  */
  bool pointInside(const PfPose& base, const PfVec& pos) const;
};


#endif //PFIELDGEOMETRY_H_


/*
* $Log: PfieldGeometry.h,v $
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.5  2003/05/20 12:43:43  tim
* Changed function representation, fixed crash on SuperCore, integrated social functions
*
* Revision 1.4  2003/04/22 14:35:17  tim
* Merged changes from GO
*
* Revision 1.4  2003/04/09 19:03:06  tim
* Last commit before GermanOpen
*
* Revision 1.3  2003/04/04 14:50:53  tim
* Fixed bugs, added minor features
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
