/**
* @file Motionfield.h
* 
* Definition of class Motionfield
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef MOTIONFIELD_H_
#define MOTIONFIELD_H_


#include "PfieldConfig.h"
#include "AStarSearch.h"
#include "Pfield.h"
#include "RandomMotionGenerator.h"


/** A type for A* search in a potential field*/
typedef AStarSearch<PotentialfieldAStarNode,PotentialfieldAStarParameterSet,double> PfieldAStarSearch;


/**
* @class Motionfield
*
* A class for computing a motion by using a potential field
*/
class Motionfield : public Potentialfield
{
public:
  /** Constructor */
  Motionfield(const std::string& name);

  /** Destructor */
  virtual ~Motionfield();

  /** Computes the result of a field
  * @param pose The pose on which the field affects
  * @param result The returned result
  */
  void execute(const PfPose& pose, PotentialfieldResult& result);

  /** Activates or deactivates the degrees of freedom of the field
  * @param translation Deactivates translation if true
  * @param rotation Deactivates rotation if true
  */
  void disableDegreesOfFreedom(bool translation, bool rotation)
  { translationDisabled = translation; rotationDisabled = rotation;}

  /** Sets the acceleration limits
  * @param maxAccel The maximum acceleration per second
  * @param maxGradientDifference Maximum difference to last gradient in radiant per second
  */
  void setAccelerationLimits(double maxAccel, double maxGradientDifference)
  { this->maxAccel = maxAccel; this->maxGradientDifference = maxGradientDifference;};

  /** Adds a random device to this field
  * @param randomMotionGenerator A random motion generator
  */
  void addRandomMotionGenerator(RandomMotionGenerator* randomMotionGenerator)
  { this->randomMotionGenerator = randomMotionGenerator;}

  /** Adds a path planner to this a field
  * @param pathPlanner The path planner
  * @param useAlways Flag: use always the path planner, if true
  * @param goal The object to plan a path to
  * @param parameterSet The parameter set for path planning
  * @param maxGradientLength The maximum allowed gradient length for not using path planning
  */
  void addPathPlanner(
      PfieldAStarSearch* pathPlanner, bool useAlways, Object* goal, 
      const PotentialfieldAStarParameterSet& parameterSet, double maxGradientLength)
  { 
    this->pathPlanner = pathPlanner; 
    alwaysUsePathPlanner = useAlways;
    this->goal = goal;
    this->aStarParameterSet = parameterSet;
    this->aStarParameterSet.field = this;
    this->maxGradientLengthForLocalMinimum = maxGradientLength;
  }

  /** Computes a random vector
  * @return A random vector
  */
  PfVec getRandomVector()
  { 
    if(randomMotionGenerator) 
      return randomMotionGenerator->getMotionVector();
    else 
      return PfVec(0.0,0.0);
  }

protected:
  /** Flag: translation is disabled or not*/
  bool translationDisabled;
  /** Flag: rotation is disabled or not*/
  bool rotationDisabled;
  /** Maximum acceleration in vector lengths per square second*/
  double maxAccel;
  /** Maximum difference to last gradient in radiant per second*/
  double maxGradientDifference;
  /** A pointer to a random motion generator*/
  RandomMotionGenerator* randomMotionGenerator;
  /** A pointer to an A* path planner*/
  PfieldAStarSearch* pathPlanner;
  /** The parameter set for path planning*/
  PotentialfieldAStarParameterSet aStarParameterSet;
  /** Flag, true, if the pathPlanner is always used*/
  bool alwaysUsePathPlanner;
  /** Flag, true if the pathPlanner is currently used*/
  bool pathPlannerActive;
  /** The goal for planning*/
  Object* goal;
  /** The length of the path generated by the planner*/
  double plannedPathLengthToGoal;
  /** The upper limit for the gradient length to detect a local minimum*/
  double maxGradientLengthForLocalMinimum;
  /** The last computed motion vector*/
  PfVec lastMotionVector;

  /** Computes a motion vector using A* search
  * @param pose The beginning of the path
  * @return A vector
  */
  PfVec getFieldVecFromAStarSearch(const PfPose& pose);

  /** Checks if the field has reached a local minimum
  * @param currentGradientLength The length of the current gradient vector
  * @return true, if a local minimum has been reached
  */
  bool reachedLocalMinimum(double currentGradientLength);

  /** Checks if the path planner may be turned off
  * @param pose The beginning of the path
  * @return false, if the path planner may be turned off
  */
  bool pathPlanningStillNeeded(const PfPose& pose);

   /** Corrects computed motion vector considering maximum acceleration values
  * @param motionVec The vector to be corrected
  */
  void correctMotionVector(PfVec& motionVec) const;
};


#endif	//MOTIONFIELD_H_



/*
* $Log: Motionfield.h,v $
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.3  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
* Revision 1.2  2003/04/02 14:39:11  tim
* Changed Bremen Byters Behavior
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
