/**
* @file Actionfield.h
* 
* Definition of class Actionfield
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef ACTIONFIELD_H_
#define ACTIONFIELD_H_


#include "PfieldConfig.h"
#include "PfieldDatatypes.h"
#include "Pfield.h"
#include "PfieldGeometry.h"
#include <vector>
#include <cfloat>

class FutureWorldModelGenerator;
class PotentialfieldTransformation;
class Object;


/** A type for setting the type of an Actionfield*/
enum ActionfieldType {SINGLE_ACTION_FIELD, FIXED_SEQUENCE_FIELD, BEST_SEQUENCE_FIELD};

/** A type for setting the type of an Action*/
enum ActionType {MOVE_OBJECT, MOVE_SELF, MEASURE_OBJECT, MEASURE_SELF};

/** The possible transformations*/
enum TransformationType {TRANSLATION, ROTATION, NO_TRANSFORMATION};


/**
* @class Action
*
* A class describing a single action
*/
class Action
{
public:
  /** The name of the action*/
  std::string name;
  /** The manipulated object*/
  Object* manipulatedObject;
  /** Index of the manipulated object in a dynamicObjects field (for internal use)*/
  unsigned int objectIdx;
  /** Flag, true if the robot joins the action*/
  bool joinAction;
  /** The impact areas*/
  std::vector<Polygon> impactAreas;
  /** The transformations*/
  std::vector<PotentialfieldTransformation*> transformations;
  /** The type of the action*/
  ActionType actionType;
  /** Motion parameters resulting from some transformations*/
  PfPose motionParameters;
  /** The time this action takes*/
  double time;

  /** Constructor */
  Action();

  /** Destructor */
  ~Action();

  /** Copy-Constructor
  * @param a Another action
  */
  Action(const Action& a);

  /** Checks if this Action can be applied
  * @param robotPose The current pose of the robot
  * @param objects A list of objects, containing a current version of the manipulated object
  */
  bool canBeApplied(const PfPose& robotPose,
                    const std::vector<Object*>& objects);

protected:
  /** Checks if a single pose is inside any of the robot's impact areas
  * @param robPose The robot pose
  * @param objPose The pose of an object
  * @return true, if the pose is inside any impact area
  */
  bool poseInsideImpactArea(const PfPose& robPose, const PfPose& objPose);
};



/**
* @class Actionfield
*
* A class for selecting an action using potentialfields
*/
class Actionfield : public Potentialfield
{
public:
  /** Constructor */
  Actionfield(const std::string& name);

  /** Destructor */
  virtual ~Actionfield();

  /** Initializes values and / or allocates additional memory for subsequent computations.
      This function should be called after all values are set and all objects and
      actions have been assigned */
  virtual void init();

  /** Computes the result of a field
  * @param pose The pose of the robot on which the field affects
  * @param result The returned result
  */
  void execute(const PfPose& pose, PotentialfieldResult& result);

  /** Adds an object to the field
  * @param object The object
  */
  virtual void addObject(Object* object);
  
  /** Adds an action to the field
  * @param action The action
  */
  void addAction(const Action& action);

  /** Sets the type of the action field
  * @param actionfieldType The type
  * @param decreasingValuesOnly Pruning option, used for sequences
  * @param searchDepth The searchDepth used if (actionfieldType==FIND_BEST_SEQUENCE)
  */
  void setActionfieldType(ActionfieldType actionfieldType, 
                          bool decreasingValuesOnly=false,
                          unsigned int searchDepth=0)
  { this->actionfieldType = actionfieldType;
    worldStateDepth = searchDepth;
    this->decreasingValuesOnly = decreasingValuesOnly;}

  /** Sets the value of considerTime
  * @param considerTime The new value for considerTime
  */
  void setConsiderTime(bool considerTime)
  { this->considerTime = considerTime;}

  /** Computes the gradient at a spezific pose in a field consisting of all
  *   static objects and all the other, possibly transformed, objects.
  * @param pose The pose to compute the gradient at
  * @param dynamicObjects A set of objects
  * @param excludedDynamicObject An object not to use for gradient computation
  * @return The gradient at pose
  */
  PfVec getFutureFieldVecAt(const PfPose& pose, const std::vector<Object*>& dynamicObjects,
                            int excludedDynamicObject = -1);

protected:
  /** Container for static objects*/
  std::vector<Object*> staticObjects;
  /** Container for dynamic objects*/
  std::vector<Object*> dynamicObjects;
  /** The future world states*/
  std::vector< std::vector<Object*> > futureWorldStates;
  /** The future poses of the robot*/
  std::vector< PfPose > futureRobotPoses;
  /** The actions to be applied*/
  std::vector<Action> actions;
  /** A pointer to the FutureWorldModelGenerator*/
  FutureWorldModelGenerator* futureWorldModelGenerator;
  /** The type of this action field*/
  ActionfieldType actionfieldType;
  /** The depth of future world states*/
  unsigned int worldStateDepth;
  /** Pruning option, accept only sequences with decreasing values*/
  bool decreasingValuesOnly;
  /** Pruning option, maximum deviation between two action values*/
  double maxTolerance;
  /** Flag: true, if the time of an action when computing its value is to be considered*/
  bool considerTime;

  /** Computes the value at a spezific pose in a field consisting of all
  *   static objects and all the other, possibly transformed, objects.
  * @param pose The pose to compute the value at
  * @param dynamicObjects A set of objects
  * @param excludedDynamicObject An object not to use for value computation
  * @return The value at pose
  */
  double computeValueAtPose(const PfPose& pose, const std::vector<Object*>& dynamicObjects,
                            int excludedDynamicObject = -1);

  /** Computes the value of an action considering the criterion
  * @param action The action
  * @param time The execution of the action / action sequence
  * @param ownPoseBefore The robot pose before the action
  * @param ownPoseAfter The robot pose after the action
  * @param objectsBefore The state of the objects before the action
  * @param objectsAfter The state of the objects after the action
  * @param passedPruningCheck Flag: true, if the action has not to be pruned
  * @return The value
  */
  double computeActionValue(const Action& action, double time,
                            const PfPose& ownPoseBefore,
                            const PfPose& ownPoseAfter,
                            const std::vector<Object*>& objectsBefore,
                            const std::vector<Object*>& objectsAfter,
                            bool& passedPruningCheck);

  /** Adds a manipulated object to the object lists (if it has not been added before) and
  *   sets the objectIdx member of the action
  * @param action The action manipulating the object
  */
  void addManipulatedObject(Action& action);
  
  /** Searches recursively for the best possible action sequence
  *   and sets the result
  * @param depth The current depth
  * @param robotPose The pose of the robot before any actions
  * @param actionSequenceList A list with the sequence of actions
  * @param time The time of the previous sequence
  * @param previousValue The previous value of the sequence
  * @param result The currently best result of the search
  */
  void findBestSequence(unsigned int depth, const PfPose& robotPose,
                        std::vector<unsigned int>& actionSequenceList, 
                        double time, double previousValue,
                        PotentialfieldResult& result);
};



/**
* @class PotentialfieldTransformation
*
* Abstract class, describing a transformation
*/
class PotentialfieldTransformation
{
public:
  /** The probability of a transformation*/
  double probability;
  /** The time a transformation takes*/
  double time;

  /** Destructor*/
  virtual ~PotentialfieldTransformation() {};

  /** Returns the type of the transformation
  * @return The type
  */
  virtual TransformationType getType() = 0;

  /** Returns a pointer to a copy of the object
  * @return A pointer to a copy
  */
  virtual PotentialfieldTransformation* copy() = 0;
};


/**
* @class Translation
*
* Describes a translation
*/
class Translation: public PotentialfieldTransformation
{
public:
  /** Translation vector*/
  PfVec translation;
  /** Flag: true, if the transformation is along the current gradient*/
  bool alongGradient;
  /** Step length on the gradient (only used, if along gradient has been set)*/
  double stepLength;
  /** Maximum deviation from the gradient (only used, if along gradient has been set)*/
  double maxGradientDeviation;
  /** Maximum length of a translation (only used, if along gradient has been set)*/
  double maxLength;
  /** An object to translate to */
  Object* toObject;
  /** The speed of a translation in millimeters per second, 
      only used in combination with alongGradient of toObject*/
  double speed;

  /** Returns the type of the action
  * @return The type
  */
  TransformationType getType()
  { return TRANSLATION;}

  /** Returns a pointer to a copy of the object
  * @return A pointer to a copy
  */
  PotentialfieldTransformation* copy()
  { 
    Translation* translation = new Translation();
    translation->translation = this->translation; 
    translation->toObject = toObject;
    translation->alongGradient = alongGradient;
    translation->stepLength = stepLength;
    translation->maxGradientDeviation = maxGradientDeviation;
    translation->maxLength = maxLength;
    translation->time = time;
    translation->speed = speed;
    PotentialfieldTransformation* transformation = translation;
    transformation->probability = probability;
    return transformation;
  }
};


/**
* @class Rotation
*
* Describes a rotation
*/
class Rotation : public PotentialfieldTransformation
{
public:
  /** The rotation angle*/
  double angle;
  /** Flag: true, if the rotation has to be the angle to the current gradient*/
  bool toGradient;
  /** An object to rotate to */
  Object* toObject;
  /** The speed of a rotation in radian per second, 
      only used in combination with toGradient of toObject*/
  double speed;

  /** Returns the type of the transformation
  * @return The type
  */
  TransformationType getType()
  { return ROTATION;}

  /** Returns a pointer to a copy of the object
  * @return A pointer to a copy
  */
  PotentialfieldTransformation* copy()
  { 
    Rotation* rotation = new Rotation();
    rotation->angle = angle; rotation->toGradient = toGradient;
    rotation->toObject = toObject;
    rotation->time = time;
    rotation->toGradient = toGradient;
    rotation->speed = speed;
    PotentialfieldTransformation* transformation = rotation;
    transformation->probability = probability;
    return transformation;
  }
};


/**
* @class NoTransformation
*
* Describes no transformation ;-)
*/
class NoTransformation : public PotentialfieldTransformation
{
public:
  /** Returns the type of the transformation
  * @return The type
  */
  TransformationType getType()
  { return NO_TRANSFORMATION;}

  /** Returns a pointer to a copy of the object
  * @return A pointer to a copy
  */
  PotentialfieldTransformation* copy()
  { 
    NoTransformation* noTransformation = new NoTransformation();
    PotentialfieldTransformation* transformation = noTransformation;
    transformation->probability = probability;
    transformation->time = time;
    return transformation;
  }
};


#endif	//ACTIONFIELD_H_



/*
* $Log: Actionfield.h,v $
* Revision 1.1  2004/01/20 15:42:20  tim
* Added potential fields implementation
*
* Revision 1.6  2003/05/22 14:23:47  tim
* Changed representation of transformations
*
* Revision 1.5  2003/05/15 16:59:57  tim
* fixed warnings
*
* Revision 1.4  2003/04/22 14:35:17  tim
* Merged changes from GO
*
* Revision 1.4  2003/04/09 19:03:06  tim
* Last commit before GermanOpen
*
* Revision 1.3  2003/04/02 14:39:10  tim
* Changed Bremen Byters Behavior
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
