/**
* @file SolutionRequest.h
*
* Declaration of class SolutionRequest
*
* @author Max Risler
* @author Martin Ltzsch
*/

#ifndef __SolutionRequest_h_
#define __SolutionRequest_h_

#include "Tools/Streams/InOut.h"

/** maximum number of solutions per module */
const int solutionRequestMaxNumOfSolutions = 40;

/** Manages runtime exhangeable modules. */
class SolutionRequest
{
public:
  /**
  * Enumerator for exchange modules
  * Add new exchangeable module here.
  * Create new selector module for new exchangeable modules
  */
  enum ModuleID
  {
#ifndef NO_COLOR_TABLE_SELECTOR
    colorTableMod,
#endif    
    sensorDataProcessor, /*!< See class SensorDataProcessorSelector */
    imageProcessor, /*!< See class ImageProcessorSelector */
    ballLocator, /*!< See class BallLocatorSelector */
    teamBallLocator, /*!< See class TeamBallLocatorSelector */
    obstaclesLocator, /*!< See class ObstaclesLocatorSelector */
    playersLocator, /*!< See class PlayersLocatorSelector */
    selfLocator, /*!< See class SelfLocatorSelector */
    robotStateDetector, /*!< See class RobotStateDetectorSelector */
    collisionDetector, /*!< See class CollisionDetectorSelector */
    behaviorControl, /*!< See class BehaviorControlSelector */
    motionControl, /*!< See class MotionControlSelector */
    walkingEngineNormal, /*!< See class WalkingEngineNormalSelector */
    headControl, /*!< See class HeadControlSelector */
    ledControl, /*!< See class LEDControlSelector */
    getupEngine, /*!< See class GetupEngineSelector */
    soundControl, /*!< See class SoundControlSelector */
    sensorBehaviorControl, /*!< See class SensorBehaviorControlSelector */
    sensorActorLoop, /*!< See class NeuroControlSelector */
    numOfModules /*!< Number of exchangeable modules */
  };

  /** enumerator for available solutions for all exchangeable modules
  * \attention add new solutions here
  * \attention dont forget to edit selector module too
  */
  enum ModuleSolutionID
  {
    disabled = -1,
    
#ifndef NO_COLOR_TABLE_SELECTOR
    // ColorTableMod solutions
    colorTable32KMod = 0,
    colorTable64Mod,
    colorTableTSLMod,
    numOfColorTableModSolutions, 
#endif
    
    // SelfLocator solutions
    gt2003SelfLocator = 0,
    fusion2003SelfLocator,
    linesSelfLocator,
    monteCarloSelfLocator,
    odometrySelfLocator,
    singleLandmarkSelfLocator,
    mrSingleLandmarkSelfLocator,
    specialPerceptSelfLocator,
    noOdometrySelfLocator,
    motionRatingSelfLocator,
    numOfSelfLocatorSolutions,


    // ImageProcessor solutions
    gt2003ImageProcessor = 0,
    gt2004ImageProcessor,
    gridImageProcessor2,
    gridImageProcessorTSL,
    checkerboardDetector,
    rasterImageProcessor,
    comboImageProcessor,
    msh2004ImageProcessor,
    ddd2004ImageProcessor,
    numOfImageProcessorSolutions,

    // BehaviorControl solutions
    gt2004BehaviorControlAgentSoccer = 0,
    gt2004BehaviorControlAgentOpenChallenge,
    gt2003BehaviorControlAgentSoccer,
    gt2003BehaviorControlAgentPenaltyShooter,
    gt2003BehaviorControlAgentManualKick,
    gt2003BehaviorControlAgentDemo,
    gt2003BehaviorControlAgentEvolution,
    gt2003BehaviorControlAgentJoystickControlled,
    ath2004BehaviorControlAgentSoccer,
    ath2004BehaviorControlAgentAnotherGoalie,
    ath2004BehaviorControlAgentDevelopAnotherGoalie1,
    ath2004BehaviorControlAgentOlliKahn,
    ath2004BehaviorControlAgentPenaltyShooter,
    ath2004BehaviorControlAgentPenaltyShooterHertha,
    ath2004BehaviorControlAgentPenaltyShooterUnion,
    ath2004BehaviorControlAgentPerceptTester,
    //viviana+michael
    ath2004BehaviorControlAgentInterceptTester,
    ath2004BehaviorControlAgentPassTester,
    ath2004BehaviorControlAgentJoystickControlled,
      
    ath2004ERS7BehaviorControlAgentSoccer,
    ath2004ERS7BehaviorControlAgentPenaltyShooter,
    ath2004ERS7BehaviorControlAgentKickTest,
    ath2004ERS7BehaviorControlAgentMLPassTest,
    ath2004ERS7BehaviorControlAgentMLDemo,

    ddd2004BehaviorControlAgentSoccer,
    ddd2004BehaviorControlAgentPenaltyShooter,
    ddd2004BehaviorControlAgentDemo,
    bb2004BehaviorControlAgentSoccer,
    bb2004BehaviorControlAgentPenaltyShooter,
    msh2004BehaviorControlAgentSoccer,
    msh2004BehaviorControlAgentDTTSoccer,
    msh2004BehaviorControlAgentPenaltyShooter,
    msh2004BehaviorControlAgentGlassBoxDemo,
    msh2004BehaviorControlAgentKickengineTraining,
    msh2004BehaviorControlAgentTest,
    msh2004BehaviorControlAgentAutonomousEvolution,
    motionRatingBehaviorControl,
   
    motionRating2BehaviorControl,
    numOfBehaviorControlSolutions,

    // CollisionDetector solutions
    defaultCollisionDetector = 0,
    numOfCollisionDetectorSolutions,

    // WalkingEngine solutions
    invKinGT2003WalkingEngine = 0,
    invKinUpsideDownWalkingEngine,
    invKinTurnWithBallWalkingEngine,
    invKinUNSW2003WalkingEngine,
    invKinFastForwardWalkingEngine,
    invKinERS7WalkingEngine,
    invKinERS7TurnWithBallWalkingEngine,
    unsw2003WalkingEngine,
    udWalkingEngine,
    yetAnotherInvKinWE,
    invKinBB2004WalkingEngine,
    invKinMSH2004WalkingEngine,
    invKinMSH2004TurnWithBallWalkingEngine,
    invKinDDDERS7WalkingEngine,
    invKinDDDERS7TurnWithBallWalkingEngine,
    invKinDDDERS7FastWalkingEngine,
    numOfWalkingEngineSolutions,

    // BallLocator solutions
    pidSmoothedBallLocator = 0,

    ath2004BallLocator,
    msh2004BallLocator, 
    perceptBallLocator,
    voteBasedBallLocator,
    kalmanBallLocator,
    ddd2004BallLocator,
    bb2004BallLocator,
    validityBasedBallLocator,
    numOfBallLocatorSolutions,

    // TeamBallLocator solutions 
    defaultTeamBallLocator = 0,
    ath2004TeamBallLocator,
    sensorFusionTeamBallLocator,
    numOfTeamBallLocatorSolutions,


    // PlayersLocator solutions
    go2003PlayersLocator = 0,
    numOfPlayersLocatorSolutions,

    // ObstaclesLocator solutions
    defaultObstaclesLocator = 0,
    numOfObstaclesLocatorSolutions,

    // HeadControl solutions
    gt2003HeadControl = 0,
    gt2004HeadControl,
    ddd2004HeadControl,
    ath2004ers7HeadControl,
    bb2004HeadControl,
    msh2004HeadControl,
    numOfHeadControlSolutions,

    // LEDControl solutions
    defaultLEDControl = 0,
    ath2004LEDControl,
    numOfLEDControlSolutions,

    // SensorDataProcessor solutions
    defaultSensorDataProcessor = 0,
    bb2004SensorDataProcessor,
    numOfSensorDataProcessorSolutions,

    // RobotStateDetector solutions
    defaultRobotStateDetector = 0,
    numOfRobotStateDetectorSolutions,

    // SensorBehaviorControl solutions
    obstacleAvoiderOnGreenField = 0,
    obstacleAvoiderOnGreenFieldERS7,
    allxRemotePresence,
    motionrecognition,
  	simpleMotionRecognition,
    bb2004Calibrator,
    jansVisionSandbox,
    numOfSensorBehaviorControlSolutions,

    // MotionControl solutions
    defaultMotionControl = 0,
    debugMotionControl,
    numOfMotionControlSolutions,

    // GetupEngine solutions
    defaultGetupEngine = 0,
    numOfGetupEngineSolutions,

    // SoundOutControl solutions
    defaultSoundControl = 0,
    numOfSoundControlSolutions,

    neuroControl = 0,
    demoSensorActorLoop,
    numOfSensorActorLoopSolutions
  };

  /** returns names for module ids
  * Add new exchangeable module here
  */
  static const char* getModuleName(ModuleID id)
  {
    switch (id)
    {
    case colorTableMod: return "ColorTableModule";
    case imageProcessor: return "ImageProcessor";
    case selfLocator: return "SelfLocator";
    case behaviorControl: return "BehaviorControl";
    case collisionDetector: return "CollisionDetector";
    case walkingEngineNormal: return "WalkingEngine:normal";
    case headControl: return "HeadControl";
    case ledControl: return "LEDControl";
    case ballLocator: return "BallLocator";
    case playersLocator: return "PlayersLocator";
    case obstaclesLocator: return "ObstaclesLocator";
    case sensorDataProcessor: return "SensorDataProcessor";
    case robotStateDetector: return "RobotStateDetector";
    case sensorBehaviorControl: return "SensorBehaviorControl";
    case motionControl: return "MotionControl";
    case getupEngine: return "GetupEngine";
    case soundControl: return "SoundControl";
    case teamBallLocator: return "TeamBallLocator";
    case sensorActorLoop: return "SensorActorLoop";

    default: return "please edit SolutionRequest::getModuleName";
    }
  }

  /** Returns number of solutions in module id
  * \attention add new exchangeable module here
  */
  static int getNumOfSolutions(ModuleID id)
  {
    switch (id)
    {
    case colorTableMod :return numOfColorTableModSolutions;
    case imageProcessor: return numOfImageProcessorSolutions;
    case selfLocator: return numOfSelfLocatorSolutions;
    case behaviorControl: return numOfBehaviorControlSolutions;
    case collisionDetector: return numOfCollisionDetectorSolutions;
    case walkingEngineNormal: return numOfWalkingEngineSolutions;
    case headControl: return numOfHeadControlSolutions;
    case ledControl: return numOfLEDControlSolutions;
    case ballLocator: return numOfBallLocatorSolutions;
    case playersLocator: return numOfPlayersLocatorSolutions;
    case obstaclesLocator: return numOfObstaclesLocatorSolutions;
    case sensorDataProcessor: return numOfSensorDataProcessorSolutions;
    case robotStateDetector: return numOfRobotStateDetectorSolutions;
    case sensorBehaviorControl: return numOfSensorBehaviorControlSolutions;
    case motionControl: return numOfMotionControlSolutions;
    case getupEngine: return numOfGetupEngineSolutions;
    case soundControl: return numOfSoundControlSolutions;
    case teamBallLocator: return numOfTeamBallLocatorSolutions;
    case sensorActorLoop: return numOfSensorActorLoopSolutions;

    default: return 0;
    }
  }

  /**
  * Returns names for module solution ids.
  * Add new solutions here.
  */
  static const char* getModuleSolutionName(ModuleID module, ModuleSolutionID solution)
  {
    if (solution == disabled) return "disabled";
    switch (module)
    {
#ifndef NO_COLOR_TABLE_SELECTOR      
    case colorTableMod: switch (solution)
                      {
    case colorTable32KMod: return "ColorTable32KModule";
    case colorTable64Mod: return "ColorTable64Module";
    case colorTableTSLMod: return "ColorTableTSLModule";      
                      }
#endif
    case selfLocator:  switch (solution)
                       {
    case monteCarloSelfLocator: return "MonteCarlo";
    case singleLandmarkSelfLocator: return "SingleLandmark";
    case mrSingleLandmarkSelfLocator: return "MRSingleLandmark";
    case linesSelfLocator: return "Lines";
    case odometrySelfLocator: return "Odometry";
    case specialPerceptSelfLocator: return "SpecialPercept";
    case gt2003SelfLocator: return "GT2003";
    case fusion2003SelfLocator: return"Fusion2003";
    case noOdometrySelfLocator: return "NoOdometry";
    case motionRatingSelfLocator: return "MotionRating";
                       }
    case behaviorControl: switch (solution)
                          {
    case gt2004BehaviorControlAgentSoccer: return "GT2004-soccer";
    case gt2004BehaviorControlAgentOpenChallenge: return "GT2004-open-challenge";

    case gt2003BehaviorControlAgentSoccer: return "GT2003-soccer";
    case gt2003BehaviorControlAgentPenaltyShooter: return "GT2003-penalty-shooter";
    case gt2003BehaviorControlAgentManualKick: return "GT2003-manual-kick";
    case gt2003BehaviorControlAgentDemo: return "GT2003-demo";
    case gt2003BehaviorControlAgentEvolution: return "GT2003-evolution";
    case gt2003BehaviorControlAgentJoystickControlled: return "GT2003-joystick-controlled";
 
    case ath2004BehaviorControlAgentSoccer: return "ATH2004-soccer";
    case ath2004BehaviorControlAgentAnotherGoalie: return "ATH2004-another-goalie";
    case ath2004BehaviorControlAgentDevelopAnotherGoalie1: return "ATH2004-develop-another-goalie1";
    case ath2004BehaviorControlAgentOlliKahn: return "ATH2004-olli-kahn";
    case ath2004BehaviorControlAgentPenaltyShooter: return "ATH2004-penalty-shooter";
    case ath2004BehaviorControlAgentPenaltyShooterHertha: return "ATH2004-penalty-shooter-hertha";
    case ath2004BehaviorControlAgentPenaltyShooterUnion: return "ATH2004-penalty-shooter-union";
    case ath2004BehaviorControlAgentPerceptTester: return "ATH2004-percept-tester";
    case ath2004BehaviorControlAgentInterceptTester: return "ATH2004-Intercept-Tester" ;
    case ath2004BehaviorControlAgentPassTester: return "ATH2004-Pass-Tester" ;
    case ath2004BehaviorControlAgentJoystickControlled: return "ATH2004-joystick-controlled";

    case ath2004ERS7BehaviorControlAgentSoccer: return "ATH2004-ERS7-soccer";
    case ath2004ERS7BehaviorControlAgentPenaltyShooter: return "ATH2004-ERS7-penalty-shooter";
    case ath2004ERS7BehaviorControlAgentKickTest: return "ATH2004-ERS7-kick-test";
    case ath2004ERS7BehaviorControlAgentMLPassTest: return "ATH2004-ERS7-ML-pass-test";
    case ath2004ERS7BehaviorControlAgentMLDemo: return "ATH2004-ERS7-ML-demo";

    case ddd2004BehaviorControlAgentSoccer: return "DDD2004-soccer";
    case ddd2004BehaviorControlAgentPenaltyShooter: return "DDD2004-penalty-shooter";
    case ddd2004BehaviorControlAgentDemo: return "DDD2004-demo";
    
    case bb2004BehaviorControlAgentSoccer: return "BB2004-soccer";
    case bb2004BehaviorControlAgentPenaltyShooter: return "BB2004-penalty-shooter";
    
    case msh2004BehaviorControlAgentSoccer: return "MSH2004-soccer";
    case msh2004BehaviorControlAgentDTTSoccer: return "MSH2004-dtt-soccer";
    case msh2004BehaviorControlAgentPenaltyShooter: return "MSH2004-penalty-shooter";
    case msh2004BehaviorControlAgentGlassBoxDemo: return"MSH2004-glassbox-demo";
    case msh2004BehaviorControlAgentKickengineTraining: return"MSH2004-kickengine-training";
    case msh2004BehaviorControlAgentAutonomousEvolution: return"MSH2004-autonomous-evolution";
    case msh2004BehaviorControlAgentTest: return"MSH2004-test";
    
    case motionRatingBehaviorControl: return "MotionRating";
    case motionRating2BehaviorControl: return "MotionRating2";
                          }

    case collisionDetector: switch (solution)
                            {
    case defaultCollisionDetector: return "Default";
                            }
    case walkingEngineNormal: switch (solution) 
                              {
    case invKinGT2003WalkingEngine: return "InvKin:GT2003";
    case invKinUpsideDownWalkingEngine: return "InvKin:UpsideDown";
    case invKinTurnWithBallWalkingEngine: return "InvKin:TurnWithBall";
    case invKinUNSW2003WalkingEngine: return "InvKin:UNSW2003";
    case invKinFastForwardWalkingEngine: return "InvKin:FastForward";
    case invKinERS7WalkingEngine: return "InvKin:ERS7";
    case invKinERS7TurnWithBallWalkingEngine: return "InvKin:ERS7TurnWithBall";
    case unsw2003WalkingEngine: return "UNSW2003";
    case udWalkingEngine: return "UD";
    case yetAnotherInvKinWE : return "YetAnotherInvKin";
    case invKinBB2004WalkingEngine: return "InvKin:BB2004";
    case invKinMSH2004WalkingEngine: return "InvKin:MSH2004";
    case invKinMSH2004TurnWithBallWalkingEngine: return "InvKin:MSH2004TurnWithBall";
    case invKinDDDERS7WalkingEngine: return "InvKin:DDDERS7";
    case invKinDDDERS7TurnWithBallWalkingEngine: return "InvKin:DDDERS7TurnWithBall";
    case invKinDDDERS7FastWalkingEngine: return "InvKin:DDDERS7fast";
                              }
    case headControl: switch (solution)
                      {
    case ath2004ers7HeadControl: return "ATH2004-ERS7-only";
    case gt2003HeadControl: return "GT2003";
    case gt2004HeadControl: return "GT2004";
    case bb2004HeadControl: return "BB2004";
    case ddd2004HeadControl: return "DDD2004";
    case msh2004HeadControl: return "MSH2004";
                      }
    case ledControl: switch (solution)
                      {
    case ath2004LEDControl: return "ATH2004";
    case defaultLEDControl: return "Default";
                      }
    case ballLocator: switch (solution)
                      {
    case pidSmoothedBallLocator: return "PIDSmoothed";

    case ath2004BallLocator: return "ATH2004";
    case msh2004BallLocator: return "MSH2004";
    case perceptBallLocator: return "Percept";
    case voteBasedBallLocator: return "VoteBased";
    case kalmanBallLocator: return "Kalman";
    case ddd2004BallLocator: return "DDD2004";
    case bb2004BallLocator: return "BB2004";
    case validityBasedBallLocator: return "Validity";
                      }
    case playersLocator: switch (solution)
                         {
    case go2003PlayersLocator: return "GO2003";
                         }
    case obstaclesLocator: switch (solution)
                           {
    case defaultObstaclesLocator: return "Default";
                           }
    case imageProcessor: switch (solution)
                         {
    case gt2003ImageProcessor: return "GT2003";
    case gt2004ImageProcessor: return "GT2004";
    case gridImageProcessor2: return "Grid2";
    case gridImageProcessorTSL: return "GridTSL";
    case checkerboardDetector: return "CheckerboardDetector";
    case rasterImageProcessor: return "RasterImageProcessor";
    case msh2004ImageProcessor: return "MSH2004";
    case comboImageProcessor: return "ComboImageProcessor";
    case ddd2004ImageProcessor: return "DDD2004";
                         }
    case sensorDataProcessor: switch(solution)
                              {
    case defaultSensorDataProcessor: return "Default";
    case bb2004SensorDataProcessor: return "BB2004";
                              }
    case robotStateDetector: switch (solution)
                             {
    case defaultRobotStateDetector: return "Default";
                             }
    case sensorBehaviorControl: switch (solution)
                                {
    case obstacleAvoiderOnGreenField: return "ObstacleAvoiderOnGreenField";
    case obstacleAvoiderOnGreenFieldERS7: return "ObsAvoider(OGF) ERS7";
    case allxRemotePresence: return "AlLx Remote Presence";
    case motionrecognition: return "MotionRec";
    case simpleMotionRecognition: return "SimpleMotionRec";
    case bb2004Calibrator: return "BB2004Calibrator";
    case jansVisionSandbox: return "Jan's Vision Sandbox";
                                }
    case motionControl: switch (solution)
                        {
    case defaultMotionControl: return "Default";
    case debugMotionControl: return "Debug";
                        }
    case getupEngine: switch (solution)
                      {
    case defaultGetupEngine: return "Default";
                      }
    case soundControl: switch (solution)
                       {
    case defaultSoundControl: return "Default";
                       }
    case teamBallLocator: switch(solution)
                          {
    case defaultTeamBallLocator: return "default";
    case ath2004TeamBallLocator: return "ATH2004";
    case sensorFusionTeamBallLocator: return "SensorFusion";
                          }

    case sensorActorLoop: switch(solution)
                          {
    case neuroControl: return "neuroControl";
    case demoSensorActorLoop: return "demoSensorActorLoop";
                          }


    }
    return "please edit SolutionRequest::getModuleSolutionName";
  }

  /** an array containing the selected solution for each module */
  ModuleSolutionID solutions[numOfModules];

  /** Constructor
  * @param setDefault if true default modules will be selected else all modules are disabled
  */
  SolutionRequest(bool setDefault = false);

  /**
  * Returns default solution for module.
  */
  static ModuleSolutionID getDefaultSolution(ModuleID id)
  {
    static SolutionRequest defaultRequest(true);
    return defaultRequest.solutions[id];
  }

  /**
  * Sets all modules to default solutions.
  */
  void setDefaultSolutions();

  /** the == operator */
  bool operator == (const SolutionRequest& other);

  /** the == operator */
  bool operator != (const SolutionRequest& other)
  { return ! (*this == other); }

  /**
  * ErrorWhileReading
  * is set to true when an error occurred while reading the SolutionRequest
  * eg when an module was missing or a solution was not recognized
  */
  bool errorWhileReading;

  /** IDs for the module solutions using a Xabsl2Engine */
  enum xabsl2EngineID
  {
    gt2004, 
    gt2003, 
    ath2004, 
    ath2004ers7, 
    ddd2004, 
    bb2004, 
    msh2004,
    numOfXabslBehaviors,
    hc_gt2004 = numOfXabslBehaviors, 
    undefined
  };

  /** returns the name of a xabsl2EngineID */
  static const char* getXabsl2EngineIDName(xabsl2EngineID id)
  {
    switch (id)
    {
    case gt2004: return "GT2004";
    case gt2003: return "GT2003";
    case ath2004: return "ATH2004";
    case ath2004ers7: return "ATH2004ERS7";
    case ddd2004: return "DDD2004";
    case bb2004: return "BB2004";
    case msh2004: return "MSH2004";
    case hc_gt2004: return "HC_GT2004";
    default: return "unknown";
    }
  }

  /** returns the beginning of the file name of a config file
  * for a xabsl2Engine
  */
  static const char* getXabsl2EngineFileID(xabsl2EngineID id)
  {
    switch (id)
    {
    case gt2004: return "gt04";
    case gt2003: return "gt03";
    case ath2004: return "ath04";
    case ath2004ers7: return "ath7";
    case ddd2004: return "ddd04";
    case bb2004: return "bb04";
    case msh2004: return "msh04";
    case hc_gt2004: return "HeadCtrl/gt04";

    default: return "unknown";
    }
  }
};

/**
* Streaming operator that reads a SolutionRequest from a stream.
* @param stream The stream from which is read.
* @param solutionRequest The SolutionRequest object.
* @return The stream.
*/
In& operator>>(In& stream,SolutionRequest& solutionRequest);

/**
* Streaming operator that writes a SolutionRequest to a stream.
* @param stream The stream to write on.
* @param solutionRequest The SolutionRequest object.
* @return The stream.
*/
Out& operator<<(Out& stream, const SolutionRequest& solutionRequest);

#endif //__SolutionRequest_h_

/*
* Change log :
*
* $Log: SolutionRequest.h,v $
* Revision 1.73  2004/05/25 12:36:57  jhoffman
* "robertson" operator demo added
*
* Revision 1.72  2004/05/18 11:04:54  roefer
* Separate view and optional parameter hc for xabsl head control
*
* Revision 1.71  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.70  2004/05/06 15:40:18  loetzsch
* yet another demo agent
*
* Revision 1.69  2004/05/05 21:47:14  kerdels
* added agent for the open challenge
*
* Revision 1.68  2004/05/04 13:56:55  tim
* added GT2004ImageProcessor
*
* Revision 1.67  2004/05/02 13:33:54  juengel
* Added GT2004BehaviorControl.
*
* Revision 1.66  2004/04/27 11:23:23  loetzsch
* preliminary passing experiments
*
* Revision 1.65  2004/04/26 15:59:01  thomas
* added new project RobotRemote based on ATHAiboControl
*
* Revision 1.64  2004/04/21 20:13:33  goehring
* BallSolution added
*
* Revision 1.63  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.62  2004/04/09 11:35:54  roefer
* Bremen Byters German Open check-in
*
* Revision 1.61  2004/04/08 15:33:08  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.60  2004/04/07 13:25:43  thomas
* ddd checkin after go04 - second part fixed solutions
*
* Revision 1.59  2004/04/07 13:00:49  risler
* ddd checkin after go04 - second part
*
* Revision 1.7  2004/04/02 23:44:40  dthomas
* added new invKinParams for turn with ball
* integrated into rotate-around-ball
* combined with updated catch-ball head-control-mode
*
* Revision 1.6  2004/04/02 09:27:54  Marc
* Removed Opponent Team Code
*
* Revision 1.5  2004/03/31 10:08:08  dthomas
* added new parameterset for ers7 based on jolly pochie
*
* Revision 1.4  2004/03/30 14:00:36  Papa Schlumpf
* added DDD2004BallLocator
*
* Revision 1.3  2004/03/29 15:19:05  Marc
* Intruduced the Black and White Image
* Normal Images (not Jpeg) images were now send as Color Image with BW
*
* Revision 1.2  2004/03/29 10:06:25  dthomas
* added new ers7 walking engine parameters
*
* Revision 1.1.1.1  2004/03/29 08:28:42  Administrator
* initial transfer from tamara
*
* Revision 1.57  2004/03/28 14:06:41  jhoffman
* renamed headcontrolmode ATH2004 in ATH2004ERS7
*
* Revision 1.58  2004/04/05 17:56:50  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.57  2004/03/28 14:06:41  jhoffman
* renamed headcontrolmode ATH2004 in ATH2004ERS7
*
* Revision 1.56  2004/03/20 17:19:19  juengel
* Added selector for LEDControl
*
* Revision 1.55  2004/03/19 11:31:55  nistico
* ColorTable32KImageProcessor removed
*
* Revision 1.54  2004/03/17 16:18:50  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
* Revision 1.53  2004/03/17 01:10:51  cesarz
* added MSH2004TurnWithBallWalkingParameters
*
* Revision 1.52  2004/03/16 14:16:06  risler
* cloned GT2003ImageProcessor to DDD2004ImageProcessor
*
* Revision 1.51  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.50  2004/03/11 14:27:32  uhrig
* KalmanBallLocator was added
*
* Revision 1.4  2004/03/16 08:34:34  loetzsch
* moved agent kick-test from ATH2004 to ATH2004ERS7
*
* Revision 1.3  2004/03/15 17:11:41  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.2  2004/03/06 18:31:21  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.49  2004/03/10 17:45:21  kindler
* - Added MSH2004 InvKinWalkingEngine parameter set.
*
* Revision 1.48  2004/03/04 23:00:55  roefer
* Added (so far empty) BB2004Calibrator
*
* Revision 1.47  2004/03/04 18:21:20  juengel
* Added ERS7TurnWithBallWalkingParameters.
*
* Revision 1.46  2004/02/27 15:03:52  wachter
* cloned GT2003HeadControl as MSH2004HeadContol
*
* Revision 1.45  2004/02/27 13:37:16  jhoffman
* added obstacle avoiding "sensor behavior control" for ers7
*
* Revision 1.44  2004/02/26 18:07:28  cesarz
* first version of evolution behavior
*
* Revision 1.43  2004/02/24 20:58:43  wachter
* Increased solutionRequestMaxNumOfSolutions to 30 because we have 27 behavior-control solutions.
*
* Revision 1.42  2004/02/23 12:52:58  schumann
* added MSH2004KickEngine
*
* Revision 1.41  2004/02/16 18:54:38  tim
* Added BB2004ImageProcessor
*
* Revision 1.40  2004/02/16 01:02:02  rempe
* agent for developing another goalie
*
* Revision 1.39  2004/02/13 13:33:25  goetzke
* added PassTester
*
* Revision 1.38  2004/02/10 17:51:20  goetzke
* no message
*
* Revision 1.37  2004/02/08 10:58:14  witzmann
* added olli-kahn (Dirk Rempe (rempe@informatik.hu-berlin.de) , Robert Sombrutzki (sombrutz@informatik.hu-berlin.de), Erik Witzmann)
*
* Revision 1.36  2004/02/05 14:23:16  spranger
* added intercept-tester agent
*
* Revision 1.35  2004/02/05 10:31:44  juengel
* tags added
*
* Revision 1.34  2004/02/04 13:41:33  roefer
* Some place holders for BB2004 modules added
*
* Revision 1.33  2004/01/30 15:44:33  nistico
* Created a Color Table Selector, which permits to switch color tables at run time using the same image processor.
* If your image processor was designed to use only ColorTable64, leave it as the default choice and you should
* (hopefully) have no problems.
* In case of trouble, i put a conditional compilation switch in SolutionRequest.h, simply uncomment the definition of NO_COLOR_TABLE_SELECTOR; in that case, the code should get back to the original solution, but the coltable which will be looked upon first for loading will be ColorTable64 (as it's the default solution), so there's no reason anymore to remove the other color tables from the CVS.
*
* Revision 1.32  2004/01/24 14:55:29  loetzsch
* created ATH AiboControl
*
* Revision 1.31  2004/01/23 23:02:25  wachter
* Reinserted the famous GaussBellTeamBallLocator from the
* as GT2003-code as SensorFusionTeamBallLocator.
*
* Revision 1.30  2004/01/20 12:40:09  nistico
* - Added support for ColorTable32K (65K elements in packed format)
* - RobotControl can now convert GT *.log files into AIBOVision (external ColorTable32K calibration tool) *.amv file format
*
* Revision 1.29  2004/01/15 15:43:00  kindler
* Added ComboImageProcessor (only temporarily, for debugging only..)
*
* Revision 1.28  2004/01/13 18:35:24  loetzsch
* added invkin:walkingparameterset ERS7
*
* Revision 1.27  2004/01/08 21:59:18  richert
* added Module SimpleMotionRecognition
*
* Revision 1.26  2004/01/08 13:53:38  loetzsch
* new kick test behaviors
*
* Revision 1.25  2004/01/06 17:49:12  sombrutz
* added another-goalie (Dirk Rempe (rempe@informatik.hu-berlin.de) , Robert Sombrutzki (sombrutz@informatik.hu-berlin.de), Erik Witzmann)
*
* Revision 1.24  2003/12/29 15:48:54  roefer
* Bremen Byters evo walking added
*
* Revision 1.23  2003/12/19 01:14:14  kerdels
* agent added
*
* Revision 1.22  2003/12/12 17:08:28  richert
* added new solution motionrecognition
*
* Revision 1.21  2003/12/11 16:18:42  risler
* added agent DDD2004-demo
*
* Revision 1.20  2003/12/11 11:25:26  seva
* Added penalty-shooter-union and penalty-shooter-hertha.
*
* Revision 1.19  2003/12/10 21:00:02  kerdels
* OptionRating aus SolutionRequest.h entfernt und Tools Ordner DynamicTeamTactic angelegt
*
* Revision 1.18  2003/12/06 18:46:25  kerdels
* neues Modul OptionRating angelegt,
* neue Representation CollectedBeliefs angelegt.
*
* Revision 1.17  2003/12/03 13:29:27  schumann
* added MSH2004 Ball Locator
*
* Revision 1.16  2003/12/02 19:45:42  spranger
* added NoOdometrySelfLocator
*
* Revision 1.15  2003/12/02 18:28:57  dueffert
* first working not yet calibrated version of UDWalkingEngine added
*
* Revision 1.14  2003/12/02 13:52:48  goehring
* PerceptBallLocator added
*
* Revision 1.13  2003/11/22 09:33:01  jhoffman
* added "Yet Another Inv Kin Walking Engine"
* experimental stage, doesn't do much yet
* no Fourier inside!
*
* Revision 1.12  2003/11/21 15:45:02  dueffert
* fast forward walking added
*
* Revision 1.11  2003/11/20 10:30:13  schmidtb
* Ball Detection added
*
* Revision 1.10  2003/11/17 14:56:04  urban
* added SensorBehaviorControl-solution "AlLx RemotePresence" and RobotControl-dialogbar "AlLx Joystick"
*
* Revision 1.9  2003/11/03 20:12:19  juengel
* Added ath-percept-tester
*
* Revision 1.8  2003/10/29 13:16:06  juengel
* Added module SensorActorLoop.
*
* Revision 1.7  2003/10/27 13:27:03  wachter
* Added agent for glass-box demo.
*
* Revision 1.6  2003/10/26 22:49:41  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
* Revision 1.5  2003/10/24 15:01:42  loetzsch
* added UNSW2003WalkingEngine
*
* Revision 1.3  2003/10/22 10:20:37  goehring
* VoteBasedBallLocator Added
*
* Revision 1.2  2003/10/15 13:39:15  goehring
* ATH2004BallLocator added
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.13  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.12  2003/09/30 09:36:11  juengel
* Removed FusionSelfLocator.
*
* Revision 1.11  2003/09/27 18:40:47  loetzsch
* removed AnnaSeghersSchoolDemo again
*
* Revision 1.10  2003/09/25 11:20:19  juengel
* Removed LinesHeadControl.
*
* Revision 1.9  2003/09/25 10:13:21  juengel
* Removed some walking engines.
*
* Revision 1.8  2003/09/14 17:47:38  loetzsch
* added solution AnnaSeghersSchooDemo
*
* Revision 1.7  2003/09/01 15:19:01  juengel
* SolutionRequest clean-up.
*
* Revision 1.6  2003/07/30 14:56:25  dueffert
* walk evolution agent added
*
* Revision 1.5  2003/07/24 13:58:22  risler
* added parameter set UNSW2003
*
* Revision 1.4  2003/07/07 22:45:33  thomas
* added: frame for penalty-shooter agent
*
* Revision 1.3  2003/07/05 12:00:10  kudlacik
* Added BallChallengeBallLocator
*
* Revision 1.2  2003/07/04 10:20:06  kudlacik
* Added BallChallangeHeadControl
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.88  2003/06/24 10:11:21  jhoffman
* added new ball locator
*
* Revision 1.87  2003/06/23 13:36:22  loetzsch
* renamed GT2003 Agent turn-and-kick-test to kick-test
*
* Revision 1.86  2003/06/14 21:45:47  roefer
* GT2003 walking parameters redesign
*
* Revision 1.85  2003/06/13 17:03:08  juengel
* Added challenge agents and options.
*
* Revision 1.84  2003/06/12 21:42:27  roefer
* GT2003 walking parameters finished
*
* Revision 1.83  2003/06/09 19:59:05  roefer
* GT2003 walking engine parameters added
*
* Revision 1.82  2003/06/06 10:55:18  dueffert
* demo agent added
*
* Revision 1.81  2003/06/02 09:07:49  loetzsch
* added option turn-and-kick and agent turn-and-kick-test
*
* Revision 1.80  2003/05/27 15:36:42  juengel
* Added HUGO2003HeadControl.
*
* Revision 1.79  2003/05/27 15:32:06  loetzsch
* added agent manual-kick
*
* Revision 1.78  2003/05/26 13:14:52  brunn
* Fusion2003SelfLocator added
*
* Revision 1.77  2003/05/22 07:53:05  roefer
* GT2003SelfLocator added
*
* Revision 1.76  2003/05/20 15:52:42  deutsch
* Added GridImageProcessorTSL.
*
* Revision 1.75  2003/05/15 15:24:37  risler
* added agent obstacle-avoider
*
* Revision 1.74  2003/05/14 13:08:38  risler
* removed DefaultObstaclesLocator
* renamed MicroSectorsObstaclesLocator to DefaultObstaclesLocator
* ObstaclesModel contains increased number of sectors
* DefaultObstaclesLocator clean up
*
* Revision 1.73  2003/05/07 17:00:49  goehring
* Added new walking solution 'Swing'
*
* Revision 1.72  2003/05/06 16:02:31  loetzsch
* added agent free-part-of-opponent-goal-test
*
* Revision 1.71  2003/05/06 13:52:31  pruente
* added collective-soccer agent
*
* Revision 1.70  2003/05/03 15:14:03  loetzsch
* GT2003BehaviorControl first draft
*
* Revision 1.69  2003/05/01 17:09:10  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.68  2003/05/01 12:56:27  roefer
* Names of ImageProcessor solutions shortend
*
* Revision 1.67  2003/04/25 19:50:17  goehring
* Added new module CollisionDetector
*
* Revision 1.66  2003/04/16 14:01:31  loetzsch
* removed Xabsl 1 from GT2003
*
* Revision 1.65  2003/04/16 07:00:17  roefer
* Bremen GO checkin
*
* Revision 1.64  2003/04/15 17:19:44  risler
* moved DDDGO2003 ImageProcessor to own module
* removed ContinuousBasicBehaviorTester
*
* Revision 1.63  2003/04/15 15:52:07  risler
* DDD GO 2003 code integrated
*
* Revision 1.61  2003/04/09 16:22:56  max
* added continuous-test agent
*
* Revision 1.59  2003/04/05 17:02:40  max
* added localization-test agent
* added tailcontrol option
*
* Revision 1.62  2003/04/14 16:18:26  loetzsch
* ATH after GermanOpen CVS merge
* added penalty player
*
* Revision 1.2  2003/04/13 12:36:10  juengel
* Added penalty player.
*
* Revision 1.1.1.1  2003/04/09 14:23:19  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.61  2003/04/04 17:18:44  juengel
* AddedHUGO2003ImageProcessor
*
* Revision 1.60  2003/04/04 09:35:01  jhoffman
* added obstacle locator
*
* Revision 1.59  2003/03/31 22:00:18  roefer
* LinesHeadControl added
*
* Revision 1.58  2003/03/31 14:57:28  jhoffman
* added "GT2003HeadControl" and put some head control stuff
* in the parent class "HeadControl"; "GT2003HeadConrol" subsumes
* "GT2002H..."
*
* Revision 1.57  2003/03/31 14:16:21  hebbel
* Walking parameters for turning with ball added
*
* Revision 1.56  2003/03/30 14:10:27  juengel
* Added BarCodeSwitcher
*
* Revision 1.55  2003/03/29 23:06:44  loetzsch
* added class SelfLocatorSwitcher
*
* Revision 1.54  2003/03/29 12:52:32  mkunz
* GO2003PlayersLocator added
*
* Revision 1.53  2003/03/25 08:47:12  dasmuli
* Added simple segmentation and the 180/360 memory. Dialog was changed in order to display segemented data.
*
* Revision 1.52  2003/03/18 16:51:53  osterhues
* Typo corrected
*
* Revision 1.51  2003/03/12 22:26:23  roefer
* LinesImageProcessor2 added
*
* Revision 1.50  2003/03/12 13:46:04  dasmuli
* PlayersPerceptor added to cognition, modules.cfg, DebugKey-table etc.
*
* Revision 1.49  2003/03/12 12:14:12  risler
* added DDD-walk-test
*
* Revision 1.48  2003/03/10 14:00:52  juengel
* Added ObstaclesLocator
*
* Revision 1.47  2003/03/05 14:07:10  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.46  2003/03/05 10:35:03  jhoffman
* added PIDSmoothedBallLocator
*
* Revision 1.45  2003/03/04 15:07:53  osterhues
* Changed GridImageProcessorDo to GridImageProcessor3
*
* Revision 1.44  2003/03/03 17:35:08  risler
* added CMUClone parameter set
*
* Revision 1.43  2003/02/27 10:06:59  schmidt
* Added three variants of a SensorFusionPlayersLocator.
*
* Revision 1.42  2003/02/24 14:48:17  kallnik
* some bugs in continuousbasicbehaviors fixed
*
* Revision 1.41  2003/02/19 16:55:17  hebbel
* Added evolved parameterset for InvKinWalkingEngine
*
* Revision 1.40  2003/02/19 15:01:23  roefer
* LinesPerceptor2 added
*
* Revision 1.39  2003/02/14 14:34:02  wachter
* Added SensorFusionBallLocator
*
* Revision 1.38  2003/02/07 16:42:26  dueffert
* Bombay stuff added after cleanup
*
* Revision 1.37  2003/01/30 18:26:49  jhoffman
* added ObstacleAvoider
*
* Revision 1.36  2003/01/30 13:16:49  loetzsch
* Redesign of class BallPosition
*
* Revision 1.35  2003/01/23 16:44:10  risler
* only one instance of InvKinWalkingEngine
* parameter sets can now be switched while walking
* added UNSWFastTurn, combining two parameter sets
*
* Revision 1.34  2003/01/23 15:22:30  dueffert
* new xabsl2 agent added
*
* Revision 1.33  2003/01/22 15:02:12  dueffert
* checkerboard stuff added
*
* Revision 1.32  2003/01/22 13:08:11  juengel
* Added getXabsl2BehaviorControlFileID()
*
* Revision 1.31  2003/01/21 18:10:12  risler
* added agent simple-follow-ball
*
* Revision 1.30  2003/01/20 12:20:34  juengel
* Added BremenBytersBehaviorControl
* and RuhrpottHellhoundsBehaviorControl.
*
* Revision 1.29  2003/01/20 08:14:17  loetzsch
* added xabsl2BehaviorControlID to the solutionRequest to distinct between the
* different Xabsl2BehaviorControls
*
* Revision 1.28  2003/01/19 22:57:09  loetzsch
* cloned Humboldt2003BehaviorControl to
* DarmstadtDribblingDackelsBehaviorControl
*
* Revision 1.27  2003/01/19 13:04:52  loetzsch
* xabsl2 agents now can be changed by using the Module and SolutionRequest
* mechanism
*
* Revision 1.26  2003/01/18 17:45:59  risler
* added upsideDown parameters
*
* Revision 1.25  2003/01/12 15:24:47  juengel
* Added OdometrySelfLocator
*
* Revision 1.24  2003/01/09 14:05:05  jhoffman
* added invkin parameter set
*
* Revision 1.23  2003/01/03 17:21:27  deutsch
* added class GridmageProcesserDo
*
* Revision 1.22  2002/12/20 13:50:54  dueffert
* CMU2002WalkingEngine added
*
* Revision 1.21  2002/12/05 11:58:37  goehring
* danielTest added
*
* Revision 1.20  2002/11/30 19:17:50  juengel
* Moved BehaviorControl solution Xabsl2Berlin to Humboldt2003.
*
* Revision 1.19  2002/11/28 14:47:17  jhoffman
* added motion detector
*
* Revision 1.18  2002/11/27 23:58:59  mkunz
* FusionSelfLocator added
*
* Revision 1.17  2002/11/27 13:47:50  dueffert
* doxygen docu corrected
*
* Revision 1.16  2002/11/26 14:17:08  juengel
* GridImageProcessor2 added.
*
* Revision 1.15  2002/11/25 14:49:57  jhoffman
* added motion detector
*
* Revision 1.14  2002/11/20 15:46:27  juengel
* PathPlanner added.
*
* Revision 1.13  2002/11/19 15:43:04  dueffert
* doxygen comments corrected
*
* Revision 1.12  2002/11/18 19:24:16  juengel
* Line Follower added.
*
* Revision 1.11  2002/11/18 17:13:45  loetzsch
* Added a new BehaviorControl solution: Xabsl2BerlinBehaviorControl
* When it works, it shall be cloned to the BehaviorControl solutions for
* the other univiersities
*
* Revision 1.10  2002/11/09 11:22:51  risler
* added errorWhileReading flag to SolutionRequest
*
* Revision 1.9  2002/11/07 17:30:10  loetzsch
* renamed Module SensorDataToMotionRequest to SensorBehaviorControl
*
* Revision 1.8  2002/10/10 13:31:16  jhoffman
* added ISL stuff
*
* Revision 1.7  2002/10/10 13:09:49  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.6  2002/09/25 10:25:13  loetzsch
* removed the "executeVisionModules" variable
* from SolutionRequest and ModuleHandler.
*
* Revision 1.5  2002/09/24 16:28:20  risler
* preparation for reading default module solutions from file
*
* Revision 1.4  2002/09/22 18:40:55  risler
* added new math functions, removed GTMath library
*
* Revision 1.3  2002/09/11 00:06:59  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.2  2002/09/10 17:53:26  loetzsch
* First draft of new Module/Solution Architecture
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.21  2002/09/07 13:36:55  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.20  2002/09/01 18:19:20  loetzsch
* removed 2 walking engines
*
* Revision 1.19  2002/08/21 11:50:43  cesarz
* GT2001PlayersPerceptor, BerlinSelfLocator, GT2001WalkingEngine,
* DoWalkingEngine, FuzzyBehaviorControl, BremenGOBehaviorControl,
* FuzzyBehaviortesterdlgbar and Behaviortesterdlgbar removed.
*
* Revision 1.18  2002/08/21 07:21:50  cesarz
* EllipseHeadControl and DefaultHeadControl removed.
*
* Revision 1.17  2002/07/29 17:45:11  jhoffman
* added braitenberg vehicle-style behavior (or actually: a "sensor data to motion request")
*
* Revision 1.16  2002/07/23 13:48:40  loetzsch
* - new streaming classes
* - removed many #include statements
* - exchanged StaticQueue by MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.15  2002/06/17 18:37:33  kspiess
* sample based multi ball locator added
*
* Revision 1.14  2002/06/17 05:06:10  risler
* added FixedSequenceWalkingEngine
*
* Revision 1.13  2002/06/13 22:11:16  roefer
* StickyBallLocator added
*
* Revision 1.12  2002/06/12 11:34:29  roefer
* SimpleLinesPerceptor removed, PerceptBehaviorControl added
*
* Revision 1.11  2002/06/06 10:52:29  kspiess
* distribution for multiple balls;
*
* Revision 1.10  2002/06/04 00:12:24  loetzsch
* added == and != operator
* added bool executeVisionModules
*
* Revision 1.9  2002/06/02 23:21:10  roefer
* Single color table and progress in LinesSelfLocator
*
* Revision 1.8  2002/06/01 13:38:35  risler
* reorganized walking engine selection
*
* Revision 1.7  2002/05/31 17:22:43  loetzsch
* added module selector for dribble walk type
*
* Revision 1.6  2002/05/16 15:05:49  risler
* added walk type walkWithBall
*
* Revision 1.5  2002/05/16 13:30:47  tim
* Added BallChallengeBehavior
*
* Revision 1.4  2002/05/15 08:08:53  risler
* changed default walk, added InvKin:Dortmund to selector
*
* Revision 1.3  2002/05/14 20:58:39  hebbel
* added SoundInProcessor
*
* Revision 1.2  2002/05/13 13:09:22  juengel
* GridImageProcessor is default solution now.
*
* Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.47  2002/05/06 12:57:39  tim
* DefaultPlayersPerceptor is now default solution
*
* Revision 1.46  2002/05/04 12:43:37  loetzsch
* The currently executed MotionRequest is now sent from the MotionControl
* to the BehaviorControl via the OdometryData structure
*
* Revision 1.45  2002/05/03 16:10:20  risler
* added UNSWClone InvKinWE Parameters
*
* Revision 1.44  2002/05/03 15:17:00  juengel
* Changed default solution for HeadControl from default to GT2002.
*
* Revision 1.43  2002/05/03 15:03:15  loetzsch
* removed berlinGT2002BehaviorControl
*
* Revision 1.42  2002/05/03 13:45:50  loetzsch
* switched default solution from BerlinGT2001 to xabsl
*
* Revision 1.41  2002/05/03 11:49:23  juengel
* GT2002HeadControl created.
*
* Revision 1.40  2002/04/29 08:36:27  risler
* no message
*
* Revision 1.39  2002/04/26 13:35:32  risler
* DarmstadtGOWalkingEngine renamed to InvKinWalkingEngine
* added InvKinParameterSets
*
* Revision 1.38  2002/04/24 10:35:55  jhoffman
* made the sub pixel thing a imagetoperceptcollection rather than an image processor
*
* Revision 1.37  2002/04/23 14:00:21  jhoffman
* added sub pixel gradient image processor
*
* Revision 1.36  2002/04/18 10:25:35  roefer
* Bremen GO
*
* Revision 1.35  2002/04/17 17:04:38  risler
* Darmstadt GO
*
* Revision 1.34  2002/04/16 21:55:17  dueffert
* a smoother head controller based on ellipses added
*
* Revision 1.33  2002/04/16 20:31:33  dueffert
* UNSW walk added
*
* Revision 1.32  2002/04/15 13:39:23  rentmeister
* DoWalkingEngine (Dortmunder WalkingEngine)
*
* Revision 1.3  2002/04/12 09:19:08  juengel
* no message
*
* Revision 1.2  2002/04/12 07:09:05  juengel
* Player Specialist added.
*
* Revision 1.2  2002/04/09 13:13:05  dueffert
* no message
*
* Revision 1.31  2002/04/05 16:46:20  kosen
* BremenGOWalkingEngine added
*
* Revision 1.30  2002/04/01 08:28:17  roefer
* Dummy LinesSelfLocator added
*
* Revision 1.29  2002/03/29 19:36:54  risler
* added DarmstadtGOWalkingEngine
*
* Revision 1.28  2002/03/28 17:44:22  loetzsch
* renamed BerlinGermanOpenBehaviorControl to XabslBehaviorControl
*
* Revision 1.27  2002/03/24 17:48:04  juengel
* LinesPercept defined and LinesPerceptor added.
*
* Revision 1.26  2002/03/21 15:17:28  koehlerj
* no message
*
* Revision 1.25  2002/03/16 13:44:49  juengel
* GridImageProcessor created.
*
* Revision 1.24  2002/03/12 14:31:30  jhoffman
* added fourier walking engine
*
* Revision 1.23  2002/03/11 12:51:00  loetzsch
* BerlinGermanOpenBehaviorControl added
*
* Revision 1.22  2002/02/21 16:28:24  risler
* removed SmoothHeadControl
*
* Revision 1.21  2002/02/20 16:57:39  kallnik
* new Headcontrol
* and
* new Function long abs(long)
*
* Revision 1.20  2002/02/20 16:40:50  risler
* added GetupEngine
*
* Revision 1.19  2002/02/18 15:35:09  loetzsch
* made MonteCarlo selflocator defaul solution for self localization
*
* Revision 1.18  2002/02/12 16:06:39  fischer
* Added FuzzyBehaviorControl.
*
* Revision 1.17  2002/02/12 09:45:09  roefer
* Progress in DefaultLandmarksPerceptor and MonteCarloSelfLocator
*
* Revision 1.16  2002/02/11 11:13:20  roefer
* BallPerceptor and BallLocator inserted
*
* Revision 1.15  2002/02/08 19:59:55  risler
* added DebugMotionControl
*
* Revision 1.14  2002/02/08 14:12:35  kspiess
* default fr selflocator auf disabled gesetzt, da montecarlo noch nicht funktioniert
*
* Revision 1.13  2002/02/06 16:20:48  brunn
* UNSWSelfLocator entfernt - dafr gibt es jetzt den SingleLandmarkSelfLocator
*
* Revision 1.12  2002/02/05 22:41:25  risler
* SimpleRunToBall added
*
* Revision 1.11  2002/02/05 03:43:22  loetzsch
* added a few new module selectors
*
* Revision 1.10  2002/02/01 20:45:42  kosen
* Bremen2001HeadControl removed
*
* Revision 1.9  2002/01/31 19:41:24  risler
* ImageProcessorSelector added
*
* Revision 1.8  2002/01/25 21:30:44  kosen
* Bremen2001HeadControl added
*
* Revision 1.7  2002/01/22 00:08:00  loetzsch
* Warnings for meaningless static const int ....() functions fixed
*
* Revision 1.6  2002/01/20 13:00:18  risler
* changed implementation of selecting default solution
*
* Revision 1.5  2002/01/19 21:36:24  risler
* added HeadMotionTester, HeadControlSelector
*
* Revision 1.4  2002/01/19 12:43:16  risler
* enabled SolutionRequest, changed HandleDebugMessage calls
*
* Revision 1.3  2002/01/18 14:00:07  risler
* BehaviorControlSelector added
*
* Revision 1.2  2002/01/17 14:35:48  risler
* SolutionRequest added
*
*/

