/**
* @file ModuleSelector.cpp
* 
* Implementation of class ModuleSelector.
*
* @author Max Risler
* @author Martin Ltzsch
*/

#include "ModuleSelector.h"
#include "Platform/GTAssert.h"
#include "Tools/Debugging/Debugging.h"

ModuleSelector::ModuleSelector(SolutionRequest::ModuleID id) 
: moduleID(id), selectedSolution(0)
{
  selectedSolutionID = SolutionRequest::getDefaultSolution(id);
}

ModuleSelector::~ModuleSelector()
{
  if (selectedSolution != 0) delete selectedSolution;
}

void ModuleSelector::selectSolution(SolutionRequest::ModuleSolutionID id) 
{
  ASSERT(id < solutionRequestMaxNumOfSolutions);
  if (id != selectedSolutionID)
  {
    if (selectedSolution != 0) 
    {
      delete selectedSolution;
    }
    selectedSolutionID=id;
    init();
  }
}

SolutionRequest::ModuleSolutionID ModuleSelector::getSelectedSolution() const 
{
  return selectedSolutionID;
}

void ModuleSelector::execute() 
{
  if (selectedSolution != 0) selectedSolution->execute();
}

bool ModuleSelector::handleMessage(InMessage& message)
{
  if (selectedSolution != 0) 
    return selectedSolution->handleMessage(message);
  else 
    return false;
}

void ModuleSelector::init()
{
  selectedSolution = this->createSolution(selectedSolutionID);
  if ((selectedSolution == 0)&&(selectedSolutionID != SolutionRequest::disabled))
  {
    OUTPUT(idText,text,"ModuleSelector::init(): could not create default solution \""
      << SolutionRequest::getModuleSolutionName(moduleID,selectedSolutionID) 
      << "\" of module \"" 
      << SolutionRequest::getModuleName(moduleID) << "\".");
  }
}

/*
* Change Log:
*
* $Log: ModuleSelector.cpp,v $
* Revision 1.3  2004/02/16 16:21:31  dueffert
* ModuleSelectors can actively disable modules now, eg for empty but initialized colorTable
*
* Revision 1.2  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2002/09/10 21:07:30  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.2  2002/09/10 17:53:26  loetzsch
* First draft of new Module/Solution Architecture
*
*/
