/**
* @file OutMessage.h
*
* Declaration of class OutMessageQueue, OutBinaryMessage, OutTextMessage and OutMessage.
*
* Include that file in implementation files that only want to write to MessageQueues.
* (Usually it lasts to include "Tools/Debugging/Debugging.h")
*
* @author Martin Ltzsch
*/

#ifndef __OutMessage_h_
#define __OutMessage_h_

#include "Tools/Streams/OutStreams.h"
#include "Platform/MessageQueueBase.h"

/**
* @class OutMessageQueue.
*
* A PhysicalOutStream that writes the data to a MessageQueue.
*/
class OutMessageQueue : public PhysicalOutStream
{
private:
  /** The queue where the data is written to */
  MessageQueueBase* queue; 
  
public:
  /** Default constructor */
  OutMessageQueue();
  
protected:
/**
* opens the stream.
* @param q A pointer to the message queue base
  */
  void open(MessageQueueBase* q);
  
/**
* The function writes a number of bytes into a physical stream.
* @param p The address the data is located at.
* @param size The number of bytes to be written.
  */
  virtual void writeToStream(const void* p,int size);
};

/** 
* @class OutBinaryMessage
*
* A binary stream into a message queue.
*/
class OutBinaryMessage : public OutStream<OutMessageQueue,OutBinary>
{
public:
/** 
* Constructor 
* @param q A pointer to the message queue base
  */
  OutBinaryMessage(MessageQueueBase* q);
};

/** 
* @class OutTextMessage
*
* A text stream into a message queue.
*/
class OutTextMessage : public OutStream<OutMessageQueue,OutText>
{
public:
/** 
* Constructor 
* @param q A pointer to the message queue base
  */
  OutTextMessage(MessageQueueBase* q);
};

/** 
* @class OutTextRawMessage
*
* A text stream into a message queue.
*/
class OutTextRawMessage : public OutStream<OutMessageQueue,OutTextRaw>
{
public:
/** 
* Constructor 
* @param q A pointer to the message queue base
  */
  OutTextRawMessage(MessageQueueBase* q);
};

/**
* @class OutMessage
*
* An Interface for writing messages into a MessageQueue.
*
* Use the bin or text member for formated writing into a message queue. 
*/
class OutMessage
{
private:
/** 
* The message queue where the messages are written into. 
  */
  MessageQueueBase& queue;
  
public:
  /** An interface for writing binary messages into the queue */ 
  OutBinaryMessage bin;
  
  /** An interface for writing text messages into the queue */ 
  OutTextMessage text;
  
  /** 
  * An interface for writing text messages in a raw style (see class OutTextRaw) 
  * into the queue 
  */ 
  OutTextMessage textRaw;
  
  /** 
  * Constructor
  * @param queue A reference to a MessageQueueBase
  */
  OutMessage(MessageQueueBase& queue);

  /** 
  * Finishes the message and allows to write a new message. 
  * Call that function after the writing of every message.
  * @param id The type id of the message
  */
  void finishMessage(MessageID id);

protected:
  /** 
  * Finishes the message and allows to write a new message. 
  * @param id The type id of the message
  * @param timeStamp The time stamp of the message
  * @param teamColor the team color of the robot that sent the message
  * @param playerNumber the player number of the robot that sent the message 
  * @param messageWasSentFromAPhysicalRobot If true, then the message was sent 
  *     from a physical robot.  
  */
  void finishMessage(MessageID id, unsigned long timeStamp,
    Player::teamColor teamColor,Player::playerNumber playerNumber,
    bool messageWasSentFromAPhysicalRobot);

  /** The player's team color that is attached to new messages */
  Player::teamColor teamColorForNewMessages;

  /** The player number that is attached to new messages */
  Player::playerNumber playerNumberForNewMessages;

  /** gives the MessageQueue class access to protected members */
  friend class MessageQueue;

  /** gives the InMessage class access to protected members */
  friend class InMessage;

  /** gives the operator that copies a InMessage to another queue access to protected members */
  friend void operator >> (InMessage& message, MessageQueue& queue);

  /** gives the In streaming opeator access to protected members */
  friend In& operator>>(In& stream,MessageQueue& messageQueue);
};


#endif //__OutMessage_h_

/*
* Change Log:
*
* $Log: OutMessage.h,v $
* Revision 1.3  2004/01/28 20:52:32  loetzsch
* Added the OutTextRaw stream writer.
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2002/11/19 17:38:32  dueffert
* doxygen bugs corrected
*
* Revision 1.2  2002/10/14 13:14:25  dueffert
* doxygen comments corrected
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/07/23 13:47:14  loetzsch
* - new streaming classes
* - new debug message handling
* - exchanged StaticQueue by MessageQueue
*
*/
